<?php
/**
 * Plugin Name: Payment system VTB
 * Description: Allows you to use Payment system VTB with the WooCommerce plugin.
 * Version: 1.12.15
 * Author: VTB
 * Author URI: https://www.vtb.ru/
 * Text Domain: wc-vtbpay
 * Domain Path: /languages/
 * Requires PHP: 7.4
 *
 * @package Vtbpay
 */

if (!defined('ABSPATH')) exit; // Exit if accessed directly

require __DIR__ . '/vendor/autoload.php';
require __DIR__ . '/hooks.php';

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Common\EventDispatcher,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Vtbpay\Traits\Atol,
    \Vtbpay\Traits\OrangeData,
    \Vtbpay\Traits\Shipping,
    \Symfony\Component\Dotenv\Dotenv;


add_action('plugins_loaded', 'init_wc_vtb_payment_gateway');

/**
 * Initialize Payment system VTB.
 */
function init_wc_vtb_payment_gateway(): void
{
    // if the WC payment gateway class is not available, do nothing
    if (!class_exists('WC_Payment_Gateway') || class_exists('WC_Vtbpay')) return;

    /**
     * Payment system VTB class.
     */
    class WC_Vtbpay extends WC_Payment_Gateway
    {
        use Atol, OrangeData, Shipping;

        const CMS_NAME = 'WordPress WooCommerce';
        const PLUGIN_VERSION = '1.12.15';

        // Payment system VTB settings
        private string $client_id;
        private string $client_secret;
        private string $merchant_authorization;
        private bool $test_mode;
        private bool $logging;
        private bool $two_stage;
        private string $transaction_end;
        private string $return_url;

        // Fiscal settings
        private bool $enable_fiscal;
        private string $ofd_fiscal;
        private string $email_fiscal;
        private string $payment_type_delivery_fiscal;
        private string $measure_fiscal;
        private string $tax_type_fiscal;
        private string $payment_type_fiscal;
        private string $payment_subject_fiscal;

        // Fiscal settings ATOL
        private bool $test_mode_atol_fiscal;
        private string $login_atol_fiscal;
        private string $pass_atol_fiscal;
        private string $kkt_atol_fiscal;
        private string $inn_atol_fiscal;
        private string $tax_system_atol_fiscal;

        // Fiscal settings Orange Data
        private bool $test_mode_orange_data_fiscal;
        private string $signature_key_orange_data_fiscal;
        private string $private_key_orange_data_fiscal;
        private string $client_key_orange_data_fiscal;
        private string $client_crt_orange_data_fiscal;
        private string $ca_cert_orange_data_fiscal;
        private string $cert_password_orange_data_fiscal;
        private string $group_orange_data_fiscal;
        private string $inn_orange_data_fiscal;
        private string $tax_system_orange_data_fiscal;


        // Common
        private object $order;
        private VtbPayLogger $logger;
        private EventDispatcher $event_dispatcher;


        /**
         * Конструктор для класса плагина "Платежная система ВТБ".
         */
        function __construct()
        {
            $plugin_dir = plugin_dir_url(__FILE__);

            $this->id = 'wc_vtbpay';
            $this->icon = apply_filters('woocommerce_vtbpay_icon', $plugin_dir . 'vtbpay.png');
            $this->method_title = __('Payment system VTB', 'wc-vtbpay');
            $this->method_description = __('Plugin "VTB Payment System" for WooCommerce, which allows you to integrate online payments.', 'wc-vtbpay');
            $this->has_fields = false;

            self::load_env_files([
                __DIR__ . '/config/.env',        // обязательный
                __DIR__ . '/config/.env.custom' // опциональный
            ]);

            $this->event_dispatcher = new EventDispatcher($this); // Инициализация менеджера событий

            // Load the settings
            $this->init_form_fields();
            $this->init_settings();

            // Initialize settings
            $this->init_payment_settings();
            $this->set_vtbpay_logger();

            //process settings with parent method
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
            add_action('woocommerce_api_' . $this->id . '_return', [$this, 'return_handler']);
            add_action('woocommerce_api_' . $this->id . '_webhook', [$this, 'webhook_handler']);
            // handler for checking and editing order status.
            add_action('woocommerce_api_' . $this->id . '_ajax', [$this, 'ajax_handler']);
        }


        /**
         * Загружает .env файлы.
         *
         * Первый файл обязателен, остальные — опциональные.
         *
         * @param array $files Пути к .env файлам, в порядке приоритета.
         * @return void
         * @throws \RuntimeException Если первый файл отсутствует.
         */
        private static function load_env_files(array $files): void
        {
            $dotenv = new Dotenv();

            foreach ($files as $index => $file) {
                if (is_readable($file)) {
                    $dotenv->load($file);
                } elseif ($index === 0) {
                    throw new \RuntimeException("Обязательный .env файл не найден: {$file}");
                }
            }
        }


        /**
         * Обрабатывает сохранение настроек в админке WooCommerce.
         *
         * @return bool Результат обработки настроек.
         */
        public function process_admin_options()
        {
            $settings_before_update = $this->settings;

            // Выполняем стандартное сохранение настроек WooCommerce
            $result = parent::process_admin_options();

            $this->handle_uploaded_files($settings_before_update);

            // Обрабатываем загрузку файлов и возможное обновление настроек
            return $result;
        }

        /**
         * Обрабатывает загрузку файлов и обновляет настройки, если они изменились.
         *
         * @param array $settings_before_update Настройки до обновления.
         * @return bool Успешно ли обработаны файлы.
         */
        protected function handle_uploaded_files(array $settings_before_update)
        {
            // Загружаем необходимые функции WordPress для загрузки файлов
            if (!function_exists('wp_handle_upload')) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }

            // Обрабатываем каждый загруженный файл
            if (!empty($_FILES)) {
                foreach ($_FILES as $field_name => $file_data) {
                    if (
                        strpos($field_name, 'woocommerce_wc_vtbpay_') === 0 &&
                        !empty($file_data['tmp_name'])
                    ) {
                        $uploaded = wp_handle_upload($file_data, ['test_form' => false]);
                        if (!isset($uploaded['error'])) {
                            // Сохраняем URL в настройках
                            $option_key = str_replace('woocommerce_wc_vtbpay_', '', $field_name);
                            $this->settings[$option_key] = $this->validate_text_field($option_key, $uploaded['file']);
                        } else {
                            WC_Admin_Settings::add_error("Ошибка загрузки файла '{$field_name}': {$uploaded['error']}");
                        }
                    }
                }
            }

            // Восстанавливаем прежние значения, если файл не был загружен повторно
            foreach ($this->get_form_fields() as $field_key => $field_data) {
                if (
                    $field_data['type'] === 'file' &&
                    !empty($settings_before_update[$field_key]) &&
                    empty($this->settings[$field_key])
                ) {
                    $this->settings[$field_key] = $settings_before_update[$field_key];
                }
            }

            update_option($this->get_option_key(), $this->settings);
        }


        /**
         * Инициализация настроек как свойства
         *
         * @return void
         */
        private function init_payment_settings(): void
        {
            foreach ($this->get_form_fields() as $field_key => $field_data) {
                if ($field_key === 'enabled') continue;
                switch ($field_data['type']) {
                  case 'text':
                  case 'textarea':
                  case 'select':
                  case 'file':
                    $this->$field_key = $this->get_option($field_key);
                    break;
                  case 'checkbox':
                    $this->$field_key = ($this->get_option($field_key) === 'yes');
                    break;
                }
            }
        }


        /**
         * Инициализируйте поля формы настроейк платежной системы ВТБ.
         *
         * @return void
         */
        public function init_form_fields(): void
        {
            $additional_text = sprintf(
                __('Current plugin version: %s', 'wc-vtbpay'),
                self::PLUGIN_VERSION
            );

            $instruction_url = $_ENV['LINK_TO_INSTRUCTION'] ?? '';
            if (!empty($instruction_url)) {
                $instruction_url = str_replace(
                    '<version>',
                    self::PLUGIN_VERSION,
                    $instruction_url
                );

                $additional_text .= sprintf(
                    __('<br>You can find more detailed information about setting up the plugin in the <a href="%s" target="_blank">instructions</a>.', 'wc-vtbpay'),
                    esc_url($instruction_url)
                );
            }

            $this->form_fields = include 'config/form_fields.php';
        }


        /**
         * Инициализируем свойство order объектом класса WC_Order,
         * полученным с помощью идентификатора и возвращаем объект текущего класса.
         *
         * @param string $order_id Идентификатор заказа.
         *
         * @return self|VtbPayException Объект заказа, если он существует, или Exception в противном случае.
         */
        private function set_order(string $order_id): self
        {
            if ($order = wc_get_order($order_id)) {
                $this->order = $order;
                return $this;
            }
            else {
                throw new VtbPayException(__('Order not found.', 'wc-vtbpay'), [
                    'order_id' => $order_id
                ]);
            }
        }


        /**
         * Создаем новый экземпляр класса VtbApi с заданными конфигурациями платёжной системы.
         *
         * @return VtbApi The new VtbApi instance.
         */
        protected function get_vtb_api(): VtbApi
        {
            return new VtbApi(
                $this->client_id,
                $this->client_secret,
                (bool) $this->test_mode,
                $this->merchant_authorization ?: ''
            );
        }


        /**
         * Обработка оплаты заказа в WooCommerce с помощью платежной системы ВТБ.
         * должно быть совместимо с WC_Payment_Gateway::process_payment($order_id)
         *
         * @param int $order_id Идентификатор заказа.
         *
         * @return array|null Массив, содержащий результат и URL перенаправления, или null, если произошла ошибка.
         */
        public function process_payment($order_id): ?array
        {
            $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

            try {
                $this->set_order($order_id);

                if ($this->order->get_payment_method() != $this->id) {
                    throw new \Exception(__('Payment method not "VTB Payment System".', 'wc-vtbpay'));
                }

                $pay_url = $this->get_pay_url();

                //Set order status pending payment
                $this->order->update_status('pending', __('Payment link generated:', 'wc-vtbpay') . $pay_url);

                //once the order is updated clear the cart and reduce the stock
                WC()->cart->empty_cart();

                return [
                    'result' => 'success',
                    'redirect' => $pay_url
                ];

            } catch (\Exception | VtbPayException $e) {
                // Handle exception and log error
                $this->error_logging($e, $order_id, __FUNCTION__);

                wc_add_notice($e->getMessage(), 'error');
                return null;
            }
        }


        /**
         * Получает URL-адрес платежа для перенаправления.
         *
         * @return string URL-адрес платежа.
         */
        private function get_pay_url(): string
        {
            $order_id = $this->order->get_id();
            $email = $this->get_customer_email();
            $total = $this->order->get_total();
            $return_url = $this->get_custom_return_url($order_id); // Get return URL

            $items = [];
            if (
                !$this->two_stage &&
                $this->enable_fiscal &&
                $this->ofd_fiscal === '1ofd'
            ) $items = $this->get_items();

            $response = $this->get_vtb_api()->getOrderLink(
                $order_id,
                $email,
                time(),
                $total,
                $return_url,
                $this->two_stage,
                $items,
                self::CMS_NAME,
                self::PLUGIN_VERSION
            );

            return $response['object']['payUrl'] ?? '';
        }


        /**
        * Подготавливает кастомный Return Url
        *
        * @param string $order_id
        *
        * @return string
        */
        private function get_custom_return_url(string $order_id): string
        {
            return home_url(
                (
                    !empty($_ENV['SHOW_RETURN_URL_FIELD']) &&
                    $_ENV['SHOW_RETURN_URL_FIELD'] ?? '0' === '1' &&
                    !empty($this->return_url ?? null)
                )
                    ? str_replace('#ID#', $order_id, $this->return_url)
                    : sprintf('/wc-api/%s_return', $this->id)
            );
        }


        /**
         * Отображение описания платежной системы ВТБ при оформлении заказа.
         *
         * @return void
         */
        public function payment_fields(): void
        {
            if (!empty($this->description)) {
                echo wpautop(wptexturize($this->description));
            }
        }


        /**
         * Обработчик return, вызываемый при переходе на страницу return_url, после попытки оплаты.
         *
         * @return void
         */
        public function return_handler(): void
        {
            $this->logger->setOption('additionalCommonText', 'return-' . rand(1111, 9999));
            $order_id = $_REQUEST['orderId'] ?? null;

            // Logging $_REQUEST
            $this->logger->debug(
                __FUNCTION__ . ' > return: ', [
                'request_data' => $_REQUEST
            ]);

            try {
                if (empty($order_id)) {
                    throw new \Exception(__('Order ID is empty.', 'wc-vtbpay'));
                }

                $this->set_order($order_id);

                $payment_status_data = $this->get_payment_status_data();

                $this->event_dispatcher->dispatch('afterGetPaymentStatusReturn', [
                    'payment_status_data' => $payment_status_data
                ]);

                $payment_status = $payment_status_data['object']['status']['value'] ?? '';

                $this->change_payment_status($payment_status);

                if (in_array($payment_status, ['PAID', 'PENDING', 'RECONCLIED'])) {
                    $redirect_url = $this->get_return_url($this->order);
                }
                else {
                    $redirect_url = str_replace('&amp;', '&', $this->order->get_cancel_order_url());
                }

                wp_safe_redirect(esc_url($redirect_url));

            } catch (\Exception | VtbPayException $e) {
                // Handle exception and log error
                $this->error_logging($e, $order_id, __FUNCTION__);

                wp_die($e->getMessage());
            }
        }


        /**
         * Обработчик return, вызываемый при переходе на страницу return_url, после попытки оплаты.
         *
         * @return void
         */
        public function webhook_handler(): void
        {
            $this->logger->setOption('additionalCommonText', 'webhook-' . rand(1111, 9999));
            $php_input = json_decode(file_get_contents('php://input'), true) ?: null;

            // Logging php input
            $this->logger->debug(
                __FUNCTION__ . ' > callback: ', [
                'php_input' => $php_input
            ]);

            $order_id = explode(
                '-',
                $php_input['object']['orderId'] ?? $php_input['external_id'] ?? $php_input['id']
            )[0];

            try {
                if (empty($order_id)) {
                    throw new \Exception(__('Order ID is empty.', 'wc-vtbpay'));
                }

                $this->set_order($order_id);

                $this->event_dispatcher->dispatch('beforeGetPaymentStatusWebhook', [
                    'php_input' => $php_input
                ]);

                $payment_status_data = $this->get_payment_status_data();

                $this->event_dispatcher->dispatch('afterGetPaymentStatusWebhook', [
                    'payment_status_data' => $payment_status_data
                ]);

                $payment_status = $payment_status_data['object']['status']['value'] ?? '';
                $this->change_payment_status($payment_status);

                die('OK');

            } catch (\Exception | VtbPayException $e) {
                // Handle exception and log error
                $this->error_logging($e, $order_id, __FUNCTION__);

                wp_die($e->getMessage());
            }
        }


        /**
         * Устанавливаем статус заказа
         *
         * @param string $payment_status Статус платежа
         *
         * @return bool
         */
        private function change_payment_status(string $payment_status): bool
        {
            $available_statuses = [
                'PAID' => fn() => $this->actions_for_paid_order(),
                'PENDING' => fn() => $this->actions_for_hold_order(),
                'REFUNDED' => fn() => $this->actions_for_refunded_order(),
                'PARTIALLY_REFUNDED' => fn() => $this->actions_for_partially_refunded_order()
            ];

            $order_status_fnc = $available_statuses[$payment_status] ?? false;

            if (!empty($order_status_fnc)) $order_status_fnc();
            else $this->actions_for_unpaid_order(); // Payment not completed

            return !empty($order_status_fnc);
        }


        /**
         * Проверяем статус платежа, и если он PAID, то устанавливаем заказ в CMS как оплаченный.
         * Если статус платежа REFUNDED, то отправляем скорректированный чек,
         * в котором указано, что он отменён и в CMS устанавливаем заказ в как отменённый.
         *
         * @return void
         */
        public function ajax_handler(): void
        {
            $this->logger->setOption('additionalCommonText', 'ajax-' . rand(1111, 9999));
            // Получает данные из AJAX-запроса через $_POST
            $order_id = $_POST['order_id'] ?? '';
            $action = $_POST['action'] ?? '';

            try {
                if (!wp_verify_nonce($_POST['nonce'] ?? '', 'vtbpay-ajax-nonce')) {
                    throw new \Exception(__('Failed ajax validation.', 'wc-vtbpay'));
                }

                if (empty($order_id)) {
                    throw new \Exception(__('Order ID is empty.', 'wc-vtbpay'));
                }

                if (empty($action)) {
                    throw new \Exception(__('Required action not specified.', 'wc-vtbpay'));
                }

                // Logging php input
                $this->logger->debug(
                    __FUNCTION__ . ' > ajax: ', [
                    'post' => $_POST
                ]);

                $this->set_order($order_id);

                $payment_status_data = $this->get_payment_status_data();

                if ($action == 'refund') {
                    // Указываем что возврат из админки
                    $this->order->update_meta_data('_vtbpay_refunded', 'yes');
                    $this->order->save();

                    $this->make_chargeback();

                    $message = __('Payment refunded.', 'wc-vtbpay');
                }
                elseif ($action == 'check_status') {
                    $payment_status = $payment_status_data['object']['status']['value'] ?? '';

                    $this->change_payment_status($payment_status);

                    $message = __('Status updated.', 'wc-vtbpay');
                }

                // Отправьте JSON-ответ и завершите выполнение скрипта
                wp_send_json([
                    'success' => true,
                    'message' => $message
                ]);

            } catch (\Exception | VtbPayException $e) {
                // Handle exception and log error
                $this->error_logging($e, $order_id, __FUNCTION__);

                wp_send_json([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        }


        /**
         * Отправляем запрос на возврат в ВТБ, возвращвем результат запроса и логируем входящие и выходящие данные.
         * Если запрос прошёл успешно, то и в CMS отображаем информацию о возврате.
         *
         * @return array
         */
        private function make_chargeback(): array
        {
            $order_id = $this->order->get_id();

            $amount = $_POST['refund_amount'] ?? 0;

            if (empty($amount)) throw new \Exception(__('Refund amount not specified.', 'wc-vtbpay'));

            $amount = self::parse_amount($amount);

            $email = $this->get_customer_email();

            $items = [];
            if (
                $this->enable_fiscal &&
                (
                    $this->ofd_fiscal === '1ofd' && !$this->two_stage ||
                    $this->ofd_fiscal === 'atol' ||
                    $this->ofd_fiscal === 'orange_data'
                )
            ) $items = $this->preparing_items_for_refunds(
                $this->get_items()
            );

            $response = $this->get_vtb_api()->setRefunds(
                $order_id,
                $amount,
                $email,
                $items
            );

            $this->event_dispatcher->dispatch('paymentRefunded', [
                'items' => $items
            ]);

            return $response;
        }


        /**
         * Подготавливает элементы для возврата.
         *
         * @param array $items Элементы для подготовки.
         * @return array Массив подготовленных элементов.
         * @throws \Exception Если отсутствуют необходимые данные для возврата.
         */
        private function preparing_items_for_refunds(array $items): array
        {
            $refund_order_item_qty = $_POST['refund_order_item_qty'] ?? [];
            $refund_line_total = $_POST['refund_line_total'] ?? [];

            if (empty($refund_order_item_qty) || empty($refund_line_total)) {
                throw new \Exception(__('The required data for a refund was not transmitted.', 'wc-vtbpay'));
            }

            $prepared_items = [];

            foreach ($items as $item) {
                $item_id = explode('-', $item['code'])[0] ?? '';
                $quantity = $refund_order_item_qty[$item_id] ?? 0;
                $amount = $refund_line_total[$item_id] ?? 0;

                if (empty($quantity) && empty($amount)) continue;

                if (!empty($amount)) $item['amount'] = self::parse_amount($amount);
                else $item['amount'] = (int) $quantity * $item['price'];

                if (!empty($quantity)) $item['quantity'] = (int) $quantity;

                $item['price'] = $item['amount'] / $item['quantity'];

                // Добавляем элемент в новый массив
                $prepared_items[] = $item;
            }

            return $prepared_items;
        }


        /**
         * Получает массив товаров заказа.
         *
         * @return array Массив товаров.
         */
        private function get_items(): array
        {
            // Настройки по умолчанию для фискальных параметров
            $fiscal_settings = [
                'measure_fiscal' => 0,
                'tax_type_fiscal' => 'none',
                'payment_type_fiscal' => 'full_prepayment',
                'payment_subject_fiscal' => 1
            ];
            $items = [];
            $key = 1;

            foreach ($this->order->get_items() as $item_id => $item) {

                $product_id = $item->get_product_id();

                $product = $item->get_product();

                // TO-DO: self::parse_amount()
                $amount = floatval($item->get_total() + $item->get_total_tax()); // Общая сумма по товару с учётом скидок и налогов
                $price = floatval($amount / $item->get_quantity());  // Цена товара с учётом скидок и налогов

                $product_fiscal = [];
                // Получение фискальных параметров для каждого продукта
                foreach ($fiscal_settings as $name => $default) {
                    $product_fiscal[$name] = str_replace('_' . $name, '', get_post_meta($product_id, 'vtbpay_' . $name, true)) ?: '-';
                    if ($product_fiscal[$name] === '-') $product_fiscal[$name] = $this->$name ?: $default;
                }

                // Получение массы (веса) товара
                $mass = (int) round(($product->get_weight() ?: 0) * 1000);

                $items[] = [
                    'positionId' => $key,
                    'name' => $item->get_name(),
                    'code' => (string) $item_id . '-' . $product_id,
                    'price' => $price,
                    'measure' => (int) $product_fiscal['measure_fiscal'],
                    'quantity' => (int) $item->get_quantity(),
                    'taxParams' => [
                        'taxType' => $product_fiscal['tax_type_fiscal']
                    ],
                    'paymentType' => $product_fiscal['payment_type_fiscal'],
                    'paymentSubject' => (int) $product_fiscal['payment_subject_fiscal'],
                    'amount' => $amount,
                    'mass' => $mass
                ];

                $key++;
            }

            foreach ($this->order->get_items('shipping') as $item_id => $item) {
                $amount = $item->get_total() + $item->get_total_tax();

                if ($amount <= 0) continue;

                $items[] = [
                    'positionId' => $key,
                    'name' => $item->get_name(),
                    'code' => (string) $item_id . '-' . $item->get_method_id(),
                    'price' => $amount,
                    'measure' => 0,
                    'quantity' => 1,
                    'taxParams' => [
                        'taxType' => $this->tax_type_fiscal ?: 'none'
                    ],
                    'paymentType' => $this->payment_type_delivery_fiscal ?: 'full_prepayment',
                    'paymentSubject' => 4,
                    'amount' => $amount
                ];

                $key++;
            }

            return $items;
        }


        /**
         * Обработка действий для оплаченного заказа.
         *
         * @return void
         */
        private function actions_for_paid_order(): void
        {
            $completed_status = str_replace('wc-', '', $this->transaction_end);
            if ($this->order->get_status() !== $completed_status) {
                $this->order->update_status(
                    $completed_status,
                    __('Payment completed successfully.', 'wc-vtbpay')
                );
                $this->order->payment_complete();
                wc_reduce_stock_levels($this->order->get_id());

                $this->event_dispatcher->dispatch('paymentPaid');
            }
        }


        /**
         * Обработка действий для удержанного заказа.
         *
         * @return void
         */
        private function actions_for_hold_order(): void
        {
            if ($this->order->get_status() !== 'on-hold') $this->order->update_status(
                'on-hold',
                __('Payment received but not confirmed.', 'wc-vtbpay')
            );
        }


        /**
         * Обработка действий в случае неоплаченного заказа.
         *
         * @return void
         */
        private function actions_for_unpaid_order(): void
        {
            // Logging an unpaid order
            $this->logger->debug(sprintf(
                __FUNCTION__ . '. ' . __('Payment not paid.', 'wc-vtbpay') . '; Order ID: %s',
                $this->order->get_id()
            ));

            if ($this->order->get_status() !== 'failed') $this->order->update_status(
                'failed',
                __('Payment not paid.', 'wc-vtbpay')
            );
            wc_add_notice(__('Payment not paid. Your order has been canceled.', 'wc-vtbpay'), 'error');
        }


        /**
         * Обработка действий в случае возвращённого заказа.
         *
         * @return void
         */
        private function actions_for_refunded_order(): void
        {
            // Logging an unpaid order
            $this->logger->debug(sprintf(
                __FUNCTION__ . '. ' . __('Refund of payment.', 'wc-vtbpay') . '; Order ID: %s',
                $this->order->get_id()
            ));

            if (
                $this->order->get_status() !== 'refunded' &&
                $this->order->get_meta('_vtbpay_refunded') !== 'yes'
            ) {
                $this->order->update_status(
                    'refunded',
                    __('Refund of payment.', 'wc-vtbpay')
                );

                $this->event_dispatcher->dispatch('paymentRefunded');
            }
            wc_add_notice(
                __('The payment was refunded.', 'wc-vtbpay'),
                'notice'
            );
        }


        /**
         * Обработка действий в случае частично возвращённого заказа.
         *
         * @return void
         */
        private function actions_for_partially_refunded_order(): void
        {
            // Logging an unpaid order
            $this->logger->debug(sprintf(
                __FUNCTION__ . '. ' . __('Partial refund of payment.', 'wc-vtbpay') . '; Order ID: %s',
                $this->order->get_id()
            ));

            $completed_status = str_replace('wc-', '', $this->transaction_end);
            if (
                $this->order->get_status() !== $completed_status &&
                $this->order->get_meta('_vtbpay_refunded') !== 'yes'
            ) {
                $this->order->update_status(
                    $completed_status,
                    __('Partial refund of payment.', 'wc-vtbpay')
                );
            }
            wc_add_notice(
                __('The payment was partial refunded.', 'wc-vtbpay'),
                'notice'
            );
        }


        /**
         * Получение данные заказа из VTB API.
         *
         * @return array
         */
        private function get_payment_status_data(): array
        {
            $response = $this->get_vtb_api()->getOrderInfo(
                $this->order->get_id()
            );

            return $response ?? [];
        }


        /**
         * Преобразует строку в число с плавающей запятой.
         *
         * @param mixed $amount Сумма для преобразования.
         * @return float Преобразованное значение суммы.
         */
        private static function parse_amount($amount): float
        {
            if (is_string($amount)) {
                $amount = str_replace([' ', ','], ['', '.'], $amount);
            }

            return floatval(number_format($amount, 2, '.', ''));
        }


        /**
         * Получает email клиента, иначе выкидывает ошибку
         */
        private function get_customer_email(): string
        {
            // Get Email
            $email = $this->order->get_billing_email() ?: $this->email_fiscal;
            if (!self::validate_email($email)) {
                throw new VtbPayException(__('The email address provided has not been validated.', 'wc-vtbpay'), [
                    'email' => $email
                ]);
            }
            return $email;
        }


        /**
         * Валидация Email.
         *
         * @param string $email Email.
         *
         * @return bool
         */
        private static function validate_email(string $email)
        {
            // Базовая проверка синтаксиса по стандарту RFC
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                return false;
            }

            // Разделение email на локальную и доменную части
            list($local_part, $domain_part) = explode('@', $email);

            // Проверка длины локальной части (от 1 до 64 символов)
            $local_length = strlen($local_part);
            if ($local_length < 1 || $local_length > 64) {
                return false;
            }

            // Проверка длины доменной части (от 1 до 255 символов)
            $domain_length = strlen($domain_part);
            if ($domain_length < 1 || $domain_length > 253) {
                return false;
            }

            // Список разрешённых популярных доменных зон
            $allowed_tlds = isset($_ENV['EMAIL_VALIDATION_ALLOWED_TLDS']) ? explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS']) : [];
            if (!empty($allowed_tlds)) {
                // Извлечение TLD из доменной части
                $domain_parts = explode('.', $domain_part);
                $tld = strtolower(end($domain_parts));

                // Проверка, что TLD входит в список разрешённых
                if (!in_array($tld, $allowed_tlds)) {
                    return false;
                }
            }

            return true;
        }


        /**
         * Сценарий, выполняемый после отлова исключения
         *
         * @param Throwable $e Объект исключения.
         * @param string $order_id Идентификатор заказа.
         * @param string $caller_func_name Название функции, котоорая вызывает обработчик исключений.
         *
         * @return void
         */
        private function error_logging(
            Throwable $e,
            string $order_id,
            string $caller_func_name = ''
        ): void {
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (
                method_exists($e, 'getContext') &&
                is_array($e->getContext())
            ) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                $caller_func_name . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $order_id ?: ''
            ), $context);
        }


        /**
         * Отправляет на почту ошибку формирования чека
         *
         * @return void
         */
        private function send_receipt_error_email($message): void
        {
            $site_name = get_bloginfo('name'); // Получаем название сайта

            // Задаем параметры для отправки письма
            $headers = [
                'From: ' . $site_name . ' <' . get_option('admin_email') . '>',
                'Content-Type: text/html; charset=UTF-8',
            ];

            $subject = __('Receipt issuance error:', 'wc-vtbpay') . ' ' . $site_name;

            // Отправляем письмо
            if (wp_mail(
                $this->email_fiscal,
                $subject,
                $message,
                $headers
            )) {
                $this->logger->debug(
                    __FUNCTION__ . ' > wp_mail: ', [
                    'email_fiscal' => $this->email_fiscal,
                    'subject' => $subject,
                    'message' => $message,
                    'headers' => $headers
                ]);
            }
        }


        /**
         * Инициализация и настройка объекта класса VtbPayLogger.
         *
         * Эта функция инициализирует и настраивает логгер, используемый плагином VtbPay для ведения журнала.
         *
         * @return void
         */
    		private function set_vtbpay_logger(): void
        {
            if (version_compare(WOOCOMMERCE_VERSION, '2.1', '<')) {
                global $woocommerce;
                $wc_log = $woocommerce->logger();
            }
            else {
                $wc_log = new WC_Logger();
            }

            $sensitive_data_keys = isset($_ENV['LOG_SENSITIVE_DATA_KEYS']) ? json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'], true) : [];

            // Настройка в админке (логироать или нет)
            $logging = $this->logging;

            $this->logger = VtbPayLogger::getInstance()
                            ->setOption('showCurrentDate', false)
                            ->setOption('showLogLevel', false)
                            ->setOption('showBacktrace', true)
                            ->setOption('sensitiveDataKeys', $sensitive_data_keys)
                            ->setCustomRecording(function($message) use ($wc_log){
                                $wc_log->error($message);
                            }, VtbPayLogger::LOG_LEVEL_ERROR)
                            ->setCustomRecording(function($message) use ($wc_log, $logging){
                                if ($logging) $wc_log->debug($message);
                            }, VtbPayLogger::LOG_LEVEL_DEBUG);
        }
    }
}
