<?php

namespace Vtbpay\Traits;

trait Shipping
{
    /**
     * Обработчик событий afterGetPaymentStatusWebhook и afterGetPaymentStatusReturn
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function on_after_get_payment_status_shipping(array $vars)
    {
        // Проверка и обновление стоимости доставки
        $input_amount = $vars['payment_status_data']['object']['amount']['value'] ?? 0;
        if ($input_amount > $this->order->get_total()) {
            $order_code = $vars['payment_status_data']['object']['orderCode'] ?? '';
            $this->update_shipping_cost($order_code, $input_amount);
        }
    }


    /**
     * Обновляет или добавляет стоимость доставки в заказе.
     *
     * @param string $order_code Код заказа.
     * @param float $new_total Новая итоговая сумма заказа.
     * @return void
     */
    private function update_shipping_cost(
        string $order_code,
        float $new_total
    ): void {
        $current_order_total = floatval($this->order->get_total());
        $difference = self::parse_amount($new_total - $current_order_total);

        $response = $this->get_vtb_api()->getBundle($order_code);

        $delivery_code = $response['order']['deliveryParams']['recipientParams']['deliveryCode'] ?? '';
        if (!empty($delivery_code)) {
            // Удаляем доставки с нулевой стоимостью
            $this->remove_zero_shipping_items();

            $this->add_shipping_item(
                $delivery_code,
                __('Delivery Russian Post', 'wc-vtbpay'),
                $difference
            );

            $this->order->calculate_totals();
            $this->order->save();
        }
    }


    /**
     * Удаляет элементы доставки с нулевой стоимостью из заказа.
     *
     * @return void
     */
    private function remove_zero_shipping_items(): void
    {
        foreach ($this->order->get_items('shipping') as $item_id => $shipping_item) {
            if ((float) $shipping_item->get_total() === 0.0) {
                $this->logger->debug(__FUNCTION__ . ' > remove_item - INPUT', [
                    'item_id' => $item_id
                ]);
                $this->order->remove_item($item_id);
            }
        }
    }


    /**
     * Добавляет элемент доставки с указанной стоимостью.
     *
     * @param float $shipping_total Стоимость доставки для добавления.
     * @return void
     */
    private function add_shipping_item(
        string $shipping_method,
        string $shipping_name,
        float $shipping_total
    ): void {
        $this->logger->debug(__FUNCTION__ . ' > WC_Order_Item_Shipping - INPUT: ', [
            'shipping_method' => $shipping_method,
            'shipping_name' => $shipping_name,
            'shipping_total' => $shipping_total
        ]);

        $shipping_item = new \WC_Order_Item_Shipping();
        $shipping_item->set_method_title($shipping_name);
        $shipping_item->set_method_id($shipping_method);
        $shipping_item->set_total($shipping_total);

        $this->order->add_item($shipping_item);
    }
}
