<?php

namespace Vtbpay\Traits;

use \Vtbpay\Classes\Api\OrangeDataApi,
    \Vtbpay\Classes\Exception\VtbPayException;

trait OrangeData
{
    /**
     * Обработчик события beforeGetPaymentStatusWebhook
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function on_before_get_payment_status_orange_data(array $vars)
    {
        // Если к вебхуку обращается OrangeData
        $document_id = $vars['php_input']['id'] ?? '';
        if (!empty($document_id)) {
            $this->webhook_handler_orange_data($document_id);
            die('OK');
        }
    }


    /**
     * Обработчик события paymentPaid
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function on_payment_paid_orange_data(array $vars)
    {
        if (
            $this->enable_fiscal &&
            $this->ofd_fiscal === 'orange_data'
        ) $this->send_receipt_orange_data();
    }


    /**
     * Обработчик события paymentRefunded
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function on_payment_refunded_orange_data(array $vars)
    {
        $items = $vars['items'] ?? [];
        if (
            $this->enable_fiscal &&
            $this->ofd_fiscal === 'orange_data'
        ) $this->send_receipt_orange_data(true, $items);
    }


    /**
     * Создаем новый экземпляр класса OrangeDataApi с заданными конфигурациями платёжной системы.
     *
     * @return OrangeDataApi The new OrangeDataApi instance.
     */
    protected function get_orange_data_api(): OrangeDataApi
    {
        return new OrangeDataApi(
            $this->inn_orange_data_fiscal,
            $this->tax_system_orange_data_fiscal,
            $this->group_orange_data_fiscal,
            $this->signature_key_orange_data_fiscal,
            $this->private_key_orange_data_fiscal,
            $this->client_key_orange_data_fiscal,
            $this->client_crt_orange_data_fiscal,
            $this->ca_cert_orange_data_fiscal,
            $this->cert_password_orange_data_fiscal,
            (bool) $this->test_mode_orange_data_fiscal
        );
    }


    /**
     * Проверяет статус формирования чека в АТОЛ и оповещает при ошибке
     *
     * @param string $document_id Полученные данные
     *
     * @return void
     */
    private function webhook_handler_orange_data(string $document_id): void
    {
        $response = $this->get_orange_data_api()->getReceiptInfo($document_id);

        if ($response['status'] === 202) {
            $this->send_receipt_error_email(sprintf(
                  __('For order %s, the check generation process is not complete. Please check the availability of the check yourself in your personal account at Orange Data.', 'wc-vtbpay'),
                  $this->order->get_id()
            ));
        }
        elseif ($response['status'] !== 200) {
            $this->send_receipt_error_email(sprintf(
                  __('An error occurred while generating a receipt for order %s. Please generate a receipt yourself in your personal account at Orange Data.', 'wc-vtbpay'),
                  $this->order->get_id()
            ));
        }
    }


    /**
     * Отправляет в АТОЛ чек прихода или возврата на печать и возвращает результат.
     *
     * @param bool $is_return Данный чек "Возврат прихода" или "Приход"
     * @param array $items Товары заказа
     *
     * @return array
     */
    private function send_receipt_orange_data(bool $is_return = false, array $items = []): array
    {
        try {
            $order_id = $this->order->get_id();

            // Если частичный возврат, значит придётся отправлять несколько чеков для одного заказа
            if (!empty($items)) $order_id .= '-' .  time();

            $email = $this->get_customer_email();

            $items = $this->preparing_items_for_orange_data(
                $items ?: $this->get_items()
            );

            $total = empty($items) ? $this->order->get_total() : $this->calculate_total_sum_orange_data($items);

            $callback_url = get_site_url() . '/wc-api/wc_vtbpay_webhook';

            return $this->get_orange_data_api()->sendReceipt(
                $order_id,
                $items,
                $total,
                $email,
                $callback_url,
                $is_return
            );

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $this->error_logging($e, $this->order->get_id(), __FUNCTION__);

            $this->send_receipt_error_email(sprintf(
                __('An error occurred while generating a receipt for order %s. Please generate a receipt yourself in your personal account at Orange Data.', 'wc-vtbpay'),
                $this->order->get_id()
            ));

            return [];
        }
    }


     /**
      * Сопоставляет внутренние значения с параметрами Orange Data.
      *
      * @param string $setting Тип настройки: 'paymentMethodType', 'tax', 'unitOfMeasurement'
      * @param string $key Ключ настройки
      * @return string Соответствующее значение или пустая строка
      */
    private static function settings_mapping_orange_data(string $setting, string $key): string
    {
        $settings_mapping = [
            'paymentMethodType' => [
                'full_prepayment'   => 1, // Предоплата 100%
                'prepayment'        => 2, // Частичная предоплата
                'advance'           => 3, // Аванс
                'full_payment'      => 4, // Полный расчёт
                'partial_payment'   => 5, // Частичный расчет и кредит
                'credit'            => 6, // Передача в кредит
                'credit_payment'    => 7, // Оплата кредита
            ],
            'tax' => [
                'vat20'   => 1,  // ставка НДС 20%
                'vat10'   => 2,  // ставка НДС 10%
                'vat120'  => 3,  // ставка НДС расч. 20/120
                'vat110'  => 4,  // ставка НДС расч. 10/110
                'vat0'    => 5,  // ставка НДС 0%
                'none'    => 6,  // НДС не облагается
                'vat5'    => 7,  // ставка НДС 5%
                'vat7'    => 8,  // ставка НДС 7%
                'vat105'  => 9,  // ставка НДС расч. 5/105
                'vat107'  => 10, // ставка НДС расч. 7/107
            ],
            'unitOfMeasurement' => [
                0   => 'pcs',    // Штучный товар
                10  => 'g',      // Грамм
                11  => 'kg',     // Килограмм
                12  => 't',      // Тонна
                20  => 'cm',     // Сантиметр
                21  => 'dm',     // Дециметр
                22  => 'm',      // Метр
                30  => 'cm²',    // Квадратный сантиметр
                31  => 'dm²',    // Квадратный дециметр
                32  => 'm²',     // Квадратный метр
                40  => 'ml',     // Миллилитр
                41  => 'l',      // Литр
                42  => 'm³',     // Кубический метр
                50  => 'kWh',    // Киловатт-час
                51  => 'Gcal',   // Гигакалория
                70  => 'd',      // Сутки
                71  => 'h',      // Час
                72  => 'min',    // Минута
                73  => 's',      // Секунда
                80  => 'KB',     // Килобайт
                81  => 'MB',     // Мегабайт
                82  => 'GB',     // Гигабайт
                83  => 'TB',     // Терабайт
                255 => 'other',  // Иная единица измерения
            ],


        ];

        return $settings_mapping[$setting][$key] ?? '';
    }


    /**
     * Подготавливает элементы для АТОЛ.
     *
     * @param array $items Элементы для подготовки.
     *
     * @return array Массив подготовленных элементов.
     */
    private function preparing_items_for_orange_data(array $items): array
    {
        return array_map(function ($item) {
            return [
                'text' => $item['name'], // Лейбл
                'price' => $item['price'], // Цена товара за 1шт
                'totalSum' => $item['amount'], // Итоговая цена (цена * кол-во)
                'quantity' => $item['quantity'], // Кол-во позиций товара (сколько шт)
                'unitOfMeasurement' => self::settings_mapping_orange_data('unitOfMeasurement', $item['measure']), // Единица измерения
                'tax' => self::settings_mapping_orange_data('tax', $item['taxParams']['taxType']), // Налоговая ставка (НДС),
                'paymentMethodType' => self::settings_mapping_orange_data('paymentMethodType', $item['paymentType']), // Способ расчета
                'paymentSubjectType' => $item['paymentSubject'], // Предмет расчета
            ];
        }, $items);
    }


    /**
     * Рассчитывает итоговую сумму товаров из массива.
     *
     * Функция принимает массив товаров, в котором у каждого товара есть поле "totalSum".
     * Суммирует значения всех полей "totalSum" и возвращает итоговую сумму.
     *
     * @param array $items Массив товаров, где каждый товар — это ассоциативный массив с ключом "sum".
     * @return float Итоговая сумма всех товаров.
     */
    private function calculate_total_sum_orange_data(array $items): float
    {
        return array_reduce($items, function($carry, $item) {
            return $carry + ($item['totalSum'] ?? 0);
        }, 0);
    }
}
