<?php

namespace Vtbpay\Traits;

use \Vtbpay\Classes\Api\AtolApi,
    \Vtbpay\Classes\Exception\VtbPayException;

trait Atol
{
    /**
     * Обработчик события beforeGetPaymentStatusWebhook
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function on_before_get_payment_status_atol(array $vars)
    {
        // Если к вебхуку обращается АТОЛ
        $uuid = $vars['php_input']['uuid'] ?? '';
        if (!empty($uuid)) {
            $this->webhook_handler_atol($uuid);
            die('OK');
        }
    }


    /**
     * Обработчик события paymentPaid
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function on_payment_paid_atol(array $vars)
    {
        if (
            $this->enable_fiscal &&
            $this->ofd_fiscal === 'atol'
        ) $this->send_receipt_atol();
    }


    /**
     * Обработчик события paymentRefunded
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function on_payment_refunded_atol(array $vars)
    {
        $items = $vars['items'] ?? [];
        if (
            $this->enable_fiscal &&
            $this->ofd_fiscal === 'atol'
        ) $this->send_receipt_atol(true, $items);
    }


    /**
     * Создаем новый экземпляр класса AtolApi с заданными конфигурациями платёжной системы.
     *
     * @return AtolApi The new AtolApi instance.
     */
    protected function get_atol_api(): AtolApi
    {
        return new AtolApi(
            $this->login_atol_fiscal,
            $this->pass_atol_fiscal,
            $this->inn_atol_fiscal,
            $this->tax_system_atol_fiscal,
            $this->kkt_atol_fiscal,
            $this->email_fiscal,
            (bool) $this->test_mode_atol_fiscal
        );
    }


    /**
     * Проверяет статус формирования чека в АТОЛ и оповещает при ошибке
     *
     * @param string $uuid Полученные данные
     *
     * @return void
     */
    private function webhook_handler_atol(string $uuid): void
    {
        $response = $this->get_atol_api()->getReceiptInfo($uuid);

        if ($response['status'] === 'fail'){
            $this->send_receipt_error_email(sprintf(
                  __('An error occurred while generating a receipt for order %s. Please generate a receipt yourself in your personal account at ATOL Online.', 'wc-vtbpay'),
                  $this->order->get_id()
            ));
        }
        elseif ($response['status'] === 'wait') {
            $this->send_receipt_error_email(sprintf(
                  __('For order %s, the check generation process is not complete. Please check the availability of the check yourself in your personal account at ATOL Online.', 'wc-vtbpay'),
                  $this->order->get_id()
            ));
        }
    }


    /**
     * Отправляет на почту ошибку формирования чека АТОЛ
     *
     * @return void
     */
    private function send_receipt_error_email($message): void
    {
        $site_name = get_bloginfo('name'); // Получаем название сайта

        // Задаем параметры для отправки письма
        $headers = [
            'From: ' . $site_name . ' <' . get_option('admin_email') . '>',
            'Content-Type: text/html; charset=UTF-8',
        ];

        $subject = __('Receipt issuance error:', 'wc-vtbpay') . ' ' . $site_name;

        // Отправляем письмо
        wp_mail(
            $this->email_fiscal,
            $subject,
            $message,
            $headers
        );
    }


     /**
      * Отправляет в АТОЛ чек прихода или возврата на печать и возвращает результат.
      *
      * @param bool $is_return Данный чек "Возврат прихода" или "Приход"
      * @param array $items Товары заказа
      *
      * @return array
      */
     private function send_receipt_atol(bool $is_return = false, array $items = []): array
     {
         try {
             $order_id = $this->order->get_id();

             // Если частичный возврат, значит придётся отправлять несколько чеков для одного заказа
             if (!empty($items)) $order_id .= '-' .  time();

             $email = $this->get_customer_email();

             $items = $this->preparing_items_for_atol(
                 $items ?: $this->get_items()
             );

             $total = empty($items) ? $this->order->get_total() : $this->calculate_total_sum($items);

             $callback_url = get_site_url() . '/wc-api/wc_vtbpay_webhook';

             $response = $this->get_atol_api()->sendReceipt(
                 $order_id,
                 $items,
                 $total,
                 $email,
                 $callback_url,
                 $is_return
             );

             return $response;

         } catch (\Exception | VtbPayException $e) {
             // Handle exception and log error
             $this->error_logging($e, $this->order->get_id(), __FUNCTION__);

             $this->send_receipt_error_email(sprintf(
                   __('An error occurred while generating a receipt for order %s. Please generate a receipt yourself in your personal account at ATOL Online.', 'wc-vtbpay'),
                   $this->order->get_id()
             ));

             return [];
         }
     }


     /**
      * Подготавливает элементы для АТОЛ.
      *
      * @param array $items Элементы для подготовки.
      *
      * @return array Массив подготовленных элементов.
      */
     private function preparing_items_for_atol(array $items): array
     {
         return array_map(function ($item) {
             return [
                 'name' => $item['name'], // Лейбл
                 'price' => $item['price'], // Цена товара за 1шт
                 'sum' => $item['amount'], // Итоговая цена (цена * кол-во)
                 'quantity' => $item['quantity'], // Кол-во позиций товара (сколько шт)
                 'measure' => $item['measure'], // Единица измерения
                 'vat' => [
                     'type' => $item['taxParams']['taxType'] // Налоговая ставка (НДС)
                 ],
                 'payment_method' => $item['paymentType'], // Способ расчета
                 'payment_object' => $item['paymentSubject'], // Предмет расчета
             ];
         }, $items);
     }


     /**
      * Рассчитывает итоговую сумму товаров из массива.
      *
      * Функция принимает массив товаров, в котором у каждого товара есть поле "sum".
      * Суммирует значения всех полей "sum" и возвращает итоговую сумму.
      *
      * @param array $items Массив товаров, где каждый товар — это ассоциативный массив с ключом "sum".
      * @return float Итоговая сумма всех товаров.
      */
     private function calculate_total_sum(array $items): float
     {
         return array_reduce($items, function($carry, $item) {
             return $carry + ($item['sum'] ?? 0);
         }, 0);
     }
}
