<?php

namespace Vtbpay\Classes\Common;

(defined('ABSPATH') || PHP_SAPI == 'cli') or die('Restricted access');

use \Vtbpay\Classes\Exception\VtbPayException;

class Request
{
    /**
     * Дополнительные опции для cURL
     * Дочерние классы могут переопределить это свойство.
     *
     * @var array
     */
    protected array $extraCurlOptions = [];

    /**
     * Отправляет HTTP-запрос и возвращает декодированный JSON-ответ.
     *
     * @param string $url     URL запроса.
     * @param array  $data    Данные запроса.
     * @param array  $headers Заголовки запроса.
     * @param string $method  HTTP-метод (POST, PUT или GET).
     *
     * @return array Декодированный JSON-ответ.
     *
     * @throws VtbPayException Если произошла ошибка при выполнении запроса или декодировании ответа.
     */
    public function sendRequest(
        string $url,
        array $data = [],
        array $headers = [],
        string $method = 'POST'
    ): array {
        $logger = VtbPayLogger::getInstance();
        // Логирование входных параметров
        $logger->debug("INPUT: ", [
            'url'     => $url,
            'data'    => $data,
            'headers' => $headers,
        ]);

        // Подготовка данных и заголовков
        $encodedData = $this->encodeData($data, $headers);
        $httpHeaders = $this->formatHeaders($headers);

        // Выполнение запроса и обработка ошибок
        $response = $this->executeRequest($url, $encodedData, $httpHeaders, $method);

        // Логирование результата
        $logger->debug("OUTPUT: ", ['response' => $response]);

        return $response;
    }


    /**
     * Подготавливает данные запроса в зависимости от заголовка Content-Type.
     *
     * @param array $data    Данные запроса.
     * @param array $headers Заголовки запроса.
     *
     * @return string|array Данные, преобразованные в строку (JSON или URL-encoded), либо исходный массив.
     */
    private function encodeData(array $data, array $headers)
    {
        $contentType = $headers['Content-Type'] ?? $headers['Content-type'] ?? null;
        if ($contentType) {
            if (stripos($contentType, 'application/json') !== false) {
                return json_encode($data, JSON_PRESERVE_ZERO_FRACTION);
            } elseif (stripos($contentType, 'application/x-www-form-urlencoded') !== false) {
                return http_build_query($data);
            }
        }
        return $data;
    }


    /**
     * Форматирует ассоциативный массив заголовков в массив строк для cURL.
     *
     * @param array $headers Ассоциативный массив заголовков.
     *
     * @return array Массив строк вида "Ключ: Значение".
     */
    private function formatHeaders(array $headers): array
    {
        $formatted = [];
        foreach ($headers as $key => $value) {
            $formatted[] = "$key: $value";
        }
        return $formatted;
    }


    /**
     * Выполняет HTTP-запрос с использованием cURL.
     *
     * @param string       $url        URL запроса.
     * @param string|array $data       Данные запроса.
     * @param array        $httpHeaders Заголовки запроса.
     * @param string       $method     HTTP-метод (POST, PUT или GET).
     *
     * @return array Декодированный JSON-ответ.
     *
     * @throws VtbPayException Если произошла ошибка при выполнении запроса или декодировании ответа.
     */
    private function executeRequest(string $url, $data, array $httpHeaders, string $method): array
    {
        $curl = curl_init();

        $this->setCurlOptions($curl, $url, $data, $httpHeaders, $method);

        $result = curl_exec($curl);

        $info = curl_getinfo($curl);

        if ($result === false) {
            $error = curl_error($curl);
            curl_close($curl);
            throw new VtbPayException('Ошибка запроса.', ['curl_error' => $error]);
        }

        curl_close($curl);

        return $this->processResponse($result, $info);
    }


    /**
     * Устанавливает опции cURL в зависимости от HTTP-метода.
     *
     * @param resource     $curl        Ресурс cURL.
     * @param string       $url         URL запроса.
     * @param string|array $data        Данные запроса.
     * @param array        $httpHeaders Заголовки запроса.
     * @param string       $method      HTTP-метод (POST, PUT или GET).
     *
     * @throws VtbPayException Если указан неподдерживаемый HTTP-метод.
     */
    private function setCurlOptions($curl, string $url, $data, array $httpHeaders, string $method): void
    {
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($curl, CURLOPT_HTTPHEADER, $httpHeaders);

        switch (strtoupper($method)) {
            case 'POST':
                curl_setopt($curl, CURLOPT_POST, true);
                curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
                break;
            case 'PUT':
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'PUT');
                curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
                break;
            case 'GET':
                if (!empty($data)) {
                    $url .= '?' . (is_array($data) ? http_build_query($data) : $data);
                }
                break;
            default:
                throw new VtbPayException('Неподдерживаемый HTTP-метод: ' . $method);
        }

        curl_setopt($curl, CURLOPT_URL, $url);

        // Применение дополнительных cURL-опций из дочернего класса
        if (!empty($this->extraCurlOptions)) {
            curl_setopt_array($curl, $this->extraCurlOptions);
        }
    }


    /**
     * Декодирует JSON-ответ и проверяет на ошибки.
     *
     * @param string $result JSON-ответ.
     *
     * @return array Декодированный JSON-ответ.
     *
     * @throws VtbPayException Если произошла ошибка декодирования JSON.
     */
    protected function processResponse(string $result, array $info): array
    {
        if (empty($result)) {
            throw new VtbPayException('Запрос curl вернул пустой ответ.');
        }

        $decoded = json_decode($result, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new VtbPayException('Ошибка декодирования JSON.', [
                'json_last_error_msg' => json_last_error_msg(),
                'result'              => $result,
            ]);
        }
        return $decoded;
    }
}
