<?php

namespace Vtbpay\Classes\Api;

(defined('ABSPATH') || PHP_SAPI === 'cli') or die('Restricted access');

use Vtbpay\Classes\Common\Request,
    Vtbpay\Classes\Exception\VtbPayException;

/**
 * Класс OrangeDataApi — взаимодействие с API Orange Data по ФФД 1.05.
 */
class OrangeDataApi extends Request
{
    private array $companyData;
    private array $certificatesData;
    private array $apiEndpoints;

    /**
     * Конструктор OrangeDataApi.
     *
     * @param string $inn ИНН организации
     * @param string $taxSystem Система налогообложения.
     * @param string $group Группа ККТ
     * @param string $key Ключ авторизации
     * @param string $privateKeyPath Путь к приватному RSA-ключу
     * @param string $clientKeyPath Путь к client.key
     * @param string $clientCrtPath Путь к client.crt
     * @param string $caCertPath Путь к cacert.pem
     * @param string|null $certPassword Пароль от ключа (если есть)
     * @param bool $test Использовать тестовый режим
     * @throws VtbPayException
     */
    public function __construct(
        string $inn,
        string $taxSystem,
        string $group,
        string $key,
        string $privateKeyPath,
        string $clientKeyPath,
        string $clientCrtPath,
        string $caCertPath,
        ?string $certPassword = null,
        bool $test = false
    ) {
        $this->companyData = compact('inn', 'taxSystem', 'group', 'key');
        $this->certificatesData = [
            'private_key' => $privateKeyPath,
            'client_key'  => $clientKeyPath,
            'client_crt'  => $clientCrtPath,
            'ca_cert'     => $caCertPath,
            'password'    => $certPassword
        ];

        $this->validateCertificates();

        $this->setExtraCurlOptions();

        $this->setApiEndpoints($test ? 'TEST' : 'PROD');
    }


    /**
     * Проверка доступности сертификатов.
     *
     * @throws VtbPayException
     */
    private function validateCertificates(): void
    {
        $requiredCerts = ['private_key', 'client_key', 'client_crt', 'ca_cert'];

        foreach ($requiredCerts as $key) {
            $path = $this->certificatesData[$key] ?? '';
            if (empty($path)) {
                throw new VtbPayException('Не указан путь к сертификату.', [
                    'key' => $key
                ]);
            }

            $path = realpath($path);
            if (!$path || !is_readable($path)) {
                throw new VtbPayException('Сертификат недоступен для чтения.', [
                    'path' => $path
                ]);
            }
            $this->certificatesData[$key] = $path; // нормализуем путь
        }
    }


    /**
     * Дополнительные параметры CURL для сертификатов.
     *
     * @return array
     */
    private function setExtraCurlOptions(): void
    {
        $options = [
            CURLOPT_SSLCERT => $this->certificatesData['client_crt'],
            CURLOPT_SSLKEY  => $this->certificatesData['client_key'],
            CURLOPT_CAINFO  => $this->certificatesData['ca_cert'],
        ];

        if (!empty($this->certificatesData['password'])) {
            $options[CURLOPT_SSLCERTPASSWD] = $this->certificatesData['password'];
        }

        $this->extraCurlOptions = $options;
    }


    /**
     * Установка API-эндпоинтов из переменных окружения.
     *
     * @param string $mode TEST или PROD
     * @throws VtbPayException
     */
    private function setApiEndpoints(string $mode): void
    {
        foreach (['RECEIPT', 'STATUS'] as $type) {
            $endpoint = $_ENV['ORANGEDATA_' . $mode . '_' . $type] ?? '';

            if (empty($endpoint)) throw new VtbPayException('Не указан обязательный эндпоинт Orange Data.', [
                'mode' => $mode,
                'type' => $type
            ]);

            $this->apiEndpoints[$type] = rtrim($endpoint, '/');
        }
    }


    /**
     * Получить URL по типу запроса.
     *
     * @param string $type Тип (RECEIPT или STATUS)
     * @return string
     */
    private function getUrl(string $type): string
    {
        return $this->apiEndpoints[$type] ?? '';
    }


    /**
     * Подписывает JSON-строку приватным RSA-ключом.
     *
     * @param string $json Строка JSON
     * @return string Базовая64 подпись
     * @throws VtbPayException
     */
    private function signRequest(string $json): string
    {
        $privateKey = file_get_contents($this->certificatesData['private_key']);
        if (!$privateKey) {
            throw new VtbPayException('Не удалось загрузить приватный ключ.');
        }

        $signature = '';
        if (!openssl_sign($json, $signature, $privateKey, OPENSSL_ALGO_SHA256)) {
            throw new VtbPayException('Ошибка подписи openssl_sign.');
        }

        return base64_encode($signature);
    }


    /**
     * Отправка фискального чека.
     *
     * @param string $document_id ID документа
     * @param array $positions Позиции товаров
     * @param float $total Общая сумма
     * @param string $customerContact Телефон или Email
     * @param string $callbackUrl URL для уведомлений
     * @param bool $isReturn Возврат или приход
     * @return array Ответ API
     * @throws VtbPayException
     */
    public function sendReceipt(
        string $document_id,
        array $positions,
        float $total,
        string $customerContact,
        string $callbackUrl,
        bool $isReturn = false
    ): array {
        $payload = [
            'id' => $document_id . '-' . ($isReturn ? 'REFUND' : 'RECEIPT'),
            'inn' => $this->companyData['inn'],
            'group' => $this->companyData['group'],
            'key' => $this->companyData['key'],
            'callbackUrl' => $callbackUrl,
            'content' => [
                'type' => ($isReturn ? 2 : 1),
                'positions' => $positions,
                'checkClose' => [
                    'payments' => [[
                        'type' => 2,
                        'amount' => $total
                    ]],
                    'taxationSystem' => $this->companyData['taxSystem']
                ],
                'customerContact' => $customerContact,
                'totalSum' => $total,
                'ffdVersion' => 2,
                'internetStore' => true
            ]
        ];

        $json = json_encode($payload, JSON_PRESERVE_ZERO_FRACTION);
        $signature = $this->signRequest($json);

        $response = $this->sendRequest(
            $this->getUrl('RECEIPT'),
            $payload,
            [
                'Content-Type' => 'application/json; charset=utf-8',
                'X-Signature'   => $signature,
                'Content-Length'=> strlen($json)
            ]
        );

        $this->validateResponseSendReceipt($response);

        return $response;
    }


    /**
     * Проверяет ответ запроса на добавление чека в очередь печати.
     *
     * @param array $response Массив резульата запроса
     *
     * @return void|VtbPayException Выбрасывает ошибку если статус не равен 201.
     */
    private function validateResponseSendReceipt(array $response): void
    {
        if (
            !isset($response['status']) ||
            $response['status'] !== 201
        ) {
            throw new VtbPayException('Ошибка при отправлении чека.', [
                'response' => $response
            ]);
        }
    }


    /**
     * Получение информации о чеке по ID.
     *
     * @param string $document_id Идентификатор чека
     * @return array Ответ API
     * @throws VtbPayException
     */
    public function getReceiptInfo(string $document_id): array
    {
        $urlTemplate = $this->getUrl('STATUS');
        $url = str_replace(
            ['<inn>', '<document_id>'],
            [$this->companyData['inn'], $document_id],
            $urlTemplate
        );

        return $this->sendRequest($url, [], [], 'GET');
    }


    /**
     * Переопределение базового метода обработки ответа.
     *
     * Декодирует JSON-ответ, выбрасывает исключение при ошибке декодирования.
     * Если тело ответа пустое, возвращает массив с HTTP статусом.
     *
     * @param string $result Тело ответа
     * @param array  $info   Информация о запросе (curl_getinfo)
     *
     * @return array Декодированный ответ или массив с HTTP статусом
     *
     * @throws VtbPayException При ошибке декодирования JSON
     */
    protected function processResponse(string $result, array $info): array
    {
        $final_result = [
            'status' => $info['http_code'] ?? 400,
        ];
        if (!empty($result)) {
            $final_result = $final_result + (json_decode($result, true) ?: []);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new VtbPayException('Ошибка декодирования JSON.', [
                    'json_last_error_msg' => json_last_error_msg(),
                    'result'              => $result,
                ]);
            }
        }

        return $final_result;
    }
}
