<?php

namespace Vtbpay\Classes\Api;

(defined('ABSPATH') || PHP_SAPI == 'cli') or die('Restricted access');

use \Vtbpay\Classes\Common\Request,
    \Vtbpay\Classes\Exception\VtbPayException;

/**
 * Класс AtolApi предоставляет методы для взаимодействия с API АТОЛ.
 */
class AtolApi extends Request
{
    private string $accessToken;
    private array $companyData = [];
    private array $apiEndpoints;


    /**
     * Конструктор класса AtolApi.
     *
     * @param string $login         Логин интеграции АТОЛ.
     * @param string $pass          Пароль интеграции АТОЛ.
     * @param string $INN           ИНН организации мерчанта.
     * @param string $taxSystem     Система налогообложения.
     * @param string $groupIdKKT    Идентификатор группы ККТ.
     * @param string $companyEmail  Email компании.
     * @param bool $test            Режим тестирования (по умолчанию - false).
     *
     * @throws VtbPayException Если не заполнены обязательные поля.
     */
    public function __construct(
        string $login,
        string $pass,
        string $INN,
        string $taxSystem,
        string $groupIdKKT,
        string $companyEmail,
        bool $test = false
    ) {
        if (
            empty($login) ||
            empty($pass) ||
            empty($INN) ||
            empty($groupIdKKT)
        ) throw new VtbPayException('Обязательные поля не заполнены.');

        $this->companyData = [
            'INN' => trim($INN),
            'groupIdKKT' => trim($groupIdKKT),
            'taxSystem' => $taxSystem,
            'email' => trim($companyEmail)
        ];

        $this->setApiEndpoints($test ? 'TEST' : 'PROD');

        $this->accessToken = $this->getAccessToken(
            trim($login),
            trim($pass)
        );
    }


    /**
     * Устанавливает эндпоинты API АТОЛ
     *
     * @return void
     */
    private function setApiEndpoints(string $mode): void
    {
        // Список типов запроса
        foreach (['TOKEN', 'RECEIPT', 'REPORT', 'REFUND'] as $type) {
            $endpoint = $_ENV['ATOL_' . $mode . '_' . $type] ?? '';

            if (empty($endpoint)) throw new VtbPayException('Не указан обязательный эндпоинт API АТОЛ.', [
                'mode' => $mode,
                'type' => $type
            ]);

            // Для прода
            $this->apiEndpoints[$type] = str_replace(
                '<group_code>',
                $this->companyData['groupIdKKT'],
                $endpoint
            );
        }
    }


    /**
     * Получает URL для заданного типа запроса.
     *
     * @param string $type Тип запроса ('TOKEN', 'RECEIPT', 'REPORT' или 'REFUND').
     *
     * @return string URL для запроса.
     */
    private function getUrl(string $type): string
    {
        return $this->apiEndpoints[$type] ?? '';
    }


    /**
     * Делает запрос в АТОЛ для полученя токена и возвращает его.
     *
     * @param string $login   Логин интеграции АТОЛ
     * @param string $pass    Пароль интеграции АТОЛ
     *
     * @return string Токен
     */
    private function getAccessToken(string $login, string $pass): string
    {
        // getToken request
        $response = $this->sendRequest(
            $this->getUrl('TOKEN'), [
                'login' => $login,
                'pass' => $pass
            ], [
                'Content-type' => 'application/json; charset=utf-8'
            ]
        );

        $this->validateResponseAccessToken($response);

        return $response['token'];
    }


    /**
     * Проверяет ответ запроса в АТОЛ на получение токена.
     *
     * @param array $response Массив резульата запроса
     *
     * @return void|VtbPayException Выбрасывает ошибку если токена нет.
     */
    private function validateResponseAccessToken(array $response): void
    {
        if (
            !isset($response['token']) ||
            empty($response['token'])
        ) {
            throw new VtbPayException('Ошибка при получении токена доступа.', [
                'response' => $response
            ]);
        }
    }


    /**
     * Запрос на добавление чека в очередь печати.
     *
     * @param string $orderId   Идентификатор заказа
     * @param array $items      Позиции чека
     * @param float $total      Сумма заказа
     * @param string $name      Полное имя клиента
     * @param string $clientEmail      Электронная почта клиента
     * @param bool $isReturn    Данный чек возврата или нет
     *
     * @return array Результат запроса
     */
    public function sendReceipt(
        string $orderId,
        array $items,
        float $total,
        string $clientEmail,
        string $callbackUrl,
        bool $isReturn = false
    ): array {
        $operationType = $isReturn ? 'REFUND' : 'RECEIPT';
        $response = $this->sendRequest(
            $this->getUrl($operationType), [
                'timestamp' => date('d.m.Y H:i:s', time()),
                'external_id' => $orderId . '-' . $operationType,
                'service' => [
                    'callback_url' => $callbackUrl
                ],
                'ism_optional' => true,
                'receipt' => [
                    'client' => [
                        'email' => $clientEmail
                    ],
                    'company' => [
                        'email' => $this->companyData['email'],
                        'sno' => $this->companyData['taxSystem'],
                        'inn' => $this->companyData['INN'],
                        'payment_address' => $_SERVER['HTTP_HOST'] ?? $_SERVER['SERVER_NAME']
                    ],
                    'items' => $items,
                    'payments' => [
                        [
                            'type' => 1,
                            'sum' => $total
                        ]
                    ],
                    'total' => $total,
                    'internet' => true
                ]
            ], [
                'Content-type' => 'application/json; charset=utf-8',
                'Token' => $this->accessToken
            ]
        );

        $this->validateResponseSendReceipt($response);

        return $response;
    }


    /**
     * Проверяет ответ запроса на добавление чека в очередь печати.
     *
     * @param array $response Массив резульата запроса
     *
     * @return void|VtbPayException Выбрасывает ошибку если статус равен fail.
     */
    private function validateResponseSendReceipt(array $response): void
    {
        if (
            !isset($response['status']) ||
            $response['status'] === 'fail'
        ) {
            throw new VtbPayException('Ошибка при отправлении чека.', [
                'response' => $response
            ]);
        }
    }


    /**
     * Получение результата обработки документа.
     *
     * @param string $uuid Массив резульата запроса
     *
     * @return array Результат запроса
     */
    public function getReceiptInfo(string $uuid): array
    {
        $response = $this->sendRequest(
            str_replace('<uuid>', $uuid, $this->getUrl('REPORT')),
            [], [
                'Token' => $this->accessToken
            ],
            'GET'
        );

        $this->validateResponseGetReceiptInfo($response);

        return $response;
    }


    /**
     * Проверяет ответ запроса в АТОЛ на получение результата обработки документа.
     *
     * @param array $response Массив резульата запроса
     *
     * @return void|VtbPayException Выбрасывает ошибку если документ не найден.
     */
    private function validateResponseGetReceiptInfo(array $response): void
    {
        if (
            !isset($response['uuid']) ||
            empty($response['uuid'])
        ) {
            throw new VtbPayException('Ошибка получения результата обработки документа.', [
                'response' => $response
            ]);
        }
    }
}
