<?php
/**
 * Функция add_vtbpay_gateway
 *
 * Добавляет платёжный метод "WC_Vtbpay" в список доступных платёжных методов WooCommerce.
 *
 * @param array $methods - Список доступных платёжных методов
 * @return array - Модифицированный список платёжных методов с добавлением WC_Vtbpay
 */
function add_vtbpay_gateway(array $methods): array
{
    $methods[] = 'WC_Vtbpay';
    return $methods;
}
add_filter('woocommerce_payment_gateways', 'add_vtbpay_gateway');


/**
 * Функция load_vtbpay_textdomain
 *
 * Загружает локализацию (текстовый перевод) для плагина WC_Vtbpay.
 */
function load_vtbpay_textdomain(): void
{
    load_plugin_textdomain('wc-vtbpay', false, dirname(plugin_basename(__FILE__)) . '/languages');
}
add_action('plugins_loaded', 'load_vtbpay_textdomain');


/**
 * Функция add_vtbpay_tab_custom_product_config
 *
 * Добавляет вкладку с настройками WC_Vtbpay к настройкам продукта в административной части WooCommerce.
 *
 * @param array $tabs - Список вкладок на странице редактирования продукта
 * @return array - Модифицированный список вкладок с добавлением WC_Vtbpay
 */
function add_vtbpay_tab_custom_product_config(array $tabs): array
{
    // Проверяет наличие настроек WC_Vtbpay и условие активации для добавления вкладки
    $vtbpay_settings = get_option('woocommerce_wc_vtbpay_settings');

    if (
        isset($vtbpay_settings['enable_fiscal']) &&
        $vtbpay_settings['enable_fiscal'] === 'yes'
    ) {
        // Добавляет вкладку с настройками WC_Vtbpay
        $tabs['custom_tab'] = [
            'label'    => __('Fiscalization settings (Payment system VTB)', 'wc-vtbpay'),
            'target'   => 'vtbpay_custom_product_data',
            'class'    => ['show_if_simple', 'show_if_variable'],
        ];
    }

    return $tabs;
}
add_filter('woocommerce_product_data_tabs', 'add_vtbpay_tab_custom_product_config');


/**
 * Функция create_custom_container_vtbpay_settings
 *
 * Создает контейнер с настройками WC_Vtbpay на странице редактирования продукта в административной части WooCommerce.
 */
function create_custom_container_vtbpay_settings(): void
{
    // Проверяет наличие настроек WC_Vtbpay и условие активации для отображения контейнера
    $vtbpay_settings = get_option('woocommerce_wc_vtbpay_settings');

    if (
        isset($vtbpay_settings['enable_fiscal']) &&
        $vtbpay_settings['enable_fiscal'] === 'yes'
    ) {
        echo '<div id="vtbpay_custom_product_data" class="panel woocommerce_options_panel">';

        // Определяет поля настроек и добавляет их на страницу редактирования продукта
        $fields = [
            'vtbpay_measure_fiscal' => __('Units of measurement', 'wc-vtbpay'),
            'vtbpay_tax_type_fiscal' => __('Tax rate', 'wc-vtbpay'),
            'vtbpay_payment_type_fiscal' => __('Calculation method', 'wc-vtbpay'),
            'vtbpay_payment_subject_fiscal' => __('Subject of calculation', 'wc-vtbpay')
        ];

        foreach ($fields as $field_id => $field_label) {
            woocommerce_wp_select([
                'id' => $field_id,
                'label' => $field_label,
                'options' => get_vtbpay_field_options($field_id),
            ]);
        }

        echo '</div>';
    }
}
add_action('woocommerce_product_data_panels', 'create_custom_container_vtbpay_settings');


/**
 * Функция get_vtbpay_field_options
 *
 * Возвращает список доступных опций для полей настроек WC_Vtbpay на основе идентификатора поля.
 *
 * @param string $field_id - Идентификатор поля настроек
 * @return array - Список опций для поля
 */
function get_vtbpay_field_options($field_id): array
{
    $options = [
        '-' => __('Select an option', 'wc-vtbpay')
    ];

    // Определяет опции для конкретного поля
    switch ($field_id) {
        case 'vtbpay_measure_fiscal':
            // Специально добавлено окончание для ключей, так как с цифрами ошибки
            $options = array_merge($options, [
                '0_measure_fiscal' => __('Applies to items that can be sold by piece or unit', 'wc-vtbpay'),
                '10_measure_fiscal' => __('Gram', 'wc-vtbpay'),
                '11_measure_fiscal' => __('Kilogram', 'wc-vtbpay'),
                '12_measure_fiscal' => __('Ton', 'wc-vtbpay'),
                '20_measure_fiscal' => __('Centimeter', 'wc-vtbpay'),
                '21_measure_fiscal' => __('Decimeter', 'wc-vtbpay'),
                '22_measure_fiscal' => __('Meter', 'wc-vtbpay'),
                '30_measure_fiscal' => __('Square centimeter', 'wc-vtbpay'),
                '31_measure_fiscal' => __('Square decimeter', 'wc-vtbpay'),
                '32_measure_fiscal' => __('Square meter', 'wc-vtbpay'),
                '40_measure_fiscal' => __('Milliliter', 'wc-vtbpay'),
                '41_measure_fiscal' => __('Liter', 'wc-vtbpay'),
                '42_measure_fiscal' => __('Cubic meter', 'wc-vtbpay'),
                '50_measure_fiscal' => __('Kilowatt hour', 'wc-vtbpay'),
                '51_measure_fiscal' => __('Gcal', 'wc-vtbpay'),
                '70_measure_fiscal' => __('Day', 'wc-vtbpay'),
                '71_measure_fiscal' => __('Hour', 'wc-vtbpay'),
                '72_measure_fiscal' => __('Minute', 'wc-vtbpay'),
                '73_measure_fiscal' => __('Second', 'wc-vtbpay'),
                '80_measure_fiscal' => __('Kilobyte', 'wc-vtbpay'),
                '81_measure_fiscal' => __('Megabyte', 'wc-vtbpay'),
                '82_measure_fiscal' => __('Gigabyte', 'wc-vtbpay'),
                '83_measure_fiscal' => __('Terabyte', 'wc-vtbpay'),
                '255_measure_fiscal' => __('Applies when using other units of measurement', 'wc-vtbpay')
            ]);
            break;

        case 'vtbpay_tax_type_fiscal':
            $options = array_merge($options, [
                'none' => __('Without VAT', 'wc-vtbpay'),
                'vat0' => __('VAT at 0%', 'wc-vtbpay'),
                'vat10' => __('VAT at 10%', 'wc-vtbpay'),
                'vat110' => __('VAT at 10/110 calculated rate', 'wc-vtbpay'),
                'vat20' => __('VAT at 20%', 'wc-vtbpay'),
                'vat120' => __('VAT at 20/120 calculated rate', 'wc-vtbpay'),
                'vat5' => __('VAT at 5%', 'wc-vtbpay'),
                'vat7' => __('VAT at 7%', 'wc-vtbpay'),
                'vat105' => __('VAT at 5/105 calculated rate', 'wc-vtbpay'),
                'vat107' => __('VAT at 7/107 calculated rate', 'wc-vtbpay')
            ]);
            break;

        case 'vtbpay_payment_type_fiscal':
            $options = array_merge($options, [
                'full_prepayment' => __('Prepayment 100%', 'wc-vtbpay'),
                'prepayment' => __('Prepayment', 'wc-vtbpay'),
                'advance' => __('Advance', 'wc-vtbpay'),
                'full_payment' => __('Full payment', 'wc-vtbpay'),
                'partial_payment' => __('Partial payment and credit', 'wc-vtbpay'),
                'credit' => __('Credit transfer', 'wc-vtbpay'),
                'credit_payment' => __('Credit payment', 'wc-vtbpay')
            ]);
            break;

        case 'vtbpay_payment_subject_fiscal':
            // Специально добавлено окончание для ключей, так как с цифрами ошибки
            $options = array_merge($options, [
                '1_payment_subject_fiscal' => __('Goods', 'wc-vtbpay'),
                '2_payment_subject_fiscal' => __('Excise goods', 'wc-vtbpay'),
                '3_payment_subject_fiscal' => __('Work', 'wc-vtbpay'),
                '4_payment_subject_fiscal' => __('Service', 'wc-vtbpay'),
                '5_payment_subject_fiscal' => __('Gambling bet', 'wc-vtbpay'),
                '6_payment_subject_fiscal' => __('Gambling win', 'wc-vtbpay'),
                '7_payment_subject_fiscal' => __('Lottery bet', 'wc-vtbpay'),
                '8_payment_subject_fiscal' => __('Lottery win', 'wc-vtbpay'),
                '9_payment_subject_fiscal' => __('Granting rights', 'wc-vtbpay'),
                '10_payment_subject_fiscal' => __('Payment', 'wc-vtbpay'),
                '11_payment_subject_fiscal' => __('Agency fee', 'wc-vtbpay'),
                '12_payment_subject_fiscal' => __('Payout', 'wc-vtbpay'),
                '13_payment_subject_fiscal' => __('Other calculation subject', 'wc-vtbpay'),
                '14_payment_subject_fiscal' => __('Property right', 'wc-vtbpay'),
                '15_payment_subject_fiscal' => __('Non-operating income', 'wc-vtbpay'),
                '16_payment_subject_fiscal' => __('Insurance contributions', 'wc-vtbpay'),
                '17_payment_subject_fiscal' => __('Trade fee', 'wc-vtbpay'),
                '18_payment_subject_fiscal' => __('Resort fee', 'wc-vtbpay'),
                '19_payment_subject_fiscal' => __('Deposit', 'wc-vtbpay'),
                '20_payment_subject_fiscal' => __('Expense', 'wc-vtbpay'),
                '21_payment_subject_fiscal' => __('Pension contributions for IP', 'wc-vtbpay'),
                '22_payment_subject_fiscal' => __('Pension contributions', 'wc-vtbpay'),
                '23_payment_subject_fiscal' => __('Health insurance contributions for IP', 'wc-vtbpay'),
                '24_payment_subject_fiscal' => __('Health insurance contributions', 'wc-vtbpay'),
                '25_payment_subject_fiscal' => __('Social insurance contributions', 'wc-vtbpay'),
                '26_payment_subject_fiscal' => __('Casino payment', 'wc-vtbpay'),
                '27_payment_subject_fiscal' => __('Cash issuance by bank payment agent', 'wc-vtbpay'),
                '30_payment_subject_fiscal' => __('Excise goods subject to marking without code', 'wc-vtbpay'),
                '31_payment_subject_fiscal' => __('Excise goods subject to marking with code', 'wc-vtbpay'),
                '32_payment_subject_fiscal' => __('Goods subject to marking without code except excise goods', 'wc-vtbpay'),
                '33_payment_subject_fiscal' => __('Goods subject to marking with code except excise goods', 'wc-vtbpay')
            ]);
            break;
    }

    return $options;
}


/**
 * Функция save_value_custom_product_fields_from_vtbpay_tab
 *
 * Сохраняет значения пользовательских полей настроек WC_Vtbpay на странице редактирования продукта.
 *
 * @param int $post_id - Идентификатор продукта
 */
function save_value_custom_product_fields_from_vtbpay_tab($post_id): void
{
    // Проверяет наличие настроек WC_Vtbpay и условие активации для сохранения полей
    $vtbpay_settings = get_option('woocommerce_wc_vtbpay_settings');

    if (
        isset($vtbpay_settings['enable_fiscal']) &&
        $vtbpay_settings['enable_fiscal'] === 'yes'
    ) {
        // Определяет поля, получает их значения из запроса и сохраняет в метаданных продукта
        $fields = [
            'vtbpay_measure_fiscal',
            'vtbpay_tax_type_fiscal',
            'vtbpay_payment_type_fiscal',
            'vtbpay_payment_subject_fiscal'
        ];

        foreach ($fields as $field) {
            $value = sanitize_text_field($_POST[$field] ?? '');
            update_post_meta($post_id, $field, $value);
        }
    }
}
add_action('woocommerce_process_product_meta', 'save_value_custom_product_fields_from_vtbpay_tab');


/**
 * Функция adding_vtbpay_button_to_orders_page
 *
 * Добавляет скрипты и стили к странице заказов и настройкам WC_Vtbpay в административной части.
 *
 * @param string $hook - Идентификатор страницы в административной части WordPress
 */
function adding_vtbpay_button_to_orders_page($hook): void
{
    global $post;
    // Проверяет наличие настроек WC_Vtbpay и условия для добавления скриптов и стилей
    $vtbpay_settings = get_option('woocommerce_wc_vtbpay_settings');

    if (
        isset($vtbpay_settings['enable_fiscal']) &&
        $vtbpay_settings['enable_fiscal'] === 'yes' &&
        ($hook == 'post-new.php' || $hook == 'post.php') &&
        !empty($post) && $post->post_type === 'shop_order'
    ) {
        $order_id = $post->ID;
        $order = wc_get_order($order_id);
        $order_status = $order->get_status();
        $payment_method = $order->get_payment_method();

        if (
            !in_array($order_status, ['failed', 'refunded', 'cancelled']) &&
            $payment_method === 'wc_vtbpay'
        ) {
            // Подключает скрипт для работы с заказами WC_Vtbpay
            wp_enqueue_script('vtbpay-order-page-script', plugins_url('/assets/js/vtbpay-order-page-script.js', __FILE__), ['jquery'], '1.0', true);
            wp_localize_script(
                'vtbpay-order-page-script',
                'vtbpay_ajax_var', [
                'nonce' => wp_create_nonce('vtbpay-ajax-nonce'),
                'api_url' => home_url('/wc-api/wc_vtbpay_ajax')
            ]);
        }
    }

    if (
        $hook == 'woocommerce_page_wc-settings' &&
        isset($_GET['tab']) && $_GET['tab'] === 'checkout' &&
        isset($_GET['section']) && $_GET['section'] === 'wc_vtbpay'
    ) {
        // Подключает скрипт для страницы настроек WC_Vtbpay
        wp_enqueue_script('vtbpay-settings-page-main-script', plugins_url('/assets/js/vtbpay-settings-page-main-script.js', __FILE__), ['jquery'], '1.0', true);
        wp_enqueue_script('vtbpay-settings-page-upload-file-script', plugins_url('/assets/js/vtbpay-settings-page-upload-file-script.js', __FILE__), ['vtbpay-settings-page-main-script'], '1.0', true);
        wp_enqueue_style('vtbpay-settings-page-style', plugins_url('/assets/css/vtbpay-settings-page-style.css', __FILE__));
    }
}
add_action('admin_enqueue_scripts', 'adding_vtbpay_button_to_orders_page');


/**
 * Убирает ограничения по типу файла при загрузке .key, .pem, .crt, .cer
 * Позволяет загружать сертификаты и ключи, задавая корректные MIME-типы.
 *
 * @param array  $data     Ассоциативный массив с ключами 'ext' и 'type'.
 * @param string $file     Путь к файлу.
 * @param string $filename Имя загружаемого файла.
 * @param array  $mimes    Разрешённые MIME-типы.
 *
 * @return array Модифицированные данные MIME-типа и расширения.
 */
function vtbpay_fix_filetype_check($data, $file, $filename, $mimes) {
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

    // Кастомные расширения и соответствующие MIME-типы
    $custom_mimes = [
        'key' => 'application/octet-stream',
        'pem' => 'application/x-pem-file',
        'crt' => 'application/x-x509-ca-cert',
        'cer' => 'application/x-x509-ca-cert',
    ];

    if (isset($custom_mimes[$ext])) {
        $data['ext']  = $ext;
        $data['type'] = $mimes[$ext] ?? $custom_mimes[$ext];
    }

    return $data;
}

add_filter('wp_check_filetype_and_ext', 'vtbpay_fix_filetype_check', 10, 4);
