(function ($) {
    // --- Конфигурация ОФД ---
    const prefix = '#woocommerce_wc_vtbpay_';

    // ОФД, которые нужно обрабатывать (без дефолтного 1ofd)
    const ofdList = {
        atol: {
            postfix: '_atol_fiscal',
            requiredFields: ['login_atol_fiscal', 'pass_atol_fiscal', 'kkt_atol_fiscal', 'inn_atol_fiscal']
        },
        orange_data: {
            postfix: '_orange_data_fiscal',
            requiredFields: ['signature_key_orange_data_fiscal', 'group_orange_data_fiscal', 'inn_orange_data_fiscal']
        }
    };

    // Основные элементы интерфейса
    const fiscalDropdown = $(prefix + 'ofd_fiscal');
    const testModeMainCheckbox = $(prefix + 'test_mode');

    /**
     * Применяет к полям класса required/required=false
     * @param {string} ofdCode - код ОФД
     * @param {boolean} isRequired - делать ли поля обязательными
     */
    function setRequiredFields(ofdCode, isRequired) {
        const fields = ofdList[ofdCode]?.requiredFields || [];

        fields.forEach(field => {
            const fieldElement = $(prefix + field);
            fieldElement.prop('required', isRequired);

            const labelElement = $('label[for="' + fieldElement.attr('id') + '"]');
            labelElement.toggleClass('required', isRequired);
        });
    }

    /**
     * Скрывает или показывает все элементы, относящиеся к конкретному ОФД
     * @param {string} ofdCode
     * @param {boolean} isVisible
     */
    function toggleFieldVisibilityByPostfix(ofdCode, isVisible) {
        const postfix = ofdList[ofdCode]?.postfix;
        if (!postfix) return;

        const elements = $(`input[id$="${postfix}"], select[id$="${postfix}"]`).not(fiscalDropdown);

        elements.each(function () {
            let container = $(this);
            if (!container.hasClass('wc-settings-sub-title')) {
                container = container.closest('tr');
            }
            container.toggle(isVisible);
        });
    }

    /**
     * Основной обработчик изменения выбранного ОФД
     */
    function handleOfdChange() {
        const selectedOfd = fiscalDropdown.val();

        // Обрабатываем каждый офд
        Object.keys(ofdList).forEach(ofdCode => {
            const isCurrent = ofdCode === selectedOfd;
            toggleFieldVisibilityByPostfix(ofdCode, isCurrent);
            setRequiredFields(ofdCode, isCurrent);
        });
    }

    /**
     * Проверка тестового режима: если включен только один флаг — выдаём предупреждение
     */
    function checkTestModeMismatch() {
        const selectedOfd = fiscalDropdown.val();
        const ofdTestCheckbox = $(prefix + `test_mode_${selectedOfd}_fiscal`);
        const isTestMain = testModeMainCheckbox.is(':checked');
        const isTestOfd = ofdTestCheckbox.length ? ofdTestCheckbox.is(':checked') : false;

        if ((isTestMain || isTestOfd) && !(isTestMain && isTestOfd)) {
            alert(`Обратите внимание: включен тестовый режим только у одного компонента — платежной системы или ОФД.`);
        }
    }

    /**
     * Инициализация обработчиков событий
     */
    function initEventListeners() {
        fiscalDropdown.on('change', handleOfdChange);

        $('button[name="save"]').on('click', function () {
            const selectedOfd = fiscalDropdown.val();
            if (ofdList[selectedOfd]) {
                checkTestModeMismatch();
            }
        });
    }

    /**
     * Первичный запуск логики при загрузке страницы
     */
    function init() {
        initEventListeners();
        handleOfdChange(); // Установить нужные поля при загрузке
    }

    // Запускаем, когда документ готов
    $(document).ready(init);

})(jQuery);
