<?php
(defined('_JEXEC') || PHP_SAPI == 'cli') or die('Restricted access');

// Checking PHP Version
if (version_compare(PHP_VERSION, '7.4.0', '<')) {
    die('Payment module "Payment system VTB" requires PHP version 7.4.0 or higher.');
}

require_once __DIR__ . '/vendor/autoload.php';

if (!class_exists('vmPSPlugin')) {
    require(JPATH_VM_ADMINISTRATOR . DS . 'plugins' . DS . 'vmpsplugin.php');
}

// Include the logger library
jimport('joomla.log.log');

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Common\EventDispatcher,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Vtbpay\Traits\Atol,
    \Symfony\Component\Dotenv\Dotenv;

class plgVmPaymentVtbpay extends vmPSPlugin
{
    use Atol;

    const CMS_NAME = 'Joomla! VirtueMart';
    const PLUGIN_VERSION = '1.6.9';

    /**
     * @var VtbPayLogger Объект для логирования.
     */
    private VtbPayLogger $logger;
    private object $payment_method;
    private array $order;
    private EventDispatcher $event_dispatcher;


    /**
     * Constructor of the plgVmPaymentVtbpay class.
     * Initializes the class, setting up necessary configurations, logger, and form fields.
     *
     * @param mixed $subject Объект для прослушивания события
     * @param array $config  Массив настроек.
     */
    public function __construct(& $subject, $config)
    {
        parent::__construct($subject, $config);

        // Получаем настройки из .env в корне
        $env_path = __DIR__ . '/config/.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);

        $this->event_dispatcher = new EventDispatcher($this);

        // unique filelanguage
        $jlang = JFactory::getLanguage();
        $jlang->load('plg_vmpayment_vtbpay', JPATH_ADMINISTRATOR, NULL, TRUE);

        $this->_loggable = true;
        $this->tableFields = array_keys($this->getTableSQLFields());
        $this->_tablepkey = $this->_tableId = 'id';
        $this->setConfigParameterable($this->_configTableFieldName, [
            'vtbpay_client_id' => ['', 'char'],
            'vtbpay_client_secret' => ['', 'char'],
            'vtbpay_test_mode' => ['', 'int'],
            'vtbpay_merchant_authorization' => ['', 'char'],
            'transaction_end_status' => ['', 'char'],
            'vtbpay_logging' => [0, 'int'],
            'vtbpay_two_stage' => [0, 'int'],
            'vtbpay_callback_url' => ['', 'char'],
            'vtbpay_enable_fiscal' => [0, 'int'],
            'vtbpay_ofd_fiscal' => ['', 'char'],
            'vtbpay_test_mode_atol_fiscal' => [0, 'int'],
            'vtbpay_login_atol_fiscal' => ['', 'char'],
            'vtbpay_pass_atol_fiscal' => ['', 'char'],
            'vtbpay_kkt_atol_fiscal' => ['', 'char'],
            'vtbpay_inn_atol_fiscal' => ['', 'char'],
            'vtbpay_tax_system_atol_fiscal' => ['', 'char'],
            'vtbpay_email_fiscal' => ['', 'char'],
            'vtbpay_delivery_fiscal' => ['', 'char'],
            'vtbpay_measure_fiscal' => ['', 'char'],
            'vtbpay_tax_fiscal' => ['', 'char'],
            'vtbpay_payment_fiscal' => ['', 'char'],
            'vtbpay_subject_fiscal' => [0, 'int']
        ]);
        JLog::addLogger(
            ['text_file' => 'plg_vmpayment_vtbpay-' . date('d-m-Y') . '.php'],
            JLog::ALL,
            ['plg_vmpayment_vtbpay']
        );
        JForm::addFieldPath(JPATH_PLUGINS . '/vmpayment/vtbpay/fields');
    }


    /**
     * Creates the table for this plugin if it doesn't exist.
     * Calls the method 'createTableSQL' with the name of the table to create.
     *
     * @return string SQL query to create the table.
     */
    function getVmPluginCreateTableSQL()
    {
        return $this->createTableSQL('Payment VTB Pay Table');
    }


    /**
     * Defines the fields for the payment table.
     *
     * @return array The SQL fields for the payment table.
     */
    function getTableSQLFields()
    {
        return [
            'id' => 'int(11) UNSIGNED NOT NULL AUTO_INCREMENT',
            'virtuemart_order_id' => 'int(1) UNSIGNED',
            'order_number' => 'char(64)',
            'virtuemart_paymentmethod_id' => 'mediumint(1) UNSIGNED',
            'payment_name' => 'char(255) NOT NULL DEFAULT \'\' ',
            'payment_order_total' => 'decimal(15,5) NOT NULL DEFAULT \'0.00000\' ',
            'payment_currency' => 'char(3) ',
            'cost_per_transaction' => 'decimal(10,2)',
            'cost_percent_total' => 'decimal(10,2)',
            'tax_id' => 'smallint(1)',
        ];
    }


    /**
     * Creates a new VtbApi instance with the given payment method configurations.
     *
     * @return VtbApi The new VtbApi instance.
     */
    private function getVtbApi()
    {
        return new VtbApi(
            $this->payment_method->vtbpay_client_id,
            $this->payment_method->vtbpay_client_secret,
            filter_var($this->payment_method->vtbpay_test_mode, FILTER_VALIDATE_BOOLEAN),
            $this->payment_method->vtbpay_merchant_authorization ?: ''
        );
    }


    /**
     * Confirms the order in the VTB Pay payment system.
     * Creates an instance of the VTB API, sends a request to get the order link, and then handles the response.
     *
     * @param array $cart Корзина с деталями заказа.
     * @param array $order Массив с данными заказа.
     *
     * @return mixed Returns NULL on failure, otherwise it doesn't return anything (the user gets redirected to the payment URL).
     */
    public function plgVmConfirmedOrder($cart, $order)
    {
        $this->order = $order;
        try {
            $this->setPaymentMethod($this->order['details']['BT']->virtuemart_paymentmethod_id)
                 ->setVtbpayLogger();
            $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

            if (!$this->selectedThisElement($this->payment_method->payment_element)) {
                return NULL;
            }

            // Once the payment has been processed (but possibly before the final status is known),
            // store the initial transaction details in the database
            $this->storePaymentData();

            return JFactory::getApplication()->redirect(
                $this->getPayUrl()
            );

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $msg = $e->getMessage() . ' Order ID: ' . $this->order['details']['BT']->virtuemart_order_id;
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s', $msg
            ), $context);

            JFactory::getApplication()->enqueueMessage(JText::_('PLG_VMPAYMENT_VTBPAY_ERROR') . ': ' . $msg, 'error');
        }

        return FALSE;
    }


    /**
     * Получение URL для оплаты заказа.
     * @return string URL для оплаты заказа.
     */
    private function getPayUrl()
    {
        $order_id = $this->order['details']['BT']->virtuemart_order_id;

        $return_url = JURI::root() . 'index.php?option=com_virtuemart&view=pluginresponse&task=pluginresponsereceived&mode=return' .
            '&on=' . $this->order['details']['BT']->order_number .
            '&pm=' . $this->order['details']['BT']->virtuemart_paymentmethod_id;

        $email = $this->getCustomerEmail();

        $two_stage = (bool) ($this->payment_method->vtbpay_two_stage ?: false);
        $enable_fiscal = (bool) ($this->payment_method->vtbpay_enable_fiscal ?: false);
        $ofd_fiscal = $this->payment_method->vtbpay_ofd_fiscal ?: '';

        $items = [];
        if (
            !$two_stage &&
            $enable_fiscal &&
            $ofd_fiscal === '1ofd'
        ) $items = $this->getItems();

        $order_total = self::parseAmount($this->order['details']['BT']->order_total);

        $response = $this->getVtbApi()->getOrderLink(
            $order_id,
            $email,
            time(),
            $order_total,
            $return_url,
            $two_stage,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает массив товаров заказа.
     * @return array Массив товаров.
     */
    private function getItems()
    {
        $items = [];
        foreach ($this->order['items'] as $key => $item) {
            $product_price = self::parseAmount($item->product_final_price);
            $items[] = [
                'positionId' => ($key + 1),
                'name' => $item->product_name,
                'code' => $item->product_sku ?: (string) $item->id,
                'price' => $product_price,
                'measure' => (int) $this->payment_method->vtbpay_measure_fiscal ?: 0,
                'quantity' => (int) $item->product_quantity,
                'taxParams' => [
                    'taxType' => $this->payment_method->vtbpay_tax_fiscal ?: 'none'
                ],
                'paymentType' => $this->payment_method->vtbpay_payment_fiscal ?: 'full_prepayment',
                'paymentSubject' => (int) $this->payment_method->vtbpay_subject_fiscal ?: 1,
                'amount' => floatval($product_price * $item->product_quantity)
            ];
        }

        $order_shipment = $this->order['details']['ST']->order_shipment ?? 0;
        // Стоимость доставки
        if ($order_shipment > 0) {
            $delivery_price = $order_shipment + $this->order['details']['ST']->order_shipment_tax;
            $delivery_price = self::parseAmount($delivery_price);

            $items[] = [
                'positionId' => ($key + 2),
                'name' => 'Доставка',
                'price' => $delivery_price,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->payment_method->vtbpay_tax_fiscal ?: 'none'
                ],
                'paymentType' => $this->payment_method->vtbpay_delivery_fiscal ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $delivery_price
            ];
        }

        return $items;
    }


    /**
     * Store payment data in the database
     *
     * @return void
     */
    protected function storePaymentData()
    {
        $order_bt = $this->order['details']['BT'];
        // Store the data using the storePSPluginInternalData method from the parent class
        $this->storePSPluginInternalData([
            'virtuemart_order_id' => $order_bt->virtuemart_order_id,
            'order_number' => $order_bt->order_number,
            'virtuemart_paymentmethod_id' => $this->payment_method->virtuemart_paymentmethod_id,
            'payment_name' => $this->renderPluginName($this->payment_method),
            'payment_order_total' => $order_bt->order_total,
            'payment_currency' => $this->getPaymentCurrency($this->payment_method),
            'cost_per_transaction' => $this->payment_method->cost_per_transaction ?? 0,
            'cost_percent_total' => $this->payment_method->cost_percent_total ?? 0,
            'tax_id' => $this->payment_method->tax_id ?? 0
        ]);
    }


    /**
     * Handles the response from VTB Pay after payment is made.
     * Verifies the order, the payment method, and the order number, then checks the payment status.
     *
     * @param string $html HTML для вывода.
     *
     * @return bool|null Returns true if payment is successful and response is handled, otherwise logs an error and returns false.
     */
    public function plgVmOnPaymentResponseReceived(&$html)
    {
        try {
            $input = JFactory::getApplication()->input;
            $virtuemart_paymentmethod_id = $input->getInt('pm');
            $mode = $input->getString('mode');

            if (!$this->selectedThisByMethodId($virtuemart_paymentmethod_id)) {
                return NULL; // Another method was selected, do nothing
            }

            $this->setPaymentMethod($virtuemart_paymentmethod_id)
                 ->setVtbpayLogger();

            // Logging request data
            if (in_array($mode, ['return', 'webhook'])) {
                $this->logger->setOption('additionalCommonText', $mode . '-' . rand(1111, 9999));
                $request_data = ($mode === 'return') ? $_REQUEST : json_decode(file_get_contents('php://input'), true);
                $this->logger->debug(
                    __FUNCTION__ . ' > ' . $mode . ': ',
                    ['request_data' => $request_data]
                );

                $order_id = $input->getString('orderId') ?:
                            $request_data['object']['orderId'] ??
                            $request_data['external_id'] ?? '';
            }

            if (!$order_id) {
                throw new \Exception(JText::_('PLG_VMPAYMENT_VTBPAY_NO_ORDER_ID_FOUND'));
            }

            if ($mode === 'return') {
                $order_number = $input->getString('on');
                if (!$order_number) {
                    throw new \Exception(JText::_('PLG_VMPAYMENT_VTBPAY_NO_ORDER_NUMBER_FOUND'));
                }
            }

            if (!class_exists('VirtueMartModelOrders')) {
                require(JPATH_VM_ADMINISTRATOR . DS . 'models' . DS . 'orders.php');
            }

            $order_model = VmModel::getModel('orders');
            $this->order = $order_model->getOrder($order_id);

            if (!$this->order) {
                throw new \Exception(JText::_('PLG_VMPAYMENT_VTBPAY_NO_ORDER_FOUND') . ' Order ID: ' . $order_id);
            }

            $this->event_dispatcher->dispatch('beforeChangePaymentStatus', [
                'mode' => $mode,
                'request_data' => $request_data
            ]);

            $this->changePaymentStatus($order_model);

            if ($mode === 'return') {
                // Remove the order details from the session
                JFactory::getSession()->clear('order_number');
                JFactory::getSession()->clear('pm');

                // Clear the cart
                $this->clearCart();

                return true;
            }
            else die('ok');

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s', $e->getMessage()
            ), $context);

            JFactory::getApplication()->enqueueMessage(JText::_('PLG_VMPAYMENT_VTBPAY_ERROR') . ': ' . $e->getMessage(), 'error');
        }

        return false;
    }


    /**
     * Очистить текущую корзину
     *
     * @return void
     */
    private function clearCart()
    {
        if (!class_exists('VirtueMartCart')) {
            require(JPATH_VM_SITE . DS . 'helpers' . DS . 'cart.php');
        }
        $cart = VirtueMartCart::getCart();
        $cart->emptyCart();
    }


    /**
     * Получение статуса платежа из внешнего источника.
     *
     * @return array Возвращает статус платежа.
     */
    private function getPaymentStatusData(): array
    {
        $response = $this->getVtbApi()->getOrderInfo(
            $this->order['details']['BT']->virtuemart_order_id
        );

        return $response;
    }


    /**
     * Initiates order as paid and updates its status.
     *
     * This function sets the order status to the specified transaction end status,
     * notifies the customer, adds a comment to the order indicating that it has been paid
     * with the "Payment system VTB," clears order-related session data, and empties the cart.
     *
     * @param object $order_model Объект заказа.
     * @return void
     */
    private function changePaymentStatus($order_model)
    {
        $payment_status_data = $this->getPaymentStatusData();
        $payment_status = $payment_status_data['object']['status']['value'] ?? '';

        $available_statuses = [
            'PAID' => [
                'order_status' => $this->payment_method->transaction_end_status,
                'comments' => JText::_('PLG_VMPAYMENT_VTBPAY_PAYMENT_SUCCESS'),
                'message' => JText::_('PLG_VMPAYMENT_VTBPAY_PAYMENT_SUCCESS'),
                'fnc' => function() {
                    $this->event_dispatcher->dispatch('paymentPaid');
                }
            ],
            'PENDING' => [
                'order_status' => 'V',
                'comments' => JText::_('PLG_VMPAYMENT_VTBPAY_PAYMENT_RECEIVED_BUT_NOT_CONFIRMED'),
                'message' => JText::_('PLG_VMPAYMENT_VTBPAY_PAYMENT_SUCCESS')
            ],
            'REFUNDED' => [
                'order_status' => 'R',
                'comments' => JText::_('PLG_VMPAYMENT_VTBPAY_PAYMENT_REFUNDED'),
                'message' => JText::_('PLG_VMPAYMENT_VTBPAY_PAYMENT_REFUNDED'),
                'fnc' => function() {
                    $this->event_dispatcher->dispatch('paymentRefunded');
                }
            ]
        ];

        $order_status_data = $available_statuses[$payment_status] ?? [
            'order_status' => 'X',
            'comments' => JText::_('PLG_VMPAYMENT_VTBPAY_PAYMENT_FAILURE'),
            'message' => JText::_('PLG_VMPAYMENT_VTBPAY_PAYMENT_FAILURE')
        ];

        $this->order['virtuemart_order_id'] = $this->order['details']['BT']->virtuemart_order_id;
        $this->order['customer_notified'] = 1;

        if (
            $payment_status !== 'REFUNDED' &&
            isset($payment_status_data['object']['transactions']['refunds'])
        ) {
            // Указываем статус возврата
            $order_status_data = $available_statuses['REFUNDED'];
        }

        $this->order['order_status'] = $order_status_data['order_status'];
        $this->order['comments'] = $order_status_data['comments'];
        JFactory::getApplication()->enqueueMessage($order_status_data['message']);

        if (
            isset($this->order['details']['BT']->order_status) &&
            $this->order['details']['BT']->order_status !== $this->order['order_status']
        ) {
            $order_model->updateStatusForOneOrder(
                $this->order['details']['BT']->virtuemart_order_id,
                $this->order,
                true
            );

            if (isset($order_status_data['fnc'])) $order_status_data['fnc']();
        }
    }


    /**
     * Получает email клиента иначе выкидывает ошибку
     */
    private function getCustomerEmail()
    {
        $email = $this->order['details']['BT']->email ?: $this->payment_method->vtbpay_email_fiscal;
        if (!self::validateEmail($email)) {
            throw new VtbPayException(JText::_('PLG_VMPAYMENT_VTBPAY_EMAIL_NOT_VALID'), [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email): bool
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Преобразует строку в число с плавающей запятой.
     *
     * @param mixed $amount Сумма для преобразования.
     * @return float Преобразованное значение суммы.
     */
    private static function parseAmount($amount): float
    {
        if (is_string($amount)) {
            $amount = str_replace([' ', ','], ['', '.'], $amount);
        }

        return floatval(number_format($amount, 2, '.', ''));
    }


    /**
     * Инциализирует свойство payment_method объектом платёжного метода
     *
     * @param int $payment_method_id ID метода оплаты
     *
     * @return self
     */
    private function setPaymentMethod($payment_method_id)
    {
        if (!($this->payment_method = $this->getVmPluginMethod($payment_method_id))) {
            throw new \Exception(JText::_('PLG_VMPAYMENT_VTBPAY_NO_METHOD_FOUND'));
        }

        return $this;
    }


    /**
     * Displays the payment information in the front-end when showing an order.
     *
     * @param int $virtuemart_order_id ID заказа
     * @param int $virtuemart_paymentmethod_id ID метода оплаты
     * @param string &$payment_name Название метода оплаты
     * @return void
     */
    public function plgVmOnShowOrderFEPayment($virtuemart_order_id, $virtuemart_paymentmethod_id, &$payment_name)
    {
        $this->onShowOrderFE($virtuemart_order_id, $virtuemart_paymentmethod_id, $payment_name);
    }


    /**
     * Checks the input data of the payment method within the checkout process.
     *
     * @param object &$cart Объект корзины
     * @return bool Always true since no data input is required for this payment method
     */
    public function plgVmOnCheckoutCheckDataPayment(&$cart)
    {
        // No data input required for this payment method
        return true;
    }


    /**
     * Adds the input data of the payment method to the cart, after the payment method is selected.
     *
     * @param object &$cart Объект корзины
     * @param string &$msg Сообщение показанное при выборе оплаты
     * @return bool Always true since no data input is required for this payment method
     */
    public function plgVmOnSelectCheckPayment(&$cart, &$msg)
    {
        // No data input required for this payment method
        return true;
    }


    /**
     * Displays the plugin methods in the cart, typically for editing shipment/payment details.
     *
     * @param VirtueMartCart $cart Объект корзины
     * @param int $selected ID выбранное метода оплаты
     * @param string &$html_in Создаваемая HTML-строка
     * @return bool|null True on success, false on failure, null when this plugin was not selected
     */
    public function plgVmDisplayListFEPayment(VirtueMartCart $cart, $selected, &$html_in)
    {
        return $this->displayListFE($cart, $selected, $html_in);
    }


    /**
     * Calculates the price (value, tax_id) of the selected payment method. This method is called by the calculator.
     *
     * @param VirtueMartCart $cart Объект текущей корзины
     * @param array &$cart_prices Цены в корзине
     * @param string &$cart_prices_name Название рассчитываемых цен корзины
     * @return bool|null|false True if successful, false if the shipping rate is not valid anymore, null if the method was not selected
     */
    public function plgVmonSelectedCalculatePricePayment(VirtueMartCart $cart, array &$cart_prices, &$cart_prices_name)
    {
        return $this->onSelectedCalculatePrice($cart, $cart_prices, $cart_prices_name);
    }


    /**
     * Checks how many payment plugins are available. If only one, the user will not have the choice. Enters edit_xxx page.
     *
     * @param VirtueMartCart $cart Объект корзины
     * @param array $cart_prices Массив цен в корзине
     * @param int &$payment_counter Счетчик кол-ва платежных методов в корзине
     * @return int|null 0 if more than one plugin was found, virtuemart_xxx_id if only one plugin is found, null if no plugin was found
     */
    function plgVmOnCheckAutomaticSelectedPayment(VirtueMartCart $cart, array $cart_prices, &$payment_counter)
    {
        return $this->onCheckAutomaticSelected($cart, $cart_prices, $payment_counter);
    }


    /**
     * This method reimplements vmPaymentPlugin::checkPaymentConditions() to check if the cart's total sales price
     * is within the acceptable range for the active payment method.
     *
     * @param VirtueMartCart $cart Объект корзины
     * @param object $active_method Объект активного метода оплаты, содержащий его параметры, такие как min_amount и max_amount.
     * @param array $cart_prices Ассоциативный массив, содержащий все сведения о ценах, связанные с корзиной, включая «salesPrice».
     * @return bool Returns true if the total sales price is within the acceptable range for the active payment method, false otherwise
     */
    function checkConditions($cart, $active_method, $cart_prices)
    {
        return true;
    }


    /**
     * Display stored payment data for an order in the backend.
     *
     * @param int $virtuemart_order_id ID заказа
     * @param int $virtuemart_payment_id ID платежной системы
     * @return string|null HTML-код, содержащий данные оплаты заказа, если они доступны, в противном случае NULL.
     */
    function plgVmOnShowOrderBEPayment($virtuemart_order_id, $virtuemart_payment_id)
    {

        if (!$this->selectedThisByMethodId($virtuemart_payment_id)) {
            return NULL; // Another method was selected, do nothing
        }

        if (!($payment_table = $this->getDataByOrderId($virtuemart_order_id))) {
            return NULL;
        }
        vmLanguage::loadJLang('com_virtuemart');

        if ($payment_table->payment_currency == 0) $payment_table->payment_currency = '₽';

        $html = '<table class="adminlist table">' . "\n";
        $html .= $this->getHtmlHeaderBE();
        $html .= $this->getHtmlRowBE('COM_VIRTUEMART_PAYMENT_NAME', $payment_table->payment_name);
        $html .= $this->getHtmlRowBE('PLG_VMPAYMENT_VTBPAY_TOTAL', $payment_table->payment_order_total . ' ' . $payment_table->payment_currency);
        $html .= '</table>' . "\n";
        return $html;
    }


    /**
     * Handles the printing of an Order's payment details.
     *
     * @param int $order_number Номер заказа
     * @param int $method_id ID используемой платежной системы
     * @return mixed The output of the onShowOrderPrint method
     */
    public function plgVmOnShowOrderPrintPayment($order_number, $method_id)
    {
        return $this->onShowOrderPrint($order_number, $method_id);
    }


    /**
     * Stores the payment info into the order.
     *
     * @param int $jplugin_id ID Joomla модуля
     * @return mixed The output of the onStoreInstallPluginTable method
     */
    public function plgVmOnStoreInstallPaymentPluginTable($jplugin_id)
    {
        return $this->onStoreInstallPluginTable($jplugin_id);
    }


    /**
     * Sets the parameters on the payment plugin table.
     *
     * @param string $name Название модуля
     * @param int $id ID модуля
     * @param object &$table таблица с параметрами модуля
     * @return mixed The result of the setOnTablePluginParams function
     */
    function plgVmSetOnTablePluginParamsPayment($name, $id, &$table)
    {
        return $this->setOnTablePluginParams($name, $id, $table);
    }


    /**
     * Declare the payment plugin parameters for VirtueMart 3.
     *
     * @param array &$data Массив параметров модуля
     * @return mixed The result of the declarePluginParams function
     */
    function plgVmDeclarePluginParamsPaymentVM3(&$data)
    {
        return $this->declarePluginParams('payment', $data);
    }


    /**
     * Declare the payment plugin parameters.
     *
     * @param string $name Название модуля
     * @param int $id ID модуля
     * @param array &$data Массив параметров модуля
     * @return mixed The result of the declarePluginParams function
     */
    function plgVmDeclarePluginParamsPayment($name, $id, &$data)
    {
  		  return $this->declarePluginParams('payment', $name, $id, $data);
  	}


    /**
     * Устанавливает объект логирования для модуля.
     */
    private function setVtbpayLogger()
    {
        $logging = (bool) $this->payment_method->vtbpay_logging;
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $this->logger = VtbPayLogger::getInstance()
                        ->setOption('showCurrentDate', false)
                        ->setOption('showLogLevel', false)
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitive_data_keys)
                        ->setCustomRecording(function($message)
                        {
                            // Логирование ошибки
                            JLog::add($message, JLog::ERROR, 'plg_vmpayment_vtbpay');

                        }, VtbPayLogger::LOG_LEVEL_ERROR)
                        ->setCustomRecording(function($message) use ($logging)
                        {
                            // Логирование сообщения дебага
                            if ($logging) JLog::add($message, JLog::DEBUG, 'plg_vmpayment_vtbpay');

                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
