<?php

namespace Vtbpay\Traits;

use \Vtbpay\Classes\Api\AtolApi,
    \Vtbpay\Classes\Exception\VtbPayException;

trait Atol
{
    /**
     * Обработчик события beforeChangePaymentStatus
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onBeforeChangePaymentStatusAtol(array $vars)
    {
        // Если есть uuid, значит к вебхуку обращается АТОЛ
        $uuid = $vars['request_data']['uuid'] ?? '';
        if ($vars['mode'] === 'webhook' && !empty($uuid)) {
            $this->webhookHandlerAtol($uuid);
            die('ok');
        }
    }


    /**
     * Обработчик события paymentPaid
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentPaidAtol(array $vars)
    {
        if (
            $this->payment_method->vtbpay_enable_fiscal &&
            $this->payment_method->vtbpay_ofd_fiscal === 'atol'
        ) $this->sendReceiptAtol();
    }


    /**
     * Обработчик события paymentRefunded
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentRefundedAtol(array $vars)
    {
        if (
            $this->payment_method->vtbpay_enable_fiscal &&
            $this->payment_method->vtbpay_ofd_fiscal === 'atol'
        ) $this->sendReceiptAtol(true);
    }


    /**
     * Создаем новый экземпляр класса AtolApi с заданными конфигурациями платёжной системы.
     *
     * @return AtolApi
     */
    private function getAtolApi(): AtolApi
    {
        return new AtolApi(
            $this->payment_method->vtbpay_login_atol_fiscal,
            $this->payment_method->vtbpay_pass_atol_fiscal,
            $this->payment_method->vtbpay_inn_atol_fiscal,
            $this->payment_method->vtbpay_tax_system_atol_fiscal,
            $this->payment_method->vtbpay_kkt_atol_fiscal,
            $this->payment_method->vtbpay_email_fiscal,
            (bool) $this->payment_method->vtbpay_test_mode_atol_fiscal
        );
    }


    /**
     * Проверяет статус формирования чека в АТОЛ и оповещает при ошибке
     *
     * @param string $uuid Полученные данные
     *
     * @return void
     */
    private function webhookHandlerAtol(string $uuid): void
    {
        $response = $this->getAtolApi()->getReceiptInfo($uuid);

        if ($response['status'] === 'fail'){
            $this->sendReceiptErrorEmail(sprintf(
                \JText::_('PLG_VMPAYMENT_VTBPAY_ERROR_ATOL_CHECK'),
                $this->order['details']['BT']->virtuemart_order_id
            ));
        }
        elseif ($response['status'] === 'wait') {
            $this->sendReceiptErrorEmail(sprintf(
                \JText::_('PLG_VMPAYMENT_VTBPAY_WAIT_ATOL_CHECK'),
                $this->order['details']['BT']->virtuemart_order_id
            ));
        }
    }


    /**
     * Отправляет в АТОЛ чек прихода или возврата на печать и возвращает результат.
     *
     * @param bool $is_return Данный чек "Возврат прихода" или "Приход"
     *
     * @return array
     */
    private function sendReceiptAtol(bool $is_return = false): array
    {
        $order_id = $this->order['details']['BT']->virtuemart_order_id;

        try {
            $email = $this->getCustomerEmail();

            $items = $this->preparingItemsForAtol(
                $this->getItems()
            );

            $total = self::parseAmount($this->order['details']['BT']->order_total);

            $callback_url = \JURI::root() . 'index.php?option=com_virtuemart&view=pluginresponse&task=pluginresponsereceived&mode=webhook' .
                '&pm=' . $this->order['details']['BT']->virtuemart_paymentmethod_id;

            $response = $this->getAtolApi()->sendReceipt(
                $order_id,
                $items,
                $total,
                $email,
                $callback_url,
                $is_return
            );

            return $response;

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s', $e->getMessage()
            ), $context);

            $this->sendReceiptErrorEmail(sprintf(
                \JText::_('PLG_VMPAYMENT_VTBPAY_ERROR_ATOL_CHECK'),
                $order_id
            ));

            return [];
        }
    }


    /**
     * Подготавливает элементы для АТОЛ.
     *
     * @param array $items Элементы для подготовки.
     *
     * @return array Массив подготовленных элементов.
     */
    private function preparingItemsForAtol(array $items): array
    {
        return array_map(function ($item) {
            return [
                'name' => $item['name'], // Лейбл
                'price' => $item['price'], // Цена товара за 1шт
                'sum' => $item['amount'], // Итоговая цена (цена * кол-во)
                'quantity' => $item['quantity'], // Кол-во позиций товара (сколько шт)
                'measure' => $item['measure'], // Единица измерения
                'vat' => [
                    'type' => $item['taxParams']['taxType'] // Налоговая ставка (НДС)
                ],
                'payment_method' => $item['paymentType'], // Способ расчета
                'payment_object' => $item['paymentSubject'], // Предмет расчета
            ];
        }, $items);
    }


    /**
     * Отправляет письмо об ошибке при выдаче чека в VirtueMart.
     *
     * Использует настройки почты из конфигурации Joomla.
     *
     * @param string $message Текст сообщения об ошибке (HTML).
     * @return void
     */
    private function sendReceiptErrorEmail($message): void
    {
        // Получаем конфигурацию Joomla
        $config = \JFactory::getConfig();
        // Название магазина и email администратора из настроек Joomla
        $store_name = $config->get('sitename', self::CMS_NAME);
        $admin_email = $config->get('mailfrom', "noreply@{$_SERVER['HTTP_HOST']}");

        // Получаем email для отправки из настроек VirtueMart (например, поле в конфигурации)
        $recipient_email = $this->payment_method->vtbpay_email_fiscal ?: $admin_email;

        // Формируем тему письма
        $subject = \JText::_('PLG_VMPAYMENT_VTBPAY_RECEIPT_ISSUANCE_ERROR') . ': ' . $store_name;

        // Создаём объект почты
        $mailer = \JFactory::getMailer();

        // Устанавливаем параметры письма
        $mailer->setSender([$admin_email, $store_name]); // Email и имя отправителя
        $mailer->addRecipient($recipient_email);         // Email получателя
        $mailer->setSubject($subject);                   // Тема письма
        $mailer->isHtml(true);                           // Устанавливаем HTML-формат
        $mailer->setBody($message);                      // Текст сообщения
        $mailer->Send();
    }
}
