<?php

require_once dirname(__DIR__, 6) . '/VtbPay/vendor/autoload.php';

use \UmiCms\Service,
    \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;

/**
 * A class intended for integration with the VTB payment system.
 * This handler allows you to set up and process payments via VTB in your online store.
 */
class VtbPayPayment extends payment
{
    const CMS_NAME = 'UMI.CMS';
    const PLUGIN_VERSION = '1.4.5';
    /** @const string[] CURRENCIES поддерживаемые валюты */
    const CURRENCIES = ['RUB', 'RUR'];

    private $logger;


    /**
     * Конструктор
     * @param iUmiObject $object объект-источник данных для способа оплаты
     * @throws Exception
     */
    public function __construct(iUmiObject $object)
    {
        call_user_func_array([parent::class, '__construct'], func_get_args());

        $envPath = dirname(__DIR__, 6) . '/VtbPay/config/.env';
        if (file_exists($envPath)) (new Dotenv())->load($envPath);

        $this->setVtbpayLogger();
    }


    /**
     * Checks whether the currency of the order is supported by the payment system
     *
     * @return bool Returns true if the order currency is supported, and false otherwise
     * @throws coreException Throws coreException when it encounters a system-level problem
     * @throws privateException Throws privateException if there's an issue with the private data
     */
    private function isSupportedCurrency() : bool
    {
        $currency = $this->getCurrencyCode();
        return in_array(strtoupper($currency), self::CURRENCIES);
    }


    /**
     * Checks if the order is suitable for processing by the payment system
     *
     * @param order $order The order to check
     * @return bool Returns true if the order is suitable, and false otherwise
     */
    public function isSuitable(order $order) : bool
    {
        return (
            $this->isSupportedCurrency() &&
            parent::isSuitable($order) &&
            ($order->getActualPrice() >= 0)
        );
    }


    /**
     * Retrieves the customer email from a customer object
     *
     * @return string The email of the customer
     * @throws publicException Throws publicException if the email is invalid
     */
    protected function getCustomerEmail()
    {
        $customer = $this->getCustomer();
        $email = $customer->getEmail() ?: $this->object->vtbpay_email_fiscal;
        if (
            !self::validateEmail($email) ||
            !umiMail::checkEmail($email)
        ) {
            throw new VtbPayException(getLabel('error-payment-wrong-customer-email'), [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($localPart, $domainPart) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $localLength = strlen($localPart);
        if ($localLength < 1 || $localLength > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domainLength = strlen($domainPart);
        if ($domainLength < 1 || $domainLength > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowedTlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowedTlds) && !empty($allowedTlds[0])) {
            // Извлечение TLD из доменной части
            $domainParts = explode('.', $domainPart);
            $tld = strtolower(end($domainParts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowedTlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Retrieves the customer object based on the customer ID in the order
     *
     * @return customer Returns the customer object
     * @throws publicException Throws publicException if the customer ID is invalid
     */
    protected function getCustomer()
    {
        $customerId = $this->order->getCustomerId();
        $customerSource = selector::get('object')->id($customerId);

        if (!$customerSource instanceof iUmiObject) {
            throw new publicException(getLabel('error-payment-wrong-customer-id'));
        }

        return new customer($customerSource);
    }


    /**
     * Processes the order by creating a payment link and redirects to it
     *
     * @param string|null $template The template to use for processing
     * @throws Exception|publicException|privateException Throws exceptions when encountering errors during processing
     * @return null
     */
    public function process($template = null)
    {
        $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            $this->order->order();
            $this->order->setPaymentStatus('initialized', true);

            $module = cmsController::getInstance()->getModule('emarket');
            if ($module) {
                $module->redirect($this->getPayUrl());
            }

        } catch (\Exception | publicException | privateException | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Log the caught exception for debugging purposes.
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s;',
                $e->getMessage(),
                $this->order->getId() ?: ''
            ), $context);

            throw $e;
        }
    }


    /**
     * Получает URL-адрес платежа для перенаправления.
     *
     * @return string URL-адрес платежа.
     */
    private function getPayUrl(): string
    {
        $orderId = $this->order->getId();
        // Order cost
        $amount = $this->formatPrice($this->order->getActualPrice());

        // Customer email
        $customerEmail = $this->getCustomerEmail();

        // Return Url
        $returnUrl = 'http://' . cmsController::getInstance()->getCurrentDomain()->getHost() .
                     '/emarket/gateway/' . $orderId . '/?request_type=return';

        // Since in this CMS the order identifier does not change when the order composition changes,
        // therefore, to create a new identifier, we associate the identifier with the price.
        $orderIdAndAmount = $orderId . '-' . $amount;

        $twoStage = (bool) ($this->object->vtbpay_two_stage ?? false);

        $enableFiscal = (bool) ($this->object->vtbpay_enable_fiscal ?? false);
        $items = [];
        if ($enableFiscal && !$twoStage) $items = $this->getItems();

        $response = $this->getVtbApi()->getOrderLink(
            $orderIdAndAmount,
            $customerEmail,
            time(),
            $amount,
            $returnUrl,
            $twoStage,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems(): array
    {
        $items = [];
        foreach ($this->order->getItems() as $key => $item) {
            $items[] = [
                'positionId' => ($key + 1),
                'name' => $item->getName(),
                'code' => (string) $item->getId(),
                'price' => floatval($item->getActualPrice()),
                'measure' => (int) $this->getValueSelectTypeObject('vtbpay-measure-fiscal') ?: 0,
                'quantity' => (int) $item->getAmount(),
                'taxParams' => [
                    'taxType' => $this->getValueSelectTypeObject('vtbpay-tax-type-fiscal') ?: 'none'
                ],
                'paymentType' => $this->getValueSelectTypeObject('vtbpay-payment-type-fiscal') ?: 'full_prepayment',
                'paymentSubject' => (int) $this->getValueSelectTypeObject('vtbpay-payment-subject-fiscal') ?: 1,
                'amount' => floatval($item->getActualPrice() * $item->getAmount())
            ];
        }

        $deliveryPrice = floatval($this->order->getValue('delivery_price'));
        if ($deliveryPrice > 0) {
            $delivery = selector::get('object')->id(
                $this->order->getValue('delivery_id')
            );

            if (!$delivery instanceof iUmiObject) {
                throw new expectObjectException(getLabel('error-unexpected-exception'));
            }
            $items[] = [
                'positionId' => ($key + 2),
                'name' => $delivery->getName() ?: 'Доставка',
                'price' => $deliveryPrice,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->getValueSelectTypeObject('vtbpay-tax-type-fiscal') ?: 'none'
                ],
                'paymentType' => $this->getValueSelectTypeObject('vtbpay-payment-type-delivery-fiscal') ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $deliveryPrice
            ];
        }

        return $items;
    }


    /**
     * Возвращает идентификатор заказа из запроса платежной системы.
     *
     * @return int
     */
    public static function getOrderId()
    {
        $mode = getRequest('request_type');
        if (!in_array($mode, ['return', 'webhook'])) throw new publicException('Undefined request type.');

        // Получение данных запроса в зависимости от типа
        $requestData = ($mode === 'return') ? $_REQUEST : json_decode(Service::Request()->getRawBody(), true);

        // Извлечение orderId
        $orderIdAndAmount = $requestData['orderId'] ?? $requestData['object']['orderId'] ?? '';
        if (empty($orderIdAndAmount)) {
            throw new publicException('Order ID not found.');
        }

        return (int) explode('-', $orderIdAndAmount)[0];
    }


    /**
     * Checks the status of the payment and updates the payment status accordingly
     *
     * @throws Exception|publicException|privateException Throws exceptions when encountering errors during polling
     */
    public function poll()
    {
        $mode = getRequest('request_type');
        $this->logger->setOption('additionalCommonText', $mode . '-' . rand(1111, 9999));

        $orderId = $this->order->getId();
        // Order cost
        $amount = $this->formatPrice($this->order->getActualPrice());
        // Так как в данной CMS идентификатор заказа не меняется при изменении состава заказа,
        // поэтому для создания нового идентификатора мы связываем идентификатор с ценой.
        $orderIdAndAmount = $orderId . '-' . $amount;

        try {
            // Получение статуса оплаты
            $paymentStatus = $this->getPaymentStatus($orderIdAndAmount);

            // Проверка и смена статуса заказа
            $this->changePaymentStatus($orderIdAndAmount, $paymentStatus);

            $buffer = Service::Response()
                      ->getCurrentBuffer();
            $buffer->clear();

            if ($mode === 'return') {
                if (in_array($paymentStatus, ['PAID', 'PENDING'])) {
                    $host = $this->getSuccessUrl();
                }
                else {
                    $host = $this->getFailUrl();
                }
                $buffer->status('301 Moved Permanently');
                $buffer->redirect($host);
            }
            else {
                $buffer->contentType('application/json');
          			$buffer->push(json_encode(['status' => 'OK']));
            }

      			$buffer->end();

        } catch (\Exception | publicException | privateException | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Log the caught exception for debugging purposes.
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order ID and amount: %s;',
                $e->getMessage(),
                $orderIdAndAmount ?: ''
            ), $context);

            throw $e;
        }
    }


    /**
     * Получение статуса заказа WooCommerce из VTB API.
     *
     * @param string $orderIdAndAmount Идентификатор заказа + цена
     * @return string Значение статуса заказа, если оно существует, или пустая строка в противном случае.
     */
    private function getPaymentStatus(string $orderIdAndAmount): string
    {
        $response = $this->getVtbApi()->getOrderInfo($orderIdAndAmount);

        $this->order->setPaymentDocumentNumber(
            $response['object']['orderCode'] ?? ''
        );

        return $response['object']['status']['value'] ?? '';
    }


    /**
     * Смена статуса заказа в зависимости от статуса оплаты.
     *
     * @param string $paymentStatus Статус оплаты
     * @return void
     * @throws LocalizedException
     */
    private function changePaymentStatus(string $orderIdAndAmount, string $paymentStatus)
    {
        $availableStatuses = [
           'PAID' => 'accepted',
           'PENDING' => 'waiting_confirmation',
           'REFUNDED' => 'refund'
        ];

        $newPaymentStatus = $availableStatuses[$paymentStatus] ?? 'declined';

        $this->logger->debug(
            __FUNCTION__ . ' - change_status:', [
            'newPaymentStatus' => $newPaymentStatus
        ]);

        // не сравниваем старый статус с новым, так это уже есть в setPaymentStatus
        $this->order->setPaymentStatus($newPaymentStatus);
        // Иначе корзина не очищается при холдировании
        if ($newPaymentStatus === 'waiting_confirmation') {
            $this->order->setValue('order_date', time());
            $this->order->setOrderStatus('waiting');
        }
        $this->order->commit();
    }


    /**
     * Получает значение объекта типа выпадающий список
     *
     * @param string $typeName Название типа
     *
     * @return string Значение
     */
    private function getValueSelectTypeObject($typeName): string
    {
        $propertyName = str_replace('-', '_', $typeName);
        $object = umiObjectsCollection::getInstance()->getObject($this->object->$propertyName ?? NULL);
        if (empty($object)) return '';
        return str_replace($typeName . '-', '', $object->getGUID());
    }


    /**
     * Получает объект для работы с API ВТБ.
     *
     * @return VtbApi Объект для работы с API ВТБ.
     */
    public function getVtbApi(): VtbApi
    {
        return new VtbApi(
            $this->object->vtbpay_client_id,
            $this->object->vtbpay_client_secret,
            (bool) $this->object->vtbpay_test_mode,
            $this->object->vtbpay_merchant_authorization ?: ''
        );
    }


    /**
     * Устанавливает объект логирования для модуля.
     */
    public function setVtbpayLogger(): void
    {
        $logging = (bool) $this->object->vtbpay_logging;
        $sensitiveDataKeys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $sysLogPath = mainConfiguration::getInstance()->includeParam('sys-log-path');
        $logPath = sprintf('%s/payment/%s/', rtrim($sysLogPath, '/'), str_replace('Payment', '', get_class($this)));
        $fileName = 'vtbpay-' . date('d-m-Y');

        $logger = new \umiLogger($logPath);
        $logger->setFileName($fileName);
        $logger->separateByIp(false);

        $this->logger = VtbPayLogger::getInstance()
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitiveDataKeys)
                        ->setLogFilePath($logPath . $fileName . '.log')
                        ->setCustomRecording(function($message) use ($logger)
                        {
                            // Логирование ошибки
                            $logger->push($message . PHP_EOL, false);
                            $logger->save();

                        }, VtbPayLogger::LOG_LEVEL_ERROR)
                        ->setCustomRecording(function($message) use ($logger, $logging)
                        {
                            // Логирование сообщения дебага
                            if ($logging) {
                                $logger->push($message . PHP_EOL, false);
                                $logger->save();
                            }

                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
