<?php

// Checking PHP Version
if (version_compare(PHP_VERSION, '7.4.0', '<')) {
    die('Payment module "Payment system VTB" requires PHP version 7.4.0 or higher.');
}

/**
 * Class name of the payment system.
 *
 * @var string
 */
$className = 'VtbPay';

/**
 * Name of the payment system.
 *
 * @var string
 */
$paymentName = 'Payment system VTB';

// include the standalone file for accessing e-commerce system functions
require_once dirname(__DIR__) . '/standalone.php';

/**
 * Access the object types collection instance to get the internal
 * type ID for the e-commerce payment type.
 */
$objectTypesCollection = umiObjectTypesCollection::getInstance();
$internalTypeId = $objectTypesCollection->getTypeIdByGUID('emarket-paymenttype');

/**
 * Create a new selector to search for objects within the objects
 * collection that match the provided parameters.
 */
$sel = new selector('objects');
$sel->types('object-type')->id($internalTypeId);
$sel->where('class_name')->equals($className);
$sel->limit(0, 1);

/**
 * Check if a payment method with the given class name already exists.
 * If not, proceed with creating a new payment method.
 */
if ($sel->length() == 0) {
    // access the objects collection instance
    $objectsCollection = umiObjectsCollection::getInstance();

    // get the parent type ID for the payment system
    $parentTypeId = $objectTypesCollection->getTypeIdByGUID('emarket-payment');

    // add a new type for the VTB payment system and get the type ID
    $typeId = $objectTypesCollection->addType($parentTypeId, $paymentName);

    // access the object type
    $objectType = $objectTypesCollection->getType($typeId);

    // добавить новую группу полей для настроек
    $objectType->addFieldsGroup('settings', 'Параметры', true, true);
    $group = $objectType->getFieldsGroupByName('settings');

    $objectType->addFieldsGroup('fiscal', 'Настройки фискализации', true, true);
    $groupFiscal = $objectType->getFieldsGroupByName('fiscal');

    // access the field types collection instance
    $fieldTypesCollection = umiFieldTypesCollection::getInstance();

    // get the IDs for the boolean and string field types
    $typeBoolean = $fieldTypesCollection->getFieldTypeByDataType('boolean')->getId();
    $typeString = $fieldTypesCollection->getFieldTypeByDataType('string')->getId();
    $typeSelect = $fieldTypesCollection->getFieldTypeByDataType('relation')->getId();

    /**
     * Define the fields to be added to the settings group of the object type.
     * Each field includes a name, type, and optional requirement flag.
     */
    $fields = [
        'vtbpay_client_id' => [
            'name' => 'Client ID',
            'type' => $typeString,
            'required' => true,
        ],
        'vtbpay_client_secret' => [
            'name' => 'Client Secret',
            'type' => $typeString,
            'required' => true,
        ],
        'vtbpay_merchant_authorization' => [
            'name' => 'Merchant-Authorization',
            'type' => $typeString
        ],
        'vtbpay_test_mode' => [
            'name' => 'Включить режим тестирования',
            'type' => $typeBoolean
        ],
        'vtbpay_logging' => [
            'name' => 'Включить логирование',
            'type' => $typeBoolean
        ],
        'vtbpay_two_stage' => [
            'name' => 'Включить двустадийный платеж',
            'type' => $typeBoolean
        ],
        'vtbpay_enable_fiscal' => [
            'name' => 'Включить фискализацию',
            'type' => $typeBoolean,
            'settings_group' => $groupFiscal,
        ],
        'vtbpay_email_fiscal' => [
            'name' => 'Email',
            'type' => $typeString,
            'settings_group' => $groupFiscal
        ],
        'vtbpay_payment_type_delivery_fiscal' => [
            'name' => 'Способ расчета для доставки',
            'type' => $typeSelect,
            'settings_group' => $groupFiscal,
            'guide_id' => addOptionsForRelationType('vtbpay-payment-type-delivery-fiscal', [
                'full_prepayment' => 'Предоплата 100 процентов',
                'prepayment'  => 'Предоплата',
                'advance' => 'Аванс',
                'full_payment' => 'Полный расчет',
                'partial_payment' => 'Частичный расчет и кредит',
                'credit' => 'Передача в кредит',
                'credit_payment' => 'Оплата кредита'
            ], $objectTypesCollection, $objectsCollection),
        ],
        'vtbpay_measure_fiscal' => [
            'name' => 'Единицы измерения',
            'type' => $typeSelect,
            'settings_group' => $groupFiscal,
            'guide_id' => addOptionsForRelationType('vtbpay-measure-fiscal', [
                '0' => 'Применяется для предметов расчета, которые могут быть реализованы поштучно или единицами',
                '10' => 'Грамм',
                '11' => 'Килограмм',
                '12' => 'Тонна',
                '20' => 'Сантиметр',
                '21' => 'Дециметр',
                '22' => 'Метр',
                '30' => 'Квадратный сантиметр',
                '31' => 'Квадратный дециметр',
                '32' => 'Квадратный метр',
                '40' => 'Миллилитр',
                '41' => 'Литр',
                '42' => 'Кубический метр',
                '50' => 'Киловатт час',
                '51' => 'Гигакалория',
                '70' => 'Сутки (день)',
                '71' => 'Час',
                '72' => 'Минута',
                '73' => 'Секунда',
                '80' => 'Килобайт',
                '81' => 'Мегабайт',
                '82' => 'Гигабайт',
                '83' => 'Терабайт',
                '255' => 'Применяется при использовании иных единиц измерения'
            ], $objectTypesCollection, $objectsCollection),
        ],
        'vtbpay_tax_type_fiscal' => [
            'name' => 'Ставка налогообложения',
            'type' => $typeSelect,
            'settings_group' => $groupFiscal,
            'guide_id' => addOptionsForRelationType('vtbpay-tax-type-fiscal', [
                'none' => 'Без НДС',
                'vat0' => 'НДС по ставке 0 процентов',
                'vat10' => 'НДС чека по ставке 10 процентов',
                'vat110' => 'НДС чека по расчетной ставке 10/110',
                'vat20' => 'НДС чека по ставке 20 процентов',
                'vat120' => 'НДС чека по расчетной ставке 20/120',
                'vat5' => 'НДС чека по ставке 5 процентов',
                'vat7' => 'НДС чека по ставке 7 процентов',
                'vat105' => 'НДС чека по расчетной ставке 5/105',
                'vat107' => 'НДС чека по расчетной ставке 7/107'
            ], $objectTypesCollection, $objectsCollection),
        ],
        'vtbpay_payment_type_fiscal' => [
            'name' => 'Способ расчета',
            'type' => $typeSelect,
            'settings_group' => $groupFiscal,
            'guide_id' => addOptionsForRelationType('vtbpay-payment-type-fiscal', [
                'full_prepayment' => 'Предоплата 100 процентов',
                'prepayment' => 'Предоплата',
                'advance' => 'Аванс',
                'full_payment' => 'Полный расчет',
                'partial_payment' => 'Частичный расчет и кредит',
                'credit' => 'Передача в кредит',
                'credit_payment' => 'Оплата кредита'
            ], $objectTypesCollection, $objectsCollection),
        ],
        'vtbpay_payment_subject_fiscal' => [
            'name' => 'Предмет расчета',
            'type' => $typeSelect,
            'settings_group' => $groupFiscal,
            'guide_id' => addOptionsForRelationType('vtbpay-payment-subject-fiscal', [
                '1' => 'Товар',
                '2' => 'Подакцизный товар',
                '3' => 'Работа',
                '4' => 'Услуга',
                '5' => 'Ставка азартной игры',
                '6' => 'Выигрыш азартной игры',
                '7' => 'Ставка лотереи',
                '8' => 'Выигрыш лотереи',
                '9' => 'Предоставление прав',
                '10' => 'Платеж',
                '11' => 'Агентское вознаграждение',
                '12' => 'Выплата',
                '13' => 'Иной предмет расчёта',
                '14' => 'Имущественное право',
                '15' => 'Внереализационный доход',
                '16' => 'Страховые взносы',
                '17' => 'Торговый сбор',
                '18' => 'Курортный сбор',
                '19' => 'Залог',
                '20' => 'Расход',
                '21' => 'Взносы на ОПС ИП',
                '22' => 'Взносы ОПС',
                '23' => 'Взносы на ОМС ИП',
                '24' => 'Взносы на ОМС',
                '25' => 'Взносы на ОСС',
                '26' => 'Платеж казино',
                '27' => 'Выдача денежных средств банковским платежным агентом',
                '30' => 'Подакцизный товар, подлежащий маркировке средством идентификации, не имеющий кода маркировки',
                '31' => 'Подакцизный товар, подлежащий маркировке средством идентификации, имеющий код маркировки',
                '32' => 'Товар, подлежащий маркировке средством идентификации, не имеющий кода маркировки, за исключением подакцизного товара',
                '33' => 'Товар, подлежащий маркировке средством идентификации, имеющий код маркировки, за исключением подакцизного товара'
            ], $objectTypesCollection, $objectsCollection),
        ]
    ];

    // access the fields collection instance
    $fieldsCollection = umiFieldsCollection::getInstance();

    // iterate over each field to be added
    foreach ($fields as $code => $arField) {
        $fieldId = $fieldsCollection->addField($code, $arField['name'], $arField['type']);
        $field = $fieldsCollection->getField($fieldId);

        if (isset($arField['guide_id'])) {
            $field->setGuideId($arField['guide_id']);
        }

        $isRequired = isset($arField['required']) ? $arField['required'] : false;
        $field->setIsRequired($isRequired);
        $field->setIsInSearch(false);
        $field->setIsInFilter(false);
        $field->commit();

        if (isset($arField['settings_group'])) {
            $arField['settings_group']->attachField($fieldId);
        }
        else {
            $group->attachField($fieldId);
        }
    }

    // create a new internal object for the payment system
    $internalObjectId = $objectsCollection->addObject($paymentName, $internalTypeId);
    $internalObject = $objectsCollection->getObject($internalObjectId);

    // set the class name and payment type ID for the internal object
    $internalObject->setValue('class_name', $className); // имя класса для реализации
    $internalObject->setValue('payment_type_id', $typeId);
    $internalObject->setValue('payment_type_guid', 'emarket-payment-' . $typeId);
    $internalObject->commit();

    // set the GUID for the object type
    $objectType = $objectTypesCollection->getType($typeId);
    $objectType->setGUID($internalObject->getValue('payment_type_guid'));
    $objectType->commit();

    // print a success message
    echo 'Модуль «Платежная система ВТБ» успешно установлен. id #' . $typeId . PHP_EOL;
} else {
    // print a failure message if a payment method with the given class name already exists
    echo 'Способ оплаты ' . $className . ' уже существует.' . PHP_EOL;
}


/**
 * Создаёт тип выпадающего списка и добавляет для него опции
 *
 * @param string $typeName Название типа
 * @param array $options Массив опций выпадающего списка
 * @param object $objectTypesCollection
 * @param object $objectsCollection
 *
 * @return int
 */
function addOptionsForRelationType($typeName, $options, $objectTypesCollection, $objectsCollection)
{
    $addedType = $objectTypesCollection->addType(1, $typeName);
    foreach ($options as $key => $value) {
        $addedObjectId = $objectsCollection->addObject($value, $addedType);
        $addedObject = $objectsCollection->getObject($addedObjectId);
        $addedObject->setGUID($typeName . '-' . $key);
        $addedObject->commit();
    }
    return $addedType;
}
