<?php

namespace Vtbpay\Classes\Common;

/**
 * VTB Payments outer installer
 *
 * This class is responsible for installing the VTB Payments payment method into the Simpla CMS.
 *
 * @copyright  VTB
 * @author:    VTB
 * @version:   1.0.1
 */
class Outer extends \Simpla
{
    private $db;
    private $money;

    /**
     * Constructor for the Outer class.
     *
     * Initializes the database and money objects and checks if VTB Payments is already installed.
     * Displays appropriate messages based on the installation status.
     */
    public function __construct()
    {
        $this->db = parent::__get('db');
        $this->money = parent::__get('money');

        if ($this->isInstalled()) {
            $this->displayErrorMessage("Платежная система ВТБ уже установлена.");
        } else {
            if ($this->install()) {
                $this->displaySuccessMessage();
            } else {
                $this->displayErrorMessage("Не удалось установить платежную систему ВТБ.");
            }
        }
    }


    /**
     * Install the VTB Payment system.
     *
     * This method checks if VTB Payments is already installed and installs it if not.
     *
     * @return bool True if installation is successful, false otherwise.
     */
    private function install()
    {
        if ($this->isInstalled()) {
            return false;
        }

        $currencyId = $this->getCurrencyId('RUB');

        $settings = serialize([
            'vtbpay_client_id' => '',
            'vtbpay_client_secret' => '',
            'vtbpay_merchant_authorization' => '',
            'vtbpay_test_mode' => 1,
            'vtbpay_logging' => 1,
            'vtbpay_email_fiscal' => '',
            'vtbpay_payment_type_delivery_fiscal' => 'full_prepayment',
            'vtbpay_measure_fiscal' => '0',
            'vtbpay_tax_type_fiscal' => 'none',
            'vtbpay_payment_type_fiscal' => 'full_prepayment',
            'vtbpay_payment_subject_fiscal' => '1',
        ]);

        $paymentData = [
            'module' => 'VtbPayments',
            'name' => 'Платежная система ВТБ',
            'description' => 'Оплачивайте товары картой любого банка без комиссии',
            'currency_id' => $currencyId,
            'settings' => $settings,
            'enabled' => 1,
            'position' => $this->getNextPosition(),
        ];

        return $this->insertPaymentMethod($paymentData);
    }


    /**
     * Check if VTB Payments is already installed.
     *
     * @return bool True if VTB Payments is installed, false otherwise.
     */
    private function isInstalled()
    {
        $query = $this->db->placehold("SELECT `id` FROM `__payment_methods` WHERE `module` = ?", 'VtbPayments');
        $result = $this->db->query($query);

        return $this->db->num_rows($result) > 0;
    }


    /**
     * Get the currency ID for a given currency code.
     *
     * @param string $code The currency code.
     * @return int The currency ID, or 0 if not found.
     */
    private function getCurrencyId($code)
    {
        foreach ($this->money->get_currencies() as $currency) {
            if ($currency->code === $code) {
                return $currency->id;
            }
        }

        return 0;
    }


    /**
     * Get the next position for the payment method.
     *
     * @return int The next available position.
     */
    private function getNextPosition()
    {
        $query = "SELECT MAX(`position`) as max_position FROM `__payment_methods`";
        $result = $this->db->query($query);
        $position = $this->db->result($result, 0, 'max_position');

        return $position !== null ? ($position + 1) : 1;
    }


    /**
     * Insert a payment method into the database.
     *
     * @param array $data The payment method data to insert.
     * @return bool True if the insertion is successful, false otherwise.
     */
    private function insertPaymentMethod($data)
    {
        $fields = array_keys($data);
        $placeholders = array_fill(0, count($fields), '?');
        $args = array_values($data);

        $query = "INSERT INTO __payment_methods (`" . implode("`, `", $fields) . "`) VALUES (" . implode(", ", $placeholders) . ")";
        $result = $this->db->query($query, $args);

        return $result;
    }


    /**
     * Display an error message.
     *
     * @param string $message The error message to display.
     */
    private function displayErrorMessage($message)
    {
        echo '<div>
                  <h2>' . htmlspecialchars($message) . '</h2>
                  <p><a href="' . $this->config->root_url . '">Перейти на главную страницу</a></p>
              </div>';
    }


    /**
     * Display a success message and redirect to the payment methods page.
     */
    private function displaySuccessMessage()
    {
        echo '<div>
                  <h2>"Платежная система ВТБ" установлена</h2>
                  <p>Вы будете перенаправлены на страницу способов оплаты через 3 секунды.</p>
              </div>
              <script>
                  setTimeout(function() {
                      location = "/simpla/index.php?module=PaymentMethodsAdmin";
                  }, 3000);
              </script>';
    }
}
