<?php
require __DIR__ . '/vendor/autoload.php';

use \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Vtbpay\Traits\VtbPay,
    \Symfony\Component\Dotenv\Dotenv;

/**
 * VtbpayPayment Class
 *
 * This class provides methods to handle payment operations with VTB API.
 */
class vtbpayPayment extends waPayment implements waIPayment
{
    use VtbPay;

    const CMS_NAME = 'Webasyst Shop-Script';
    const PLUGIN_VERSION = '1.3.5';

    private string $order_id;
    private string $request_type;
    private VtbPayLogger $logger;


    /**
     * Инициализация настроек
     */
    protected function init()
    {
        parent::init();

        $env_path = __DIR__ . '/config/.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);

        return $this;
    }


    /**
     * Payment Method
     *
     * This method manages the payment process. It creates a new order, validates it,
     * retrieves the payment URL from VTB API, and then returns a view with the form URL.
     *
     * @param array $payment_form_data The form data for the payment
     * @param array $order_data The order data
     * @param bool $auto_submit Whether to auto-submit the form
     *
     * @return string The rendered view with the payment form
     *
     * @throws waException If the order currency is not supported
     */
    public function payment($payment_form_data, $order_data, $auto_submit = false)
    {
        $this->setVtbpayLogger();
        $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));
        $order = waOrder::factory($order_data);

        try {
            if (!in_array($order->currency_id, $this->allowedCurrency())) {
                throw new waException($this->_w('Currency not supported.'));
            }

            $view = wa()->getView();
            $view->assign(
                'form_url',
                $this->getPayUrl($order)
            );
            $view->assign('auto_submit', $auto_submit);
            $view->assign('form_desc', $this->_w('Redirecting to the bank website for payment...'));
            $view->assign('form_btn', $this->_w('Go to the payment form, to the bank page'));
            return $view->fetch($this->path . '/templates/payment.html');

        } catch (\Exception | waException | waPaymentException | VtbPayException $e) {
            // Handle exception and log error
            $this->executeErrorScenario(
                $e,
                $order->id,
                __FUNCTION__
            );
        }
    }


    /**
     * Callback Init Method
     *
     * This method initializes the callback by extracting parameters from the request,
     * and then calls the parent callback method for further processing.
     *
     * @param array $request The request parameters
     *
     * @return array Parent's callback method result
     *
     * @throws waPaymentException If the invoice number is invalid
     */
    protected function callbackInit($request)
    {
        try {
            if (isset($request['params']) && !empty($request['params'])) {

                $params = json_decode(base64_decode($request['params']), true);

                $this->app_id = $params['app_id'] ?? '';
                $this->merchant_id = $params['merchant_id'] ?? '';
                $this->request_type = $params['type'] ?? '';

                $this->setOrderId($request);
            }
            else {
                throw new waPaymentException($this->_w('Invalid invoice number.'));
            }
            // calling parent's method to continue plugin initialization
            return parent::callbackInit($request);

        } catch (\Exception | waException | waPaymentException | VtbPayException $e) {
            // Handle exception and log error
            $this->executeErrorScenario(
                $e,
                null,
                __FUNCTION__, [
                'Request data' => $request
            ]);
        }
    }


    /**
     * Устанавливаем идентификатор заказа для разных типов запроса (return/webhook)
     *
     * @param array $request The request parameters
     *
     * @return void
     */
    private function setOrderId(array $request): void
    {
        if ($this->request_type === 'return') {
            $this->order_id = $request['orderId'] ?? '';
        }
        elseif ($this->request_type === 'webhook') {
            $php_input = json_decode(file_get_contents('php://input'), true) ?: null;
            $this->order_id = $php_input['object']['orderId'] ?? '';
        }
    }


    /**
     * Callback Handler Method
     *
     * This method manages the callback process. It retrieves the order data based on the order_id,
     * processes the transaction, and then redirects the customer based on the transaction status.
     *
     * @param array $request The request parameters
     *
     * @throws waPaymentException If the order ID or the order itself is not found
     */
    protected function callbackHandler($request)
    {
        $this->setVtbpayLogger();
        $this->logger->setOption('additionalCommonText', $this->request_type . '-' . rand(1111, 9999));

        try {
            if (!$this->order_id) {
                throw new waPaymentException($this->_w('Order ID not found.'));
            }

            $order = $this->getAdapter()->getOrderData($this->order_id, $this);

            if (!$order) {
                throw new waPaymentException($this->_w('Order not found.'));
            }

            $status_data = $this->getPaymentStatusData();

            $transaction_data = $this->changeStatus($status_data, $request, $order);

            if ($this->request_type === 'return') {
              // Redirect customer
              wa()->getResponse()->redirect(
                  $this->getFinalRedirectUrl($transaction_data)
              );
            }
            elseif ($this->request_type === 'webhook') {
                die('OK');
            }
        } catch (\Exception | waException | waPaymentException | VtbPayException $e) {
            // Handle exception and log error
            $this->executeErrorScenario(
                $e,
                $this->order_id,
                __FUNCTION__, [
                'Request data' => $request
            ]);
        }
    }


    /**
     * Получени информации о платеже и зменение статуса заказа
     *
     * @param array $status_data Массив данных статуса платежа.
     *
     * @return array
     */
    private function changeStatus(
        array $status_data,
        array $request,
        waOrder $order
    ): array {
        $available_statuses = [
            'PAID' => [
                'method' => self::CALLBACK_PAYMENT,
                'state' => self::STATE_CAPTURED,
                'type' => self::OPERATION_AUTH_CAPTURE
            ],
            'PENDING' => [
                'method' => self::CALLBACK_AUTH,
                'state' => self::STATE_AUTH,
                'type' => self::OPERATION_AUTH_ONLY
            ],
            'REFUNDED' => [
                'method' => self::CALLBACK_REFUND,
                'state' => self::STATE_REFUNDED,
                'type' => self::OPERATION_REFUND
            ]
        ];

        $payment_status = $status_data['object']['status']['value'] ?? '';
        $transaction_data = $this->formalizeData($status_data) + $available_statuses[$payment_status] ?? [
            'method' => self::CALLBACK_CANCEL,
            'state' => self::STATE_CANCELED,
            'type' => self::OPERATION_CANCEL
        ];
        $current_order_status = self::getTransaction($order->__get('params')['payment_transaction_id'] ?? []);

        if (
            $current_order_status['state'] !== $transaction_data['state'] &&
            $current_order_status['type'] !== $transaction_data['type']
        ) {
            $transaction_data = $this->saveTransaction(
                $transaction_data,
                $request
            );

            $this->execAppCallback(
                $transaction_data['method'],
                $transaction_data
            );
        }

        return $transaction_data;
    }


    /**
     * Получение ссылки для редиректа на страницу результата оплаты
     *
     * @param array $transaction_data Массив данных транзакции.
     *
     * @return string
     */
    private function getFinalRedirectUrl(array $transaction_data): string
    {
        switch ($transaction_data['state']) {
          case self::STATE_CAPTURED:
          case self::STATE_AUTH:
              $url = $this->getAdapter()->getBackUrl(waAppPayment::URL_SUCCESS, $transaction_data);
              break;
          default:
              $url = $this->getAdapter()->getBackUrl(waAppPayment::URL_FAIL, $transaction_data);
              break;
        }

        return $url;
    }


    /**
     * Сценарий, выполняемый после отлова исключения
     *
     * @param \Exception|VtbPayException $e   Объект исключения.
     * @param int $order_id                   Идентификатор заказа.
     * @param string $caller_func             Название функции, котоорая вызывает обработчик исключений.
     * @param bool $context                   Массив данных контекста
     *
     * @return void
     */
    private function executeErrorScenario(
        $e,
        $order_id = null,
        $caller_func = '',
        $context = []
    ): void {
        if (method_exists($e, 'getContext')) $context = $e->getContext();

        $this->logger->error(sprintf(
            $caller_func . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
            $e->getMessage(),
            $order_id ?: ''
        ), $context);

        throw $e;
    }


    /**
     * Formalize Data Method
     *
     * This method formats the raw transaction data into a more usable format.
     *
     * @param array $transaction_raw_data The raw transaction data
     *
     * @return array The formatted transaction data
     */
    protected function formalizeData($transaction_raw_data): array
    {
        $transaction_data = parent::formalizeData($transaction_raw_data);
        $transaction_data['order_id'] = ifset($transaction_raw_data['object']['orderId']);
        $transaction_data['amount'] = ifset($transaction_raw_data['object']['amount']['value']);
        $transaction_data['currency_id'] = ifset($transaction_raw_data['object']['amount']['code']);
        $transaction_data['native_id'] = $this->order_id;
        return $transaction_data;
    }


    /**
     * Allowed Currency Method
     *
     * This method returns the list of allowed currencies for the payment.
     *
     * @return array List of allowed currencies
     */
    public function allowedCurrency()
    {
        return [
            'RUB'
        ];
    }


    /**
     * Инициализация и настройка объекта класса VtbPayLogger.
     *
     * Эта функция инициализирует и настраивает логгер, используемый плагином VtbPay для ведения журнала.
     *
     * @return void
     */
    private function setVtbpayLogger(): void
    {
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];
        $obj_id = $this->id; // Идентификатор объекта
        $logging = $this->logging; // Настройка в админке (логироать или нет)

        $this->logger = VtbPayLogger::getInstance()
                        ->setOption('showCurrentDate', false)
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitive_data_keys)
                        ->setCustomRecording(function($message) use ($obj_id) {
                            self::log($obj_id, $message);
                        }, VtbPayLogger::LOG_LEVEL_ERROR)
                        ->setCustomRecording(function($message) use ($obj_id, $logging) {
                            if ($logging) self::log($obj_id, $message);
                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
