<?php

namespace Vtbpay\Traits;

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Exception\VtbPayException;

trait VtbPay
{
    /**
     * Get VTB API instance
     *
     * This method initializes and returns an instance of the VTB API using
     * the provided configuration values.
     *
     * @return VtbApi An instance of the VtbApi
     */
    private function getVtbApi()
    {
        return new VtbApi(
            $this->client_id,
            $this->client_secret,
            (bool) $this->test_mode,
            $this->merchant_authorization ?: ''
        );
    }


    /**
     * Получает URL-адрес платежа для перенаправления.
     *
     * @return string URL-адрес платежа.
     */
    private function getPayUrl($order): string
    {
        // Return url
        $params = base64_encode(json_encode([
            'type' => 'return',
            'app_id' => $this->app_id,
            'merchant_id' => $this->merchant_id
        ]));
        $return_url = $this->getRelayUrl() . '?params=' . $params;

        // Customer data
        $email = $this->getCustomerEmail($order);

        // total
        $total = $order->tax_included == false && $order->tax ? $order->tax + $order->total : $order->total;

        $items = [];
        if (!$this->two_stage && $this->enable_fiscal) $items = $this->getItems($order);

        $response = $this->getVtbApi()->getOrderLink(
            $order->id,
            $email,
            time(),
            $total,
            $return_url,
            $this->two_stage,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает массив товаров заказа.
     *
     * @param \waOrder $order Объект заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems(\waOrder $order): array
    {
        $items = [];

        foreach ($order->items as $key => $item) {
            $items[] = [
                'positionId' => ($key + 1),
                'name' => $item['name'],
                'description' => $item['description'],
                'code' => $item['sku'],
                'price' => $item['price'],
                'measure' => (int) $this->measure_fiscal ?: 0,
                'quantity' => $item['quantity'],
                'taxParams' => [
                    'taxType' => $this->tax_type_fiscal ?: 'none'
                ],
                'paymentType' => $this->payment_type_fiscal ?: 'full_prepayment',
                'paymentSubject' => (int) $this->payment_subject_fiscal ?: 1,
                'amount' => $item['total'],
                'discount' => $item['total_discount']
            ];
        }

        $shipping = $order->shipping ?: 0;
        if ($shipping > 0) {
            $items[] = [
                'positionId' => ($key + 2),
                'name' => $order->shipping_name,
                'price' => $shipping,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->tax_type_fiscal ?: 'none'
                ],
                'paymentType' => $this->payment_type_delivery_fiscal ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $shipping
            ];
        }

        return $items;
    }


    /**
     * Получение Email покупателя
     *
     * @param \waOrder $order Объект данных заказа
     *
     * @return string Email клиента
     *
     * @throws VtbPayException If the invoice number is invalid
     */
    private function getCustomerEmail(\waOrder $order): string
    {
        $email = $order->getContact()->get('email', 'default') ?: $this->email_fiscal;
        if (!self::validateEmail($email)) {
            throw new VtbPayException($this->_w('The email address provided has not been validated.'), [
                'email' => $email
            ]);
        }

        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Получение статуса заказа из VTB API.
     *
     * @return array Массив данных статуса платежа.
     */
    private function getPaymentStatusData(): array
    {
        return $this->getVtbApi()->getOrderInfo($this->order_id) ?: [];
    }

}
