<?php

namespace VtbPay\Src\Classes\Api;

use \VtbPay\Src\Classes\Common\Request,
    \VtbPay\Src\Classes\Exception\VtbPayException;

/**
 * Класс VtbApi предоставляет методы для взаимодействия с API VTB.
 */
class VtbApi extends Request
{
    private string $clientMerchant;
    private array $apiEndpoints;
    private array $accessData;


    /**
     * Конструктор класса VtbApi.
     *
     * @param string $clientId     Идентификатор клиента.
     * @param string $clientSecret Секрет клиента.
     * @param bool   $test         Режим тестирования (по умолчанию - false).
     * @param string $merchant     Идентификатор продавца (по умолчанию - пустая строка).
     *
     * @throws VtbPayException Если не заполнены обязательные поля $clientId или $clientSecret.
     */
    public function __construct(
        string $clientId,
        string $clientSecret,
        bool $test = false,
        string $merchant = ''
    ) {
        if (empty($clientId) || empty($clientSecret)) {
            throw new VtbPayException('Обязательные поля не заполнены.');
        }

        $this->clientMerchant = trim($merchant);

        $this->setApiEndpoints($test ? 'TEST' : 'PROD');

        $this->accessData = $this->getAccessData(
            trim($clientId),
            trim($clientSecret)
        );
    }


    /**
     * Устанавливает эндпоинты API VTB
     *
     * @param string $mode Режим тестирования
     *
     * @return void
     */
    private function setApiEndpoints(string $mode): void
    {
        // Список типов запроса
        $requestTypes = ['TOKEN', 'ORDER', 'ORDER_PREAUTH', 'REFUND'];
        foreach ($requestTypes as $type) {
            $endpoint = $_ENV[$mode . '_' . $type] ?? '';

            if (empty($endpoint)) throw new VtbPayException('Не указан обязательный эндпоинт API VTB.', [
                'mode' => $mode,
                'type' => $type
            ]);

            $this->apiEndpoints[$type] = $endpoint;
        }
    }



    /**
     * Получает URL для заданного типа запроса.
     *
     * @param string $type Тип запроса ('TOKEN', 'ORDER', 'REFUND' и т.д.).
     *
     * @return string URL для запроса.
     */
    private function getUrl(string $type): string
    {
        return $this->apiEndpoints[$type] ?? '';
    }


    /**
     * Делает запрос в VTB для полученя токена и возвращает его.
     *
     * @param string $clientId     Идентификатор клиента.
     * @param string $clientSecret Секрет клиента.
     *
     * @return array Данные авторизации
     */
    private function getAccessData(string $clientId, string $clientSecret): array
    {
        $response = $this->sendRequest(
            $this->getUrl('TOKEN'), [
                'client_id' => $clientId,
                'client_secret' => $clientSecret,
                'grant_type' => 'client_credentials'
            ],
            [
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Merchant-Authorization' => $this->clientMerchant
            ]
        );

        if (!isset($response['access_token']) || !isset($response['token_type'])) {
            throw new VtbPayException('Ошибка при получении токена доступа.', [
                'response' => $response
            ]);
        }

        return [
            'X-IBM-Client-Id' => explode('@', $clientId)[0],
            'Authorization' => $response['token_type'] . ' ' . $response['access_token']
        ];
    }


    /**
     * Получает сгенерированный данные о странице оплаты заказа.
     *
     * @param string $orderId             Идентификатор заказа.
     * @param string $email               Email покупателя.
     * @param int    $orderCreate         Дата создания заказа.
     * @param float  $amount              Сумма заказа.
     * @param string $returnUrl           URL для возврата после оплаты.
     * @param bool   $twoStage            Двухстадийный платеж.
     * @param array  $items               Массив товаров заказа.
     * @param string $pluginName          Название плагина.
     * @param string $pluginVersion       Версия плагина.
     * @param string $appendedOrderName   Добавочный текст к названию заказа.
     *
     * @return array Результат запроса.
     * @throws VtbPayException Если не удается получить ссылки для оплаты.
     */
    public function getOrderLink(
        string $orderId,
        string $email,
        int $orderCreate,
        float $amount,
        string $returnUrl,
        bool $twoStage = false,
        array $items = [],
        string $pluginName = '',
        string $pluginVersion = '',
        string $appendedOrderName = ''
    ): array {
        $timestamp = $orderCreate + 86400; // +1 day
        $expire = date('Y-m-d\TH:i:s.v\Z', $timestamp); // ISO 8601

        $body = [
            'orderId' => $orderId,
            'orderName' => 'Заказ #' . $orderId  . ($appendedOrderName ? '. ' . $appendedOrderName : ''),
            'customer' => ['email' => $email],
            'amount' => ['value' => $amount, 'code' => 'RUB'],
            'expire' => $expire,
            'returnUrl' => $returnUrl,
            'incomingRegisterOrderChannel' => 'CMS',
            'pluginName' => $pluginName,
            'pluginVersion' => $pluginVersion
        ];

        if (!empty($items)) {
            $body['bundle'] = [
                'fiscalInfo' => [
                    'clientEmail' => $email
                ],
                'items' => $items
            ];
        }

        $headers = [
            'Content-Type' => 'application/json',
            'Merchant-Authorization' => $this->clientMerchant
        ] + $this->accessData;

        $response = $this->sendRequest(
            $this->getUrl($twoStage ? 'ORDER_PREAUTH' : 'ORDER'),
            $body,
            $headers
        );

        if (
            !isset($response['object']['payUrl']) ||
            empty($response['object']['payUrl'])
        ) {
            throw new VtbPayException('Ошибка при получении ссылки для оплаты.', [
                'response' => $response
            ]);
        }

        return $response;
    }


    /**
     * Получает информацию о заказе.
     *
     * @param string $orderId Идентификатор заказа.
     *
     * @return array Результат запроса.
     * @throws VtbPayException Если не удается получить статус платежа.
     */
    public function getOrderInfo(string $orderId): array
    {
        $headers = [
            'Merchant-Authorization' => $this->clientMerchant
        ] + $this->accessData;

        $response = $this->sendRequest(
            $this->getUrl('ORDER') . '/' . $orderId,
            [],
            $headers,
            'GET'
        );

        if (
            !isset($response['object']['status']['value']) ||
            empty($response['object']['status']['value'])
        ) {
            throw new VtbPayException('Информация о статусе платежа отсутствует.', [
                'response' => $response
            ]);
        }

        return $response;
    }


    /**
     * Получает информацию о заказе и товарной корзине.
     *
     * @param string $orderCode Код заказа.
     *
     * @return array Результат запроса.
     * @throws VtbPayException Если не удается получить данные платежа.
     */
    public function getBundle(string $orderCode): array
    {
        $response = $this->sendRequest(
            str_replace('<order_code>', $orderCode, $this->getUrl('BUNDLE')),
            [],
            [],
            'GET'
        );

        if (
            !isset($response['order']['items']) ||
            empty($response['order']['items'])
        ) {
            throw new VtbPayException('Информация о товарной корзине платежа отсутствует.', [
                'response' => $response
            ]);
        }

        return $response;
    }


    /**
     * Устанавливает возврат средств.
     *
     * @param string $orderId Идентификатор заказа.
     * @param float  $amount              Сумма заказа.
     * @param string $email               Email покупателя.
     * @param array  $items               Массив товаров заказа.
     *
     * @return array Результат запроса.
     * @throws VtbPayException Если нет информации об платеже
     */
    public function setRefunds(
        string $orderId,
        float $amount = 0,
        string $email = '',
        array $items = []
    ): array
    {
        $response = $this->getOrderInfo($orderId);
        $transactionPayment = $response['object']['transactions']['payments'][0]['object'] ?? '';

        if (empty($transactionPayment)) {
            throw new VtbPayException('Информация о платеже отсутствует.', [
                'response' => $response
            ]);
        }

        if ($amount > $transactionPayment['amount']['value']) {
            throw new VtbPayException('Сумма возврата превышает сумму оплаты.', [
                'response' => $response
            ]);
        }

        $body = [
            'refundId' => $orderId . '-REFUND-' . time(),
            'paymentId' => $transactionPayment['paymentId'],
            'amount' => [
                'value' => $amount ?: $transactionPayment['amount']['value'],
                'code' => $transactionPayment['amount']['code']
            ]
        ];

        if (!empty($items) && !empty($email)) {
            $body['bundle'] = [
                'fiscalInfo' => [
                    'clientEmail' => $email
                ],
                'items' => $items
            ];
        }

        $headers = [
            'Content-Type' => 'application/json',
            'Merchant-Authorization' => $this->clientMerchant
        ] + $this->accessData;

        $response = $this->sendRequest(
            $this->getUrl('REFUND'),
            $body,
            $headers
        );

        if (isset($response['error']) && !empty($error = $response['error'])) {
            throw new VtbPayException(
                is_array($error) ? ($error['description'] ?? 'Ошибка возврата средств.') : $error, [
                'response' => $response
            ]);
        }

        return $response;
    }
}
