<?php
/**
 * @package       VtbPay Module for ReadyScript
 * @author        ВТБ (ПАО)
 */

namespace VtbPay\Model\PaymentType;

require_once dirname(__DIR__, 2) . '/vendor/autoload.php';

use \Shop\Model\PaymentType\AbstractType,
    \Shop\Model\Orm\Transaction,
    \RS\Orm\FormObject,
    \RS\Orm\PropertyIterator,
    \RS\Router\Manager,
    \RS\Http\Request,
    \Shop\Model\Cart,
    \Shop\Model\ChangeTransaction,
    \Catalog\Model\CurrencyApi,
    \VtbPay\Src\Classes\Api\VtbApi,
    \VtbPay\Src\Classes\Common\VtbPayLogger,
    \VtbPay\Src\Classes\Exception\VtbPayException,
    \VtbPay\Model\Log\VtbLog,
    \VtbPay\Config\File,
    \Symfony\Component\Dotenv\Dotenv;


/**
 * Способ оплаты - VTBPay
 */
class VtbPay extends AbstractType
{
    const CMS_NAME = 'ReadyScript';

    public VtbPayLogger $logger;
    public ?VtbApi $vtb_api = null;


    public function __construct()
  	{
        $env_path = dirname(__DIR__, 2) . '/config/.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);
  	}


    /**
     * Возвращает название расчетного модуля (типа доставки). Можно использовать HTML
     * @return string
     */
    public function getTitle()
    {
        return t('Платежная система ВТБ');
    }


    /**
     * Возвращает описание способа оплаты. Можно использовать HTML
     * @return string
     */
    public function getDescription()
    {
        return t('Оплата картой любого банка без комиссии.');
    }


    /**
     * Возвращает идентификатор способа оплаты. (только английский)
     * @return string
     */
    public function getShortName()
    {
        return 'vtbpay';
    }


    /**
     * Возвращает true, если этот способ оплаты поддерживает проведение платежа через интернет
     * @return bool
     */
    public function canOnlinePay()
    {
        return true;
    }


    /**
     * Проверка отправлены данные с помощью POST
     */
    public function isPostQuery()
    {
        return false;
    }


    /**
     * Создаёт объект для работы с API ВТБ.
     *
     * @return self Объект для работы с API ВТБ.
     */
    public function setVtbApi(): self
    {
        if (empty($this->vtb_api)) $this->vtb_api = new VtbApi(
            $this->getOption('client_id'),
            $this->getOption('client_secret'),
            (bool) $this->getOption('test_mode'),
            $this->getOption('client_merchant') ?: ''
        );

        return $this;
    }


    /**
     * Возвращает ORM объект для генерации формы или null
     *
     * @return \RS\Orm\FormObject | null
     */
    public function getFormObject()
    {
        $properties = new PropertyIterator(include dirname(__DIR__, 2) . '/config/payment_form_fields.php');

        $form_object = new FormObject($properties);
        $form_object->setParentObject($this);
        $form_object->setParentParamMethod('Form');
        return $form_object;
    }


    /**
     * Возвращает URL для перехода на сайт сервиса оплаты
     * @param Transaction $transaction - ORM объект транзакции
     * @return string
     */
    public function getPayUrl(Transaction $transaction)
    {
        try {
            $this->setVtbpayLogger()->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));
            $this->setVtbApi();

            $currency = CurrencyApi::getCurrecyCode();
            if ($currency !== 'RUB') {
                throw new \Exception('Платежный система ВТБ принимает платежи только в российских рублях.');
            }

            return $this->getVtbPayUrl($transaction);

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Transaction id: %s;',
                $e->getMessage(),
                $transaction->id ?: ''
            ), $context);

            return $this->getFailUrl($transaction->id);
        }
    }


    /**
     * Возвращает ссылку на страницу оплаты
     * @param Transaction $transaction
     * @return string
     */
    private function getVtbPayUrl(Transaction $transaction)
    {
        $email = $this->getUserEmail($transaction);
        $date = strtotime($transaction->dateof);
        $total = round($transaction->cost, 2);
        $return_url = $this->getReturnUrl($transaction->id);
        $plugin_version = File::getDefaultValues()['version'] ?? '';

        $two_stage = (bool) $this->getOption('two_stage');
        $enable_fiscal = (bool) $this->getOption('enable_fiscal');
        $items = [];
        if (!$two_stage && $enable_fiscal) $items = $this->getItems();

        $response = $this->vtb_api->getOrderLink(
            $transaction->id,
            $email,
            $date,
            $total,
            $return_url,
            $two_stage,
            $items,
            self::CMS_NAME,
            $plugin_version
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Возвращает email клиента
     * @param Transaction $transaction
     * @return string
     */
    private function getUserEmail(Transaction $transaction)
    {
        $user = $transaction->getUser(); //Получим текущего пользователя
        $email = $user['e_mail'] ?? $this->getOption('email_fiscal');
        if (!self::validateEmail($email)) {
            throw new VtbPayException('Указанный адрес электронной почты не прошёл валидацию.', [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Возвращает URL возврата после оплаты
     * @param int $transaction_id
     * @return string
     */
    private function getReturnUrl($transaction_id)
    {
        return Manager::obj()->getUrl('vtbpay-front-returns', ['transactionId' => $transaction_id], true);
    }


    /**
     * Возвращает ID заказа из REQUEST-параметров
     * @param \RS\Http\Request $request
     * @return bool
     */
    public function getTransactionIdFromRequest(Request $request)
    {
        return $request->request('transactionId', TYPE_STRING);
    }


    /**
     * Возвращает URL не проведённого платежа
     * @param int $transaction_id Номер транзакции
     */
    public function getFailUrl($transaction_id)
    {
        return $this->makeRightAbsoluteUrl(
            Manager::obj()->getUrl('shop-front-onlinepay', [
                'Act' => 'fail',
                'PaymentType' => $this->getShortName(),
                'transactionId' => $transaction_id
            ])
        );
    }


    /**
     * Возвращает URL успешной оплаты
     * @param int $transaction_id Номер транзакции
     */
    public function getSuccessUrl($transaction_id)
    {
        return $this->makeRightAbsoluteUrl(
            Manager::obj()->getUrl('shop-front-onlinepay', [
                'Act' => 'success',
                'PaymentType' => $this->getShortName(),
                'transactionId' => $transaction_id
            ])
        );
    }


    /**
     * Получение информации о платеже и замене статуса заказа
     * @throws VtbPayException
     * @throws Exception
     */
    public function changePaymentStatus($payment_status_data)
    {
        $payment_status = $payment_status_data['object']['status']['value'] ?? '';
        $available_statuses = [
            'PAID' => [
                'order_status' => Transaction::STATUS_SUCCESS,
                'reason_text' => t('Платеж успешно завершен.')
            ],
            'PENDING' => [
                'order_status' => Transaction::STATUS_HOLD,
                'reason_text' => t('Платёж получен, но не подтверждён.')
            ],
            'REFUNDED' => [
                'order_status' => Transaction::STATUS_FAIL,
                'reason_text' => t('Платеж возвращён.')
            ]
        ];

        $order_status_data = $available_statuses[$payment_status] ?? [
            'order_status' => Transaction::STATUS_FAIL,
            'reason_text' => t('Платеж не выполнен.')
        ];

        // Если есть возврат, но статус ещё не установлен
        if (
            $payment_status !== 'REFUNDED' &&
            isset($payment_status_data['object']['transactions']['refunds'])
        ) $order_status_data = [
            'order_status' => Transaction::STATUS_FAIL,
            'reason_text' => t('Платеж возвращён.')
        ];

        if ($this->transaction->status !== $order_status_data['order_status']) {
            $this->changeTransactionAndOrderStatus(
                $order_status_data['order_status'],
                $order_status_data['reason_text']
            );

            if (isset($order_status_data['fnc'])) $order_status_data['fnc']();
        }
    }


    /**
     * Изменить статус заказа
     *
     * @return self
     */
    public function changeTransactionAndOrderStatus($transaction_status, $reason_text)
    {
        $this->logger->debug(
            __FUNCTION__ . ' - INPUT: ', [
            'old_status' => $this->transaction->status,
            'new_status' => $transaction_status,
            'reason_text' => $reason_text
        ]);

        $change = new ChangeTransaction($this->transaction);
        $change->setNewStatus($transaction_status);
        $change->setChangelog($reason_text);
        $change->applyChanges();

        if ($transaction_status === Transaction::STATUS_FAIL) {
            $order = $this->transaction->getOrder();
            $order['status'] = $this->payment->holding_cancel_status;
            $order['is_payed'] = 0;
            $order->update();
        }
    }


    /**
     * Получение статуса заказа из VTB API.
     *
     * @return array
     */
    public function getPaymentStatusData(): array
    {
        $response = $this->vtb_api->getOrderInfo($this->transaction->id);

        return $response ?: [];
    }


    /**
     * Получает массив товаров заказа и доставку.
     *
     * @return array Массив товаров.
     */
    private function getItems(): array
    {
        // Настройки по умолчанию для фискальных параметров
        $fiscal_settings = [
            'measure_fiscal' => 0,
            'tax_type_fiscal' => 'none',
            'payment_type_fiscal' => 'full_prepayment',
            'payment_subject_fiscal' => 1
        ];
        $items = [];
        $key = 1;
        if ($cart = $this->transaction->getOrder()->getCart()) {
            // Добавление в массив товаров
            foreach ($cart->getProductItems() as $item) {
                $cartitem = $item['cartitem'];
                $product = $item['product'];

                $product_fiscal = [];
                // Получение фискальных параметров для каждого продукта
                foreach ($fiscal_settings as $name => $default) {
                    $product_fiscal[$name] = $product->{$name} ?: '-';
                    if ($product_fiscal[$name] === '-') $product_fiscal[$name] = $this->getOption($name) ?: $default;
                }

                $items[] = [
                    'positionId' => $key,
                    'name' => $cartitem->title,
                    'code' => (string) $cartitem->barcode ?: $cartitem->id,
                    'price' => floatval($cartitem->single_cost),
                    'measure' => (int) $product_fiscal['measure_fiscal'],
                    'quantity' => (int) $cartitem->amount,
                    'taxParams' => [
                        'taxType' => $product_fiscal['tax_type_fiscal']
                    ],
                    'paymentType' => $product_fiscal['payment_type_fiscal'],
                    'paymentSubject' => (int) $product_fiscal['payment_subject_fiscal'],
                    'amount' => floatval($cartitem->price)
                ];

                $key++;
            }

            foreach($cart->getCartItemsByType(Cart::TYPE_DELIVERY) as $shippment) {
                if ($shippment->price > 0) $items[] = [
                    'positionId' => $key,
                    'name' => $shippment->title,
                    'price' => floatval($shippment->price),
                    'measure' => 0,
                    'quantity' => 1,
                    'taxParams' => [
                        'taxType' => $this->getOption('tax_type_fiscal') ?: 'none'
                    ],
                    'paymentType' => $this->getOption('payment_type_delivery_fiscal') ?: 'full_prepayment',
                    'paymentSubject' => 4,
                    'amount' => floatval($shippment->price)
                ];
            }
        }

        return $items;
    }


    /**
     * Устанавливает объект логирования для модуля.
     */
    public function setVtbpayLogger(): VtbPayLogger
    {
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];
        $logging = (bool) $this->getOption('logging');
        $log = VtbLog::getInstance();

        $this->logger = VtbPayLogger::getInstance()
            ->setOption('showCurrentDate', false)
            ->setOption('showLogLevel', false)
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitive_data_keys)
            ->setCustomRecording(function ($message) use ($log) {
                // Логирование ошибки
                $log->write($message, $log::LEVEL_ERROR);

            }, VtbPayLogger::LOG_LEVEL_ERROR)
            ->setCustomRecording(function ($message) use ($log, $logging) {
                // Логирование сообщения дебага
                if ($logging) $log->write($message, $log::LEVEL_DEBUG);

            }, VtbPayLogger::LOG_LEVEL_DEBUG);

        return $this->logger;
    }
}
