<?php
/**
 * @package       VtbPay Module for ReadyScript
 * @author        ВТБ (ПАО)
 */

namespace VtbPay\Controller\Front;

use \Shop\Model\Orm\Transaction,
    \Shop\Model\PaymentType\ResultException,
    \RS\Controller\Front,
    \VtbPay\Model\PaymentType\VtbPay,
    \VtbPay\Src\Classes\Exception\VtbPayException;

class Returns extends Front
{
    private VtbPay $payment;
    private Transaction $transaction;


    /**
     * Подготовка данных и запуск проверки и смены статуса транзакции
     */
    public function actionIndex()
    {
        $transaction_id = $this->url->request('transactionId', TYPE_INTEGER);

        $this->transaction = new Transaction($transaction_id);
        if (empty($this->transaction->id)) {
            throw (new ResultException(t('Транзакция не найдена.')))
                ->setUpdateTransaction(false); // Не устанавливаем статус FAIL транзакции
        }

        $this->checkProcess();

        return $this->result->getOutput();
    }


    /**
     * Проверяет статус оплаты в ВТБ и меняет статус транзакции
     */
    private function checkProcess()
    {
        try {
            $this->setPayment();
            $this->payment->setVtbpayLogger()->setOption('additionalCommonText', 'return-' . rand(1111, 9999));
            $this->payment->setVtbApi();

            // Logging request data
            $this->payment->logger->debug(
                __CLASS__ . ' > ' . __FUNCTION__ . ' > return - INPUT: ', [
                'request_data' => $_REQUEST
            ]);

            $payment_status_data = $this->payment->getPaymentStatusData();
            $this->payment->changePaymentStatus($payment_status_data);

            if (in_array(
                $payment_status_data['object']['status']['value'] ?? '',
                ['PAID', 'PENDING']
            )) {
                $this->result->setSuccess(true)->setRedirect(
                    $this->payment->getSuccessUrl($this->transaction->id)
                );
            }
            else {
                $this->result->setSuccess(false)->setRedirect(
                    $this->payment->getFailUrl($this->transaction->id)
                );
            }

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->payment->logger->error(sprintf(
                __CLASS__ . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Transaction id: %s;',
                $e->getMessage(),
                $this->transaction->id ?: ''
            ), $context);

            // Set payment response for failure.
            $order_status = $context['order_status'] ?? Transaction::STATUS_FAIL;
            $reason_text = $context['reason_text'] ?? $e->getMessage();

            $this->payment->changeTransactionAndOrderStatus(
                $order_status,
                $reason_text
            );

            $this->view->assign([
                'heading_title' => t('При проверке платежа произошла ошибка.'),
                'content_body' => $e->getMessage()
            ]);
            $this->result->setTemplate('error.tpl');
        }
    }


    /**
     * Инициализация объекта оплаты
     *
     * @return self
     */
    private function setPayment(): self
    {
        $this->payment = $this->transaction->getPayment()->getTypeObject();
        return $this;
    }

}
