<?php
/**
 * Модуль для интеграции с платежной системой ВТБ.
 *
 * @author    VTB <acquiring_support@vtb.ru>
 */

if (!defined('_PS_VERSION_')) exit;

require __DIR__ . '/vendor/autoload.php';

use PrestaShop\PrestaShop\Core\Payment\PaymentOption,
    \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Common\EventDispatcher,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Vtbpay\Traits\Shipping,
    \Vtbpay\Form\Modifier\ProductFormModifier,
    \Symfony\Component\Dotenv\Dotenv;


class VtbPay extends PaymentModule
{
    use Shipping;

    const CMS_NAME = 'PrestaShop';

    /**
     * @var array Массив с конфигурацией модуля.
     */
    public array $configure;

    /**
     * @var VtbApi|null Объект для работы с API ВТБ.
     */
    public ?VtbApi $vtb_api = null;

    /**
     * @var VtbPayLogger Объект для логирования.
     */
    public VtbPayLogger $logger;

    /**
     * @var EventDispatcher Объект менеджера событий.
     */
    public EventDispatcher $event_dispatcher;


    /**
     * В конструкторе инициализируем настройки модуля.
     */
    public function __construct()
    {
        $this->name = 'vtbpay';
        $this->tab = 'payments_gateways';
        $this->version = '1.6.10';
        $this->author = 'VTB';
        $this->author_uri = 'https://acquiring.vtb.ru';
        $this->logo = __DIR__ . '/videws/img/logotype.png';
        $this->need_instance = 1;

        $this->controllers = [
            'payment',
            'return',
            'webhook'
        ];

        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Платежная система ВТБ');
        $this->description = $this->l('Расплачивайтесь любой картой банка без комиссии.');

        $this->confirmUninstall = $this->l('Вы уверены, что хотите удалить этот модуль?');

        $this->ps_versions_compliancy = [
            'min' => '1.7',
            'max' => _PS_VERSION_
        ];

       self::loadEnvFiles([
           __DIR__ . '/config/.env',        // обязательный
           __DIR__ . '/config/.env.custom' // опциональный
       ]);

        $this->event_dispatcher = new EventDispatcher($this); // Инициализация менеджера событий

        $this->configure = $this->getConfigFormValues();
    }


    /**
     * Загружает .env файлы.
     *
     * Первый файл обязателен, остальные — опциональные.
     *
     * @param array $files Пути к .env файлам, в порядке приоритета.
     * @return void
     * @throws \RuntimeException Если первый файл отсутствует.
     */
    private static function loadEnvFiles(array $files): void
    {
        $dotenv = new Dotenv();

        foreach ($files as $index => $file) {
            if (is_readable($file)) {
                $dotenv->load($file);
            } elseif ($index === 0) {
                throw new \RuntimeException("Обязательный .env файл не найден: {$file}");
            }
        }
    }


    /**
     * Получает объект для работы с API ВТБ.
     *
     * @return VtbApi Объект для работы с API ВТБ.
     */
    public function setVtbApi(): self
    {
        if (empty($this->vtb_api)) {
            $config = $this->configure;
            $this->vtb_api = new VtbApi(
                $config['VTBPAY_CLIENT_ID'],
                $config['VTBPAY_CLIENT_SECRET'],
                (bool) $config['VTBPAY_TEST_MODE'],
                $config['VTBPAY_CLIENT_MERCHANT'] ?: ''
            );
        }

        return $this;
    }


    /**
     * Действия выполняемые при установке.
     *
     * @return bool Результат установки модуля.
     */
    public function install()
    {
        // Checking PHP Version
        if (version_compare(PHP_VERSION, '7.4.0', '<')) {
            $this->_errors[] = $this->l('Для работы платежного модуля "Платежная система ВТБ" требуется PHP версии 7.4.0 или выше.');
            return false;
        }

        if (extension_loaded('curl') == false) {
            $this->_errors[] = $this->l('Для использования этого модуля необходимо включить на сервере расширение cURL.');
            return false;
        }

        Configuration::updateValue('VTBPAY_TEST_MODE', true);

        return parent::install() &&
               $this->registerHook('paymentOptions') &&
               $this->registerHook(['actionProductFormBuilderModifier']) &&
               $this->registerHook(['actionProductSave']);
    }


    /**
     * Действия при удалении модуля.
     *
     * @return bool Результат удаления модуля.
     */
    public function uninstall()
    {
        // Delete module config variable
        Configuration::deleteByName('VTBPAY_TEST_MODE');
        Configuration::deleteByName('VTBPAY_CLIENT_ID');
        Configuration::deleteByName('VTBPAY_CLIENT_SECRET');
        Configuration::deleteByName('VTBPAY_CLIENT_MERCHANT');
        Configuration::deleteByName('VTBPAY_LOGGING');
        Configuration::deleteByName('VTBPAY_TWO_STAGE');
        Configuration::deleteByName('VTBPAY_SUCCESSFUL_PAYMENT_PAGE');
        Configuration::deleteByName('VTBPAY_FAILED_PAYMENT_PAGE');
        Configuration::deleteByName('VTBPAY_ENABLE_FISCAL');
        Configuration::deleteByName('VTBPAY_EMAIL_FISCAL');
        Configuration::deleteByName('VTBPAY_PAYMENT_TYPE_DELIVERY_FISCAL');
        Configuration::deleteByName('VTBPAY_MEASURE_FISCAL');
        Configuration::deleteByName('VTBPAY_TAX_TYPE_FISCAL');
        Configuration::deleteByName('VTBPAY_PAYMENT_TYPE_FISCAL');
        Configuration::deleteByName('VTBPAY_PAYMENT_SUBJECT_FISCAL');

        return parent::uninstall();
    }


    /**
     * Отображает настройки модуля в административной части.
     *
     * @return string HTML-код настроек модуля.
     */
    public function getContent()
    {
        if (((bool)Tools::isSubmit('submitVtbpayModule')) == true) $this->postProcess();

        $this->context->smarty->assign('module_dir', $this->_path);

        return $this->renderForm();
    }


    /**
     * Генерирует форму настроек модуля.
     *
     * @return string HTML-код формы настроек.
     */
    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitVtbpayModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) .
                                '&configure=' . $this->name .
                                '&tab_module=' . $this->tab .
                                '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = [
            'fields_value' => $this->getConfigFormValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        ];

        return $helper->generateForm([$this->getConfigForm()]);
    }


    /**
     * Возвращает массив с полями настроек.
     *
     * @return array Массив с полями настроек.
     */
    protected function getConfigForm(): array
    {
        return include 'config/config_form.php';
    }


    /**
     * Возвращает массив значений конфигурации модуля.
     *
     * @return array Массив значений конфигурации.
     */
    protected function getConfigFormValues()
    {
        return [
            'VTBPAY_TEST_MODE' => Configuration::get('VTBPAY_TEST_MODE'),
            'VTBPAY_CLIENT_ID' => Configuration::get('VTBPAY_CLIENT_ID'),
            'VTBPAY_CLIENT_SECRET' => Configuration::get('VTBPAY_CLIENT_SECRET'),
            'VTBPAY_CLIENT_MERCHANT' => Configuration::get('VTBPAY_CLIENT_MERCHANT'),
            'VTBPAY_LOGGING' => Configuration::get('VTBPAY_LOGGING'),
            'VTBPAY_TWO_STAGE' => Configuration::get('VTBPAY_TWO_STAGE'),
            'VTBPAY_SUCCESSFUL_PAYMENT_PAGE' => Configuration::get('VTBPAY_SUCCESSFUL_PAYMENT_PAGE'),
            'VTBPAY_FAILED_PAYMENT_PAGE' => Configuration::get('VTBPAY_FAILED_PAYMENT_PAGE'),
            'VTBPAY_ENABLE_FISCAL' => Configuration::get('VTBPAY_ENABLE_FISCAL'),
            'VTBPAY_EMAIL_FISCAL' => Configuration::get('VTBPAY_EMAIL_FISCAL'),
            'VTBPAY_PAYMENT_TYPE_DELIVERY_FISCAL' => Configuration::get('VTBPAY_PAYMENT_TYPE_DELIVERY_FISCAL'),
            'VTBPAY_MEASURE_FISCAL' => Configuration::get('VTBPAY_MEASURE_FISCAL'),
            'VTBPAY_TAX_TYPE_FISCAL' => Configuration::get('VTBPAY_TAX_TYPE_FISCAL'),
            'VTBPAY_PAYMENT_TYPE_FISCAL' => Configuration::get('VTBPAY_PAYMENT_TYPE_FISCAL'),
            'VTBPAY_PAYMENT_SUBJECT_FISCAL' => Configuration::get('VTBPAY_PAYMENT_SUBJECT_FISCAL')
        ];
    }


    /**
     * Обновляет отправленные настройки модуля.
     */
    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();

        foreach (array_keys($form_values) as $key) {
            Configuration::updateValue($key, Tools::getValue($key));
        }
    }


    /**
     * Возвращает опции платежа для отображения на странице оплаты.
     *
     * @param array $params Параметры платежа.
     * @return array Массив опций платежа.
     */
    public function hookPaymentOptions($params)
    {
        if (!$this->active || !$this->checkCurrency($params['cart'])) {
            return;
        }

        $payment_option = new PaymentOption();
        $payment_option->setModuleName($this->name);
        $payment_option->setCallToActionText($this->l('Платежная система ВТБ'));
        $payment_option->setAction($this->context->link->getModuleLink($this->name, 'payment', [], true));
        $payment_option->setLogo(Media::getMediaPath(__DIR__ . '/views/img/label.png'));

        return [$payment_option];
    }


    /**
     * Добавление кастомных полей на страницу редактирования товара
     *
     * @param array $params Параметры товара.
     * @return void
     */
    public function hookActionProductFormBuilderModifier(array $params): void
    {
        $this->get(ProductFormModifier::class)->modify(
            (int) $params['id'],
            $params['form_builder']
        );
    }


    /**
     * Сохранение значений кастомных полей со страницы редактирования товара
     *
     * @param array $params Параметры товара.
     * @return void
     */
    public function hookActionProductSave(array $params): void
    {
        // We are using configuration table to save the data
        $productData = Tools::getValue('product');
        $idProduct = $params['id_product'];

        $fields = [
            'VTBPAY_MEASURE_FISCAL' => $productData['vtbpay_settings']['VTBPAY_MEASURE_FISCAL'],
            'VTBPAY_TAX_TYPE_FISCAL' => $productData['vtbpay_settings']['VTBPAY_TAX_TYPE_FISCAL'],
            'VTBPAY_PAYMENT_TYPE_FISCAL' => $productData['vtbpay_settings']['VTBPAY_PAYMENT_TYPE_FISCAL'],
            'VTBPAY_PAYMENT_SUBJECT_FISCAL' => $productData['vtbpay_settings']['VTBPAY_PAYMENT_SUBJECT_FISCAL'],
        ];

        foreach ($fields as $key => $value) {
            Configuration::updateValue($key . '_PRODUCT_' . $idProduct, $value);
        }
    }


    /**
     * Проверяет, доступна ли валюта для данного заказа.
     *
     * @param object $cart Объект корзины.
     *
     * @return bool Результат проверки валюты.
     */
    public function checkCurrency($cart): bool
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);
        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
            return false;
        } else {
            return true;
        }
    }


    /**
     * Получение URL для оплаты заказа.
     *
     * @return string URL для оплаты заказа.
     */
    public function getPayUrl()
    {
        $cart = $this->context->cart;
        $order_create = strtotime($cart->date_add);
        $total = $cart->getOrderTotal(true);
        $email = $this->getCustomerEmail();

        $return_url = $this->context->link->getModuleLink(
            $this->name,
            'return', [
                'key' => $cart->secure_key
            ],
            true
        );

        $two_stage = (bool) ($this->configure['VTBPAY_TWO_STAGE'] ?? false); // двухстадийная оплата
        $enable_fiscal = (bool) ($this->configure['VTBPAY_ENABLE_FISCAL'] ?? false);  // включение фискализации
        $items = [];
        if (!$two_stage && $enable_fiscal) $items = $this->getItems();

        $response = $this->vtb_api->getOrderLink(
            $this->currentOrder,
            $email,
            $order_create,
            $total,
            $return_url,
            $two_stage,
            $items,
            self::CMS_NAME,
            $this->version
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает email клиента, иначе выкидывает ошибку
     */
    private function getCustomerEmail(): string
    {
        // Get Email
        $customer = new Customer($this->context->cart->id_customer);
        $email = $customer->email ?: $this->configure['VTBPAY_EMAIL_FISCAL'];

        if (!self::validateEmail($email)) {
            throw new VtbPayException($this->l('Указанный адрес электронной почты не прошёл валидацию.'), [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems(): array
    {
        $items = [];
        $key = 1;

        // Настройки по умолчанию для фискальных параметров
        $fiscal_settings = [
            'VTBPAY_MEASURE_FISCAL' => 0,
            'VTBPAY_TAX_TYPE_FISCAL' => 'none',
            'VTBPAY_PAYMENT_TYPE_FISCAL' => 'full_prepayment',
            'VTBPAY_PAYMENT_SUBJECT_FISCAL' => 1
        ];

        // Перебор всех продуктов в корзине
        foreach ($this->context->cart->getProducts() as $item) {

            $product_fiscal = [];
            // Получение фискальных параметров для каждого продукта
            foreach ($fiscal_settings as $name => $default) {
                $product_fiscal[$name] = Configuration::get($name . '_PRODUCT_' . $item['id_product']) ?: '-';
                if ($product_fiscal[$name] === '-') $product_fiscal[$name] = $this->configure[$name] ?: $default;
            }

            $items[] = [
                'positionId' => $key,
                'name' => $item['name'],
                'description' => strip_tags($item['description_short']),
                'code' => (string) $item['reference'] ?: $item['id_product'],
                'price' => floatval($item['price']),
                'measure' => (int) $product_fiscal['VTBPAY_MEASURE_FISCAL'],
                'quantity' => (int) $item['cart_quantity'],
                'mass' => (int) round(($item['weight'] ?: 0) * 1000),
                'taxParams' => [
                    'taxType' => $product_fiscal['VTBPAY_TAX_TYPE_FISCAL']
                ],
                'paymentType' => $product_fiscal['VTBPAY_PAYMENT_TYPE_FISCAL'],
                'paymentSubject' => (int) $product_fiscal['VTBPAY_PAYMENT_SUBJECT_FISCAL'],
                'amount' => floatval($item['total']),
                'discount' => floatval($item['reduction'] ?? 0)
            ];

            $key++;
        }

        // Стоимость доставки
        $shipping_cost = floatval($this->context->cart->getOrderTotal(true, Cart::ONLY_SHIPPING)); // Стоимость доставки
        if ($shipping_cost > 0) {
            $shipping_method = new Carrier($this->context->cart->id_carrier);
            $items[] = [
                'positionId' => $key,
                'name' => $shipping_method->name ?: $this->l('Доставка'),
                'price' => $shipping_cost,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->configure['VTBPAY_TAX_TYPE_FISCAL'] ?: 'none'
                ],
                'paymentType' => $this->configure['VTBPAY_PAYMENT_TYPE_DELIVERY_FISCAL'] ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $shipping_cost
            ];
        }

        return $items;
    }


    /**
     * Получение статуса платежа из внешнего источника.
     *
     * @param string $order_id Идентификатор заказа.
     *
     * @return array Возвращает статус платежа.
     */
    public function getPaymentStatusData($order_id): array
    {
        $response = $this->vtb_api->getOrderInfo($order_id);

        return $response;
    }


    /**
     * Изменение статуса платежа в заказе (используется во front контроллерах).
     *
     * @param \Order $order Объект заказа.
     * @param string $payment_status Статус заказа.
     *
     * @return void
     */
    public function changePaymentStatus(
        \Order $order,
        string $payment_status
    ): void {
        $new_order_history = new OrderHistory();
        $new_order_history->id_order = (int) $order->id;
        $new_order_history->id_employee = 0; // ID сотрудника (0 для автоматической записи)

        $available_statuses_data = [
            'PAID' => [
                'status' => Configuration::get('PS_OS_PAYMENT')
            ],
            'PENDING' => [
                'status' => Configuration::get('PS_OS_PREPARATION')
            ],
            'REFUNDED' => [
                'status' => Configuration::get('PS_OS_REFUND')
            ],
            'VOIDED' => [
                'status' => Configuration::get('PS_OS_CANCELED')
            ]
        ];
        $new_order_status_data = $available_statuses_data[$payment_status] ?? Configuration::get('PS_OS_ERROR');
        $current_order_status = $order->getCurrentState(); // Получаем текущий статус заказа

        if ($current_order_status !== $new_order_status_data['status']) {
            if (isset($new_order_status_data['fnc'])) $new_order_status_data['fnc']();

            $new_order_history->changeIdOrderState(
                (int) $new_order_status_data['status'],
                (int) $order->id
            );

            if (!$new_order_history->addWithemail(true)) {
                throw new \Exception($this->l('Статус платежа не изменился.'));
            }
        }
    }


    /**
     * Устанавливает объект логирования для модуля.
     *
     * @return void
     */
    public function setVtbpayLogger(): void
    {
        // Ключи массива которые нужно замаскировать
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $logging = $this->configure['VTBPAY_LOGGING'] ?? false;

        $file_logger = new FileLogger(PrestaShopLoggerInterface::DEBUG);

        $file_logger->setFilename(dirname(__DIR__, 2) . '/var/logs/vtbpay-' . date('d-m-Y') . '.log');

        $this->logger = VtbPayLogger::getInstance()
                        ->setOption('showCurrentDate', false)
                        ->setOption('showLogLevel', false)
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitive_data_keys)
                        ->setCustomRecording(function($message) use ($file_logger)
                        {
                            // Логирование ошибки
                            $file_logger->logError($message);

                            PrestaShopLogger::addLog(
                                explode('Context:', $message)[0],
                                PrestaShopLoggerInterface::ERROR,
                                1,
                                'VtbPay'
                            );

                        }, VtbPayLogger::LOG_LEVEL_ERROR)
                        ->setCustomRecording(function($message) use ($file_logger, $logging)
                        {
                            // Логирование сообщения дебага
                            if ($logging) $file_logger->logDebug($message);

                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
