<?php

namespace Vtbpay\Traits;

trait Shipping
{
    /**
     * Обработчик события afterGetPaymentStatusReturn
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onAfterGetPaymentStatusShipping(array $vars)
    {
        // Проверка и обновление стоимости доставки
        $input_amount = $vars['payment_status_data']['object']['amount']['value'] ?? 0;
        if ($input_amount > $vars['order']->total_paid) {
            $order_code = $vars['payment_status_data']['object']['orderCode'] ?? '';
            $this->addDeliveryAndDeleteOld($vars['order'], $order_code, $input_amount);
        }
    }


    /**
     * Добавляем доставку в заказ и удаляем старую.
     *
     * @param Order $order Объект заказа.
     * @param string $order_code Код заказа.
     * @param float $new_total Новая итоговая сумма заказа.
     *
     * @return void
     */
    private function addDeliveryAndDeleteOld(
        \Order $order,
        string $order_code,
        float $new_total
    ): void {
        $order->total_shipping = round($new_total - $order->total_products_wt, 2);
        $order->total_shipping_tax_incl = $order->total_shipping;
        $order->total_paid = $new_total;
        $order->total_paid_tax_incl = $order->total_paid;

        $delivery_code = $this->getDeliveryCode($order_code);
        if (!empty($delivery_code)) {
            // Удаляем доставки с нулевой стоимостью
            $this->removeZeroShippingItems($order);

            $this->addShippingItem(
                $order,
                $delivery_code,
                $this->l('Доставка Почтой РФ'),
                $order->total_shipping
            );

            $order->update();
        }
    }


    /**
     * Получает код доставки
     *
     * @param string $order_code Код заказа.
     *
     * @return string
     */
    private function getDeliveryCode(string $order_code): string
    {
        $response = $this->vtb_api->getBundle($order_code);

        return $response['order']['deliveryParams']['recipientParams']['deliveryCode'] ?? '';
    }


    /**
     * Удаляет элементы доставки с нулевой стоимостью.
     *
     * @param Order $order Объект заказа.
     *
     * @return void
     */
    private function removeZeroShippingItems(\Order $order): void
    {
        $order_carriers = $order->getShipping();

        foreach ($order_carriers as $carrier) {
            if ((float) $carrier['shipping_cost_tax_incl'] === 0.0) {
                $this->logger->debug(__FUNCTION__ . ' > remove carrier: ', [
                    'id_order_carrier' => $carrier['id_order_carrier']
                ]);

                \Db::getInstance()->delete('order_carrier', 'id_order_carrier = ' . (int) $carrier['id_order_carrier']);
            }
        }
    }


    /**
     * Добавляет элемент доставки с указанной стоимостью.
     *
     * @param \Order $order Объект заказа.
     * @param string $shipping_method Метод доставки (delivery_code).
     * @param string $shipping_name Название метода доставки.
     * @param float $shipping_total Итоговая стоимость доставки.
     *
     * @return void
     */
    private function addShippingItem(
        \Order $order,
        string $shipping_method,
        string $shipping_name,
        float $shipping_total
    ): void {
        $this->logger->debug(__FUNCTION__ . ' > add shipping: ', [
            'shipping_method' => $shipping_method,
            'shipping_name' => $shipping_name,
            'shipping_total' => $shipping_total
        ]);

        // Поиск существующего элемента доставки по delivery_code
        $existing_carrier = \Carrier::getCarrierByReference((int) $shipping_method);

        if (!$existing_carrier) {
            // Создание нового объекта Carrier, если не найдено
            $existing_carrier = new \Carrier();
            $existing_carrier->name = $shipping_name;
            $existing_carrier->active = 1;
            $existing_carrier->is_module = 1;
            $existing_carrier->external_module_name = $this->name;
            $existing_carrier->shipping_external = 1;
            $existing_carrier->add();

            $this->logger->debug(__FUNCTION__ . ' > new carrier created: ', [
                'carrier_id' => $existing_carrier->id
            ]);
        }

        // Добавление доставки к заказу
        $order_carrier = new \OrderCarrier();
        $order_carrier->id_order = (int) $order->id;
        $order_carrier->id_carrier = (int) $existing_carrier->id;
        $order_carrier->shipping_cost_tax_incl = $shipping_total;
        $order_carrier->shipping_cost_tax_excl = $shipping_total;
        $order_carrier->add();
    }
}
