<?php
/**
 * "Платёжная система ВТБ"
 * Контроллер отвечает за обработку колбека и проверку статуса оплаты
 *
 * @author    VTB <acquiring_support@vtb.ru>
 */

/**
 * This Controller receive customer after approval on bank payment page
 */
class vtbpayWebhookModuleFrontController extends ModuleFrontController
{
    /**
     * @var int Идентификатор заказа
     */
    private ?int $order_id = null;


    /**
     * Assigns Smarty variables for the page main content.
     */
    public function initContent() {
        parent::initContent();
        $this->ajax = true; // enable ajax
    }


    /**
     * Обработчик POST-запроса.
     *
     * Этот метод обрабатывает POST-запрос после возврата клиента с банковской страницы оплаты.
     * Здесь происходит валидация данных, изменение статуса заказа и редирект на соответствующую страницу.
     *
     * @throws Exception Исключение выбрасывается в случае ошибки.
     * @return void
     */
    public function postProcess()
    {
        $this->module->setVtbpayLogger();
        $this->module->logger->setOption('additionalCommonText', 'webhook-' . rand(1111, 9999));

        $php_input = json_decode(file_get_contents('php://input'), true) ?: null;
        $this->order_id = $php_input['object']['orderId'] ?? null;

        $this->module->logger->debug(
            __CLASS__ . ' > ' . __FUNCTION__ . ' > webhook: ', [
            'php_input' => $php_input
        ]);

        try {
            // Валидация
            if (empty($this->order_id)) {
                throw new \Exception($this->module->l('Идентификатор заказа пуст.'));
            }

            $order = new Order((int) $this->order_id);
            if (!Validate::isLoadedObject($order)) {
                throw new \Exception($this->module->l('Заказ не найден.'));
            }
            $this->module->setVtbApi();
            $payment_status_data = $this->module->getPaymentStatusData($this->order_id);

            $this->module->event_dispatcher->dispatch('afterGetPaymentStatusWebhook', [
                'order' => $order,
                'payment_status_data' => $payment_status_data
            ]);

            $payment_status = $payment_status_data['object']['status']['value'] ?? '';
            $this->module->changePaymentStatus($order, $payment_status);

            $this->ajaxRender(json_encode([
                'success' => true
            ]));

        } catch (\Exception | VtbPayException $e) {
            // Обрабатываем исключение и логируем ошибку
            $this->handleErrorAndRedirect($e);
        }
    }


    /**
     * Обработка ошибки и перенаправление.
     *
     * @param Exception|VtbPayException $error_obj Объект ошибки.
     * @return void
     */
    private function handleErrorAndRedirect($error_obj)
    {
        // Handle exception and log error
        $context = [];
        if (method_exists($error_obj, 'getContext')) $context = $error_obj->getContext();

        $this->module->logger->error(sprintf(
            __CLASS__ . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
            $error_obj->getMessage(),
            $this->order_id ?: ''
        ), $context);

        $this->ajaxRender(json_encode([
            'success' => false,
            'message' => $error_obj->getMessage()
        ]));
    }
}
