<?php
/**
 * "Платёжная система ВТБ"
 * Контроллер отвечает за проверку статуса оплаты и редирект на страницу соответствующую статусу
 *
 * @author    VTB <acquiring_support@vtb.ru>
 */

/**
 * This Controller receive customer after approval on bank payment page
 */
class vtbpayReturnModuleFrontController extends ModuleFrontController
{
    /**
     * @var int Идентификатор заказа
     */
    private ?int $order_id = null;


    /**
     * Обработчик POST-запроса.
     *
     * Этот метод обрабатывает POST-запрос после возврата клиента с банковской страницы оплаты.
     * Здесь происходит валидация данных, изменение статуса заказа и редирект на соответствующую страницу.
     *
     * @throws Exception Исключение выбрасывается в случае ошибки.
     * @return void
     */
    public function postProcess()
    {
        $this->module->setVtbpayLogger();
        $this->module->logger->setOption('additionalCommonText', 'return-' . rand(1111, 9999));

        $this->order_id = $_GET['orderId'] ?? null;

        $this->module->logger->debug(
            __CLASS__ . ' > ' . __FUNCTION__ . ' > return: ', [
            'request' => $_REQUEST
        ]);

        try {
            // Валидация
            if (empty($this->order_id)) {
                throw new \Exception($this->module->l('Идентификатор заказа пуст.'));
            }

            if (!$this->isPaymentOptionAvailable()) {
                throw new \Exception($this->module->l('Способ оплаты не доступен.'));
            }

            if (!Validate::isLoadedObject($this->context->customer)) {
                throw new \Exception($this->module->l('Клиент не найден.'));
            }

            $order = new Order($this->order_id);
            if (!Validate::isLoadedObject($order)) {
                throw new \Exception($this->module->l('Заказ не найден.'));
            }
            
            $this->module->setVtbApi();
            $payment_status_data = $this->module->getPaymentStatusData($this->order_id);

            $this->module->event_dispatcher->dispatch('afterGetPaymentStatusReturn', [
                'order' => $order,
                'payment_status_data' => $payment_status_data
            ]);

            $payment_status = $payment_status_data['object']['status']['value'] ?? '';
            $this->module->changePaymentStatus($order, $payment_status);

            Tools::redirect($this->getRedirectPage($payment_status));

        } catch (\Exception | VtbPayException $e) {
            // Обрабатываем исключение и логируем ошибку
            $this->handleErrorAndRedirect($e);
        }
    }


    /**
     * Получение URL для редиректа.
     *
     * @param string $payment_status Статус платежа.
     * @return string URL для редиректа.
     */
    private function getRedirectPage($payment_status)
    {
        $redirect_page = $this->context->link->getPageLink(
            'order-detail',
            true,
            (int) $this->context->language->id, [
                'id_cart' => (int) $this->context->cart->id,
                'id_module' => (int) $this->module->id,
                'id_order' => (int) $this->order_id,
                'key' => $this->context->customer->secure_key,
            ]
        );

        $configure = $this->module->configure;

        $suitable_statuses_available = (
            $payment_status === 'PAID' ||
            $payment_status === 'RECONCLIED' ||
            $payment_status === 'PENDING' ||
            $payment_status === 'REFUNDED'
        );

        if (
            $suitable_statuses_available &&
            isset($configure['VTBPAY_SUCCESSFUL_PAYMENT_PAGE']) &&
            !empty($configure['VTBPAY_SUCCESSFUL_PAYMENT_PAGE'])
        ) {
            $redirect_page = $configure['VTBPAY_SUCCESSFUL_PAYMENT_PAGE'];
        }
        elseif (
            !$suitable_statuses_available &&
            isset($configure['VTBPAY_FAILED_PAYMENT_PAGE']) &&
            !empty($configure['VTBPAY_FAILED_PAYMENT_PAGE'])
        ) {
            $redirect_page = $configure['VTBPAY_FAILED_PAYMENT_PAGE'];
        }

        return $redirect_page;
    }


    /**
     * Проверка доступности опции оплаты.
     *
     * @return bool Возвращает `true`, если опция оплаты доступна, иначе `false`.
     */
    private function isPaymentOptionAvailable()
    {
        if (!empty($modules = Module::getPaymentModules())) {
            foreach ($modules as $module) {
                if (isset($module['name']) && $this->module->name === $module['name']) {
                    return true;
                }
            }
        }

        return false;
    }


    /**
     * Обработка ошибки и перенаправление.
     *
     * @param Exception|VtbPayException $error_obj Объект ошибки.
     * @return void
     */
    private function handleErrorAndRedirect($error_obj)
    {
        // Handle exception and log error
        $context = [];
        if (method_exists($error_obj, 'getContext')) $context = $error_obj->getContext();

        $this->module->logger->error(sprintf(
            __CLASS__ . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
            $error_obj->getMessage(),
            $this->order_id ?: ''
        ), $context);

        $this->errors[] = $error_obj->getMessage();
        $this->redirectWithNotifications($this->context->link->getPageLink(
            'cart',
            true,
            (int) $this->context->language->id, [
                'action' => 'show',
            ]
        ));
    }
}
