<?php
/**
 * "Платёжная система ВТБ"
 * Контроллер отвечает за генерацию ссылки оплаты и перенаправление
 *
 * @author    VTB <acquiring_support@vtb.ru>
 */
class vtbpayPaymentModuleFrontController extends ModuleFrontController
{
    /**
     * @var int Идентификатор заказа
     */
    private ?int $order_id = null;


    /**
     * Обработчик POST-запроса.
     *
     * @throws Exception Исключение выбрасывается в случае ошибки.
     * @return void
     */
    public function postProcess()
    {
        $this->module->setVtbpayLogger();
        $this->module->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            // Проверяем валидность контекста и доступность способа оплаты
            if (!$this->isValidContext() || !$this->isPaymentOptionAvailable()) {
                throw new \Exception($this->module->l('Валидация не удалась.'));
            }

            if (!Validate::isLoadedObject($this->context->customer)) {
                throw new \Exception($this->module->l('Клиент не найден.'));
            }

            // Инициализируем заказ
            if (!$this->initializeOrder()) {
                throw new \Exception($this->module->l('Заказ не создан.'));
            }

            if (
                !isset($this->module->currentOrder) ||
                empty($this->order_id = $this->module->currentOrder)
            ) {
                throw new \Exception($this->module->l('Идентификатор заказа пуст.'));
            }

            $this->module->setVtbApi();
            Tools::redirect($this->module->getPayUrl());

        } catch (\Exception | VtbPayException $e) {
            // Обрабатываем исключение и логируем ошибку
            $this->handleErrorAndRedirect($e);
        }
    }


    /**
     * Инициализация заказа.
     *
     * @return bool Результат инициализации заказа.
     */
    private function initializeOrder()
    {
        $cart = $this->context->cart;
        return $this->module->validateOrder(
            $cart->id,
            Configuration::get('PS_OS_BANKWIRE'),
            (float) $cart->getOrderTotal(true, Cart::BOTH),
            $this->module->displayName,
            $this->module->l('Заказ инициализирован "Платежной системой ВТБ"'),
            [],
            (int) $cart->id_currency,
            false,
            $cart->secure_key,
            $this->context->shop,
            null
        );
    }


    /**
     * Проверка, является ли контекст валидным.
     *
     * @return bool Результат проверки валидности контекста.
     */
    private function isValidContext()
    {
        $cart = $this->context->cart;
        return Validate::isLoadedObject($cart)
            && Validate::isUnsignedInt($cart->id_customer)
            && Validate::isUnsignedInt($cart->id_address_delivery)
            && Validate::isUnsignedInt($cart->id_address_invoice);
    }


    /**
     * Убедитесь, что данный способ оплаты по-прежнему доступен.
     *
     * @return bool Результат проверки доступности способа оплаты.
     */
    private function isPaymentOptionAvailable()
    {
        if (!empty($modules = Module::getPaymentModules())) {
            foreach ($modules as $module) {
                if (isset($module['name']) && $this->module->name === $module['name']) {
                    return true;
                }
            }
        }

        return false;
    }


    /**
     * Обработка ошибки и перенаправление.
     *
     * @param Exception|VtbPayException $error_obj Объект ошибки.
     * @return void
     */
    private function handleErrorAndRedirect($error_obj)
    {
        $context = [];
        if (method_exists($error_obj, 'getContext')) $context = $error_obj->getContext();

        $this->module->logger->error(sprintf(
            __CLASS__ . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
            $error_obj->getMessage(),
            $this->order_id ?? ''
        ), $context);

        $this->errors[] = $error_obj->getMessage();
        $this->redirectWithNotifications($this->context->link->getPageLink(
            'cart',
            true,
            (int) $this->context->language->id,[
                'action' => 'show',
            ]
        ));
    }
}
