<?php
/**
 * Payment Handler for VTB Payment System Notification
 *
 * This script handles payment notifications from the VTB payment system.
 * It verifies and processes payment updates for orders.
 *
 * @package VtbPay
 */

// Start the session
session_start();

// Include necessary classes and files
$_classPath = '../../../';
include($_classPath . 'class/obj.class.php');
include_once($_classPath . 'class/mail.class.php');

// Load required PHPShop classes
$loadedClasses = [
    'base', 'lang', 'order', 'file', 'xml', 'orm',
    'payment', 'modules', 'system', 'security',
    'parser', 'string'
];
foreach ($loadedClasses as $class) {
    PHPShopObj::loadClass($class);
}

// Initialize PHPShopBase, PHPShopSystem, and PHPShopModules
$PHPShopBase = new PHPShopBase($_classPath . 'inc/config.ini');
$PHPShopSystem = new PHPShopSystem();
$PHPShopModules = new PHPShopModules($_classPath . 'modules/');

// Include VTB API and Logger classes
include_once("{$_SERVER['DOCUMENT_ROOT']}/{$GLOBALS['SysValue']['class']['composer']}");

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;


$PHPShopModules->checkInstall('vtbpay');

/**
 * Payment class for handling VTB payment system notifications.
 */
class Payment extends PHPShopPaymentResult {

    private VtbPayLogger $logger;
    private string $request_type;
    private string $uid;

    /**
     * Constructor for the Payment class.
     */
    function __construct()
    {
        $envPath = dirname(__DIR__) . '/config/.env';
        if (file_exists($envPath)) (new Dotenv())->load($envPath);

        $this->option();
        parent::__construct();
    }


    /**
     * Configure the payment module options.
     *
     * This method retrieves the payment settings from the database and configures
     * the payment module accordingly. It sets the payment name and logging status.
     *
     * @throws Exception If a database error occurs during option retrieval.
     *
     * @return void
     */
    function option()
    {
        // An instance of PHPShopOrm to interact with the database.
        $PHPShopOrm = new PHPShopOrm($GLOBALS['SysValue']['base']['vtbpay']['vtbpay_system']);
        // The payment settings retrieved from the database.
        $this->option = $PHPShopOrm->select();

        $this->payment_name = $this->option['title'];
        $this->log = true;
        $this->request_type = $_GET['type'] ?? '';
        $this->setVtbpayLogger();
        $this->setOrderUid();
    }


    /**
     *     uid  $_GET   php://input.
     *
     * @return void
     */
    private function setOrderUid(): void
    {
        if ($this->request_type === 'return') {
            // Logging $_GET
            $this->logger->debug(
                __FUNCTION__ . ' > return: ', [
                'request_data' => $_GET
            ]);
            $this->uid = $_GET['orderId'] ?? '';
        }
        elseif ($this->request_type === 'webhook') {
            $phpInput = json_decode(file_get_contents('php://input'), true) ?: null;
            // Logging php input
            $this->logger->debug(
                __FUNCTION__ . ' > callback: ', [
                'php_input' => $phpInput
            ]);

            $this->uid = $phpInput['object']['orderId'] ?? '';
        }
    }


    /**
     * Update order information based on payment status from the payment provider.
     *
     * This method is responsible for processing payment status updates received from
     * the payment provider. It retrieves the payment status, logs the request if
     * logging is enabled, checks if the payment is successfully paid, updates the
     * order status, logs the payment, and redirects the user to the order details page.
     *
     * @throws \Exception If there's an issue with retrieving payment status, order information,
     *                   or if the payment is not successfully paid.
     *
     * @return void
     */
    function updateorder()
    {
        try {
            // Check if the order ID is present in the request
            if (empty($this->uid)) {
                throw new \Exception('  .');
            }

            $paymentStatusData = $this->getPaymentStatusData();

            $PHPShopOrm = new PHPShopOrm($GLOBALS['SysValue']['base']['orders']);
            $PHPShopOrm->debug = false;

            // Retrieve order information from the database
            $row = $PHPShopOrm->getOne(['id', 'uid'], ['uid' => "='{$this->uid}'"]);

            // Check if the order is found
            if (empty($row['id'])) {
                throw new \Exception('  .');
            }

            // Log the payment information
            $PHPShopOrm = new PHPShopOrm($GLOBALS['SysValue']['base']['payment']);
            $PHPShopOrm->insert([
                'uid_new' => str_replace('-', '', $row['uid']),
                'name_new' => $this->payment_name,
                'sum_new' => $paymentStatusData['object']['amount']['value'],
                'datas_new' => time()
            ]);

            $resultChangeStatus = $this->changeStatus($paymentStatusData);

            if ($this->request_type === 'return') {
                // Redirect the user back to the order details page after processing
                header("Location: /users/order.html?order_info={$row['uid']}#Order");
            }
            elseif ($this->request_type === 'webhook') {
                if ($resultChangeStatus) $this->done();
                else die('Error');
            }

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $this->uid
            ), $context);

            die(
                PHPShopString::utf8_win1251($e->getMessage()) .
                ($this->request_type === 'return' ? ' <a href="/"></a>' : '')
            );
        }
    }


    /**
     *        
     *
     * @param array $paymentStatusData    .
     *
     * @return bool
     */
    private function changeStatus(array $paymentStatusData): bool
    {
        $availableStatuses = [
            'PAID' => [
                'statusi' => '101',
                'fnc' => fn() => $this->setOrderPaid($this->uid)
            ],
            'PENDING' => [
                'statusi' => '102',
                'fnc' => fn() => $this->setOrderPending()
            ],
            'REFUNDED' => [
                'statusi' => '103',
                'fnc' => fn() => $this->setOrderRefunded()
            ]
        ];

        $paymentStatus = $paymentStatusData['object']['status']['value'] ?? '';
        $newOrderStatus = $availableStatuses[$paymentStatus] ?? [];
        $currentOrderStatus = $this->getOrderStatus();

        if (!empty($newOrderStatus)) {
            if ($currentOrderStatus !== $newOrderStatus['statusi']) $newOrderStatus['fnc']();
        }
        else {
            if ($currentOrderStatus !== '1') $this->setOrderStatus('1', 0);
            $this->logger->debug(sprintf(
                '    . Order ID: %s',
                $this->uid
            ));
        }

        return !empty($newOrderStatus);
    }


    /**
     *    
     *
     * @return string
     */
    private function getOrderStatus(): string
    {
        $PHPShopOrm = new PHPShopOrm($GLOBALS['SysValue']['base']['orders']);
        $PHPShopOrm->debug = $this->debug;
        $data = $PHPShopOrm->select(['statusi'], ['uid' => '=' . $this->uid], false, ['limit' => 1]);
        return $data['statusi'] ?? '';
    }


    /**
     *   
     *
     * @param string $id   
     * @param string $name   
     * @param string $color   
     *
     * @return void
     */
    private function createOrderStatus(
        string $id,
        string $name,
        string $color
    ): void {
        $PHPShopOrm = new PHPShopOrm($GLOBALS['SysValue']['base']['order_status']);
        $PHPShopOrm->debug = $this->debug;
        $data = $PHPShopOrm->select(['id'], ['id' => '=' . $id], false, ['limit' => 1]);

        if (!is_array($data)) {
            $PHPShopOrm->clean();
            $PHPShopOrm->insert([
                'id_new' => $id,
                'name_new' => $name,
                'color_new' => $color
            ]);
        }
    }


    /**
     *   
     *
     * @param string $statusId   
     * @param int $paid   
     *
     * @return void
     */
    private function setOrderStatus(
        string $statusId,
        int $paid
    ): void {
        $PHPShopOrm = new PHPShopOrm($GLOBALS['SysValue']['base']['orders']);
        $PHPShopOrm->debug = $this->debug;
        $PHPShopOrm->update(
            ['statusi_new' => $statusId, 'paid_new' => $paid],
            ['uid' => '="' . $this->uid . '"']
        );
    }


    /**
     *     .
     *
     * @return void
     */
    private function setOrderPending()
    {
        $this->createOrderStatus(
            '102',
            ' ',
            '#f8dda7'
        );

        $this->setOrderStatus(
            '102',
            0
        );

        $this->logger->debug(sprintf(
            ' ,   . Order ID: %s',
            $this->uid
        ));
    }


    /**
     *      .
     *
     * @return void
     */
    private function setOrderRefunded(): void
    {
        $this->createOrderStatus(
            '103',
            '',
            '#b22222'
        );

        $this->setOrderStatus(
            '103',
            0
        );

        $this->logger->debug(sprintf(
            '  . Order ID: %s',
            $this->uid
        ));
    }


    /**
     *     VTB API.
     *
     * @return array
     */
    private function getPaymentStatusData(): array
    {
        return $this->getVtbApi()->getOrderInfo($this->uid) ?: [];
    }


    /**
     * Create a new VtbApi instance with the given payment method configurations.
     *
     * @return VtbApi The new VtbApi instance.
     *
     * @throws Exception If unable to create the VtbApi instance.
     */
    private function getVtbApi(): VtbApi
    {
        return new VtbApi(
            html_entity_decode($this->option['client_id']),
            html_entity_decode($this->option['client_secret']),
            (bool) $this->option['test_mode'],
            $this->option['merchant_authorization'] ?: ''
        );
    }


    /**
     *     .
     *
     * @return void
     */
    private function setVtbpayLogger(): void
    {
        $logger = VtbPayLogger::getInstance();
        $logging = (bool) $this->option['logging'];
        $sensitiveDataKeys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];
        $requestTypeKey = $this->request_type . '-' . rand(1111, 9999);

        $this->logger = $logger->setOption('additionalCommonText', $requestTypeKey)
            ->setLogFilePath(dirname(__DIR__) . '/log/vtbpay-' . date('d-m-Y') . '.log')
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitiveDataKeys)
            ->setCustomRecording(function ($message) use ($logger, $logging) {
                //   
                if ($logging) $logger->writeToFile(PHPShopString::utf8_win1251($message));
            }, VtbPayLogger::LOG_LEVEL_DEBUG)
            ->setCustomRecording(function ($message) use ($logger, $logging) {
                //   
                $logger->writeToFile(PHPShopString::utf8_win1251($message));
            }, VtbPayLogger::LOG_LEVEL_ERROR);
    }
}

new Payment();
