<?php
/**
 * VTBPay Module Integration
 *
 * This script defines a hook function for integrating the VTBPay payment system with the PHPShop module.
 * It handles payment processing, API requests, and logging for the VTBPay integration.
 *
 * @package VtbPay
 */
//   autoload,     payment/result.php
include_once($GLOBALS['SysValue']['class']['composer']);

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;


class VtbPayDone
{
    public const CMS_NAME = 'PHPShop';

    private $obj;
    private $value;
    private $paymetSettings;
    private $logger;

    public function __construct($obj, $value)
    {
        $this->obj = $obj;
        $this->value = $value;

        $envPath = dirname(__DIR__) . '/config/.env';
        if (file_exists($envPath)) (new Dotenv())->load($envPath);

        // Initialize PHPShop ORM and get settings
        $PHPShopOrm = new PHPShopOrm($GLOBALS['SysValue']['base']['vtbpay']['vtbpay_system']);
        $this->paymetSettings = $PHPShopOrm->select();

        // Initialize VTBPay logger
        $this->setVtbpayLogger();
    }


    /**
     * VTBPay Payment Hook
     *
     * This function is a hook for processing payments through the VTBPay payment system.
     *
     * @param string $rout The routing information for the current order.
     *
     * @return void This function doesn't return a value directly, but it modifies the $obj parameter with payment information.
     *
     * @throws \Exception If there are errors during payment processing or API communication.
     */
    public function handle($rout)
    {
        if ($rout === 'END' && $this->obj->PHPShopPayment->objRow['name'] === $this->paymetSettings['title']) {
            try {
                // Check if the payment URL is present and not empty in the response.
                if (empty($payUrl = $this->getPayUrl())) {
                    throw new \Exception('   URL- .');
                }

                // Set payment information and build payment form
                if (
                    isset($this->paymetSettings['payment_info']) &&
                    !empty($paymentInfo = $this->paymetSettings['payment_info'])
                ) {
                    $this->obj->set('payment_info', $paymentInfo);
                }

                $this->obj->set('payment_forma', PHPShopText::button(
                    $this->paymetSettings['btn_text'],
                    "window.location.replace('{$payUrl}')",
                    'btn btn-primary btn-pill'
                ));

                $form = ParseTemplateReturn($GLOBALS['SysValue']['templates']['vtbpay']['vtbpay_payment_form'], true);

                // Clear the shopping cart
                unset($_SESSION['cart']);
            } catch (\Exception | VtbPayException $e) {
                // Handle exception and log error
                $context = [
                    'file_exception' => $e->getFile(),
                    'line_exception' => $e->getLine(),
                ];
                if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

                $this->logger->error(sprintf(
                    __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                    $e->getMessage(),
                    $this->value['ouid'] ?? ''
                ), $context);

                $this->obj->set('mesageText', '<span style="color:red;">' . PHPShopString::utf8_win1251($e->getMessage()) . '</span>');
                $form = ParseTemplateReturn($GLOBALS['SysValue']['templates']['order_forma_mesage']);
            }

            // Set order message with payment form
            $this->obj->set('orderMesage', $form);
        }
    }


    /**
     *     
     *
     * @return string
     */
    private function getPayUrl(): string
    {
        $mail = $this->getCustomerEmail();

        // Check if the total amount is set in the order object
        if (!isset($this->obj->total) || empty($total = $this->obj->total)) {
            throw new \Exception('   .');
        }

        // Build the return URL
        $returnUrl = 'http://' . $_SERVER['HTTP_HOST'] . '/phpshop/modules/vtbpay/payment/result.php?type=return';

        //  
        $items = [];
        if (
            !$this->paymetSettings['two_stage'] &&
            $this->paymetSettings['enable_fiscal']
        ) {
            $items = $this->getItems();
        }

        // Get the payment link from VTB API
        $response = $this->getVtbApi()->getOrderLink(
            $this->value['ouid'],
            $mail,
            time(),
            $total,
            $returnUrl,
            (bool) $this->paymetSettings['two_stage'],
            $items,
            self::CMS_NAME,
            $this->paymetSettings['version']
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     *  email    
     *
     * @return string E-mail .
     */
    private function getCustomerEmail(): string
    {
        $email = $_POST['mail'] ?? $this->paymetSettings['email_fiscal'];

        if (!self::validateEmail($email)) {
            throw new VtbPayException('      .', [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     *  Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email): bool
    {
        //      RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        //  email     
        list($localPart, $domainPart) = explode('@', $email);

        //     ( 1  64 )
        $localLength = strlen($localPart);
        if ($localLength < 1 || $localLength > 64) {
            return false;
        }

        //     ( 1  255 )
        $domainLength = strlen($domainPart);
        if ($domainLength < 1 || $domainLength > 253) {
            return false;
        }

        //     
        $allowedTlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowedTlds) && !empty($allowedTlds[0])) {
            //  TLD   
            $domainParts = explode('.', $domainPart);
            $tld = strtolower(end($domainParts));

            // ,  TLD    
            if (!in_array($tld, $allowedTlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     *      .
     *
     * @return array  .
     */
    private function getItems(): array
    {
        $orders = unserialize($this->obj->order);
        $items = [];
        $key = 1;
        //      
        $fiscalSettings = [
            'measure_fiscal' => 0,
            'tax_type_fiscal' => 'none',
            'payment_type_fiscal' => 'full_prepayment',
            'payment_subject_fiscal' => 1
        ];

        //    
        foreach ($orders['Cart']['cart'] as $item) {
            $PHPShopProduct = new PHPShopProduct($item['id']);
            $productFiscal = [];
            //      
            foreach ($fiscalSettings as $name => $default) {
                $productFiscal[$name] = $PHPShopProduct->getParam('vtbpay_' . $name) ?: '-';
                if ($productFiscal[$name] === '-') $productFiscal[$name] = $this->paymetSettings[$name] ?? $default;
            }

            // 
            if ((float) $this->obj->discount > 0 && empty($item['promo_price'])) {
                $price = $item['price'] - ($item['price'] * (float) $this->obj->discount / 100);
            } else {
                $price = $item['price'];
            }

            // 
            if ($this->obj->bonus_minus > 0 && $key == 1) {
                $price = $price - $this->obj->bonus_minus;
            }

            $price = floatval($price);
            $amount = floatval($price * (int) $item['num']);

            $items[] = [
                'positionId' => $key,
                'name' => PHPShopString::win_utf8($item['name']),
                'code' => (string) ($item['uid'] ?: $item['id']),
                'price' => $price,
                'measure' => (int) $productFiscal['measure_fiscal'],
                'quantity' => (int) $item['num'],
                'taxParams' => [
                    'taxType' => $productFiscal['tax_type_fiscal']
                ],
                'paymentType' => $productFiscal['payment_type_fiscal'],
                'paymentSubject' => (int) $productFiscal['payment_subject_fiscal'],
                'amount' => $amount
            ];

            $key++;
        }

        if ($this->obj->delivery > 0) {
            $delivery = $this->obj->PHPShopDelivery->objRow;
            $deliveryPrice = floatval($delivery['price']);
            $items[] = [
                'positionId' => $key,
                'name' => PHPShopString::win_utf8($delivery['city'] ?? ''),
                'price' => $deliveryPrice,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->paymetSettings['tax_type_fiscal'] ?? 'none'
                ],
                'paymentType' => $this->paymetSettings['payment_type_delivery_fiscal'] ?? 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $deliveryPrice
            ];
        }

        return $items;
    }


    /**
     *      API .
     *
     * @return VtbApi     API .
     */
    private function getVtbApi(): VtbApi
    {
        return new VtbApi(
            html_entity_decode($this->paymetSettings['client_id']),
            html_entity_decode($this->paymetSettings['client_secret']),
            (bool)$this->paymetSettings['test_mode'],
            $this->paymetSettings['merchant_authorization'] ?: ''
        );
    }


    /**
     *     .
     */
    private function setVtbpayLogger(): void
    {
        $logger = VtbPayLogger::getInstance();
        $logging = (bool) $this->paymetSettings['logging'];
        $sensitiveDataKeys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $this->logger = $logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999))
            ->setLogFilePath(dirname(__DIR__) . '/log/vtbpay-' . date('d-m-Y') . '.log')
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitiveDataKeys)
            ->setCustomRecording(function ($message) use ($logger, $logging) {
              //   
                if ($logging) $logger->writeToFile(PHPShopString::utf8_win1251($message));
            }, VtbPayLogger::LOG_LEVEL_DEBUG)
            ->setCustomRecording(function ($message) use ($logger) {
                //   
                $logger->writeToFile(PHPShopString::utf8_win1251($message));
            }, VtbPayLogger::LOG_LEVEL_ERROR);
    }
}


//  ,   
function send_to_order_mod_vtbpay_hook($obj, $value, $rout)
{
    $handler = new VtbPayDone($obj, $value);
    $handler->handle($rout);
}

// $addHandler  
$addHandler = [
    'send_to_order' => 'send_to_order_mod_vtbpay_hook'
];
