<?php
namespace Opencart\Admin\Controller\Extension\Vtbpay\Payment;

class Vtbpay extends \Opencart\System\Engine\Controller
{

		/**
		 * Install method
		 *
		 * Called on an install event
		 *
		 * @return void
		 */
		public function install()
		{
				$this->load->model('extension/vtbpay/payment/vtbpay');
				$this->model_extension_vtbpay_payment_vtbpay->install();

				$this->load->model('setting/event');

				// Удаляем старые события, если они есть
        $this->model_setting_event->deleteEventByCode('vtbpay_product_form');
				$this->model_setting_event->deleteEventByCode('vtbpay_add_product');
				$this->model_setting_event->deleteEventByCode('vtbpay_edit_product');

				// Регистрируем новые события
				$this->model_setting_event->addEvent([
						'code'        => 'vtbpay_product_form',
						'description' => 'Add Vtbpay fields to product form',
						'trigger'     => 'admin/view/catalog/product_form/after',
						'action'      => 'extension/vtbpay/event/vtbpay.onProductForm',
						'status'      => 1,
						'sort_order'  => 2
				]);
        $this->model_setting_event->addEvent([
            'code'        => 'vtbpay_add_product',
            'description' => 'VTB Pay: Add fields to product',
            'trigger'     => 'admin/model/catalog/product/addProduct/after',
            'action'      => 'extension/vtbpay/event/vtbpay.onAddProduct',
            'status'      => 1,
            'sort_order'  => 2
        ]);
        $this->model_setting_event->addEvent([
            'code'        => 'vtbpay_edit_product',
            'description' => 'VTB Pay: Edit fields in product',
            'trigger'     => 'admin/model/catalog/product/editProduct/after',
            'action'      => 'extension/vtbpay/event/vtbpay.onEditProduct',
            'status'      => 1,
            'sort_order'  => 2
        ]);
		}


		/**
		 * Uninstall method
		 *
		 * Called on a delete event
		 *
		 * @return void
		 */
		public function uninstall()
		{
				$this->load->model('extension/vtbpay/payment/vtbpay');
				$this->model_extension_vtbpay_payment_vtbpay->uninstall();

				// Удаляем события, если они есть
        $this->model_setting_event->deleteEventByCode('vtbpay_product_form');
        $this->model_setting_event->deleteEventByCode('vtbpay_add_product');
				$this->model_setting_event->deleteEventByCode('vtbpay_edit_product');
		}


		/**
		 * Index function
		 *
		 * This method handles the loading of the VTB Pay extension page in the OpenCart admin.
		 * It sets up the necessary data for the view, including breadcrumbs, save and back URLs,
		 * and VTB Pay specific configuration values.
		 *
		 * @return void
		 */
		public function index(): void
		{
				$this->load->language('extension/vtbpay/payment/vtbpay');
				$this->document->setTitle($this->language->get('heading_title'));
				$this->load->model('localisation/order_status');

				// set value for form fields
        $fields = $this->getSettingsFields();

        foreach ($fields as $field => $field_data) {
            $data['settings']['payment_vtbpay_' . $field] = $field_data;
            if ($field_data['type'] === 'separator') continue;
						$data['settings']['payment_vtbpay_' . $field]['value'] = $this->config->get('payment_vtbpay_' . $field);
        }

				$this->showPaymentSettingsPage($data);
		}


		/**
		 * getSettingsFields method
		 *
		 * Getting an array of plugin settings
		 *
		 * @return array
		 */
		private function getSettingsFields()
		{
				return [
						'desc' => [
								'type' => 'textarea',
								'title' => $this->language->get('entry_desc'),
								'default' => $this->language->get('default_value_desc'),
								'desc' => $this->language->get('desc_desc')
						],
						'client_id' => [
								'type' => 'text',
								'title' => $this->language->get('entry_client_id'),
								'required' => true,
								'desc' => $this->language->get('desc_client_id')
						],
						'client_secret' => [
								'type' => 'text',
								'title' => $this->language->get('entry_client_secret'),
								'required' => true,
								'desc' => $this->language->get('desc_client_secret')
						],
						'merchant_authorization' => [
								'type' => 'text',
								'title' => $this->language->get('entry_merchant_authorization'),
								'desc' => $this->language->get('desc_merchant_authorization')
						],
						'successful_transaction_status' => [
								'type' => 'select',
								'title' => $this->language->get('entry_successful_transaction_status'),
								'desc' => $this->language->get('desc_successful_transaction_status'),
								'default' => '15',
								'options' => $this->getPreparedOrderStatuses()
						],
						'failed_transaction_status' => [
								'type' => 'select',
								'title' => $this->language->get('entry_failed_transaction_status'),
								'desc' => $this->language->get('desc_failed_transaction_status'),
								'default' => '10',
								'options' => $this->getPreparedOrderStatuses()
						],
						'test_mode' => [
								'type' => 'checkbox',
								'title' => $this->language->get('entry_test_mode'),
								'desc' => $this->language->get('desc_test_mode')
						],
						'logging' => [
								'type' => 'checkbox',
								'title' => $this->language->get('entry_logging'),
								'desc' => $this->language->get('desc_logging')
						],
						'two_stage' => [
								'type' => 'checkbox',
								'title' => $this->language->get('entry_two_stage'),
								'desc' => $this->language->get('desc_two_stage')
						],
						'status' => [
								'type' => 'checkbox',
								'title' => $this->language->get('entry_status')
						],
						'sort_order' => [
								'type' => 'text',
								'title' => $this->language->get('entry_sort_order')
						],
						/**
						 * Поля настроек фискализации
						 */
						'section_fiscal' => [
								'type' => 'separator',
								'title' => $this->language->get('entry_section_fiscal')
						],
						'enable_fiscal' => [
								'type' => 'checkbox',
								'title' => $this->language->get('entry_enable_fiscal'),
								'desc' => $this->language->get('desc_enable_fiscal')
						],
						'ofd_fiscal' => [
								'type' => 'select',
								'title' => $this->language->get('entry_ofd_fiscal'),
								'desc' => $this->language->get('desc_ofd_fiscal'),
								'default' => '1ofd',
								'options' => [
										'1ofd' => $this->language->get('option_1ofd'),
										'atol' => $this->language->get('option_atol')
								]
						],
						// Настройки фискализации только для АТОЛ
						'test_mode_atol_fiscal' => [
								'type' => 'checkbox',
								'title' => $this->language->get('entry_test_mode_atol_fiscal'),
								'desc' => $this->language->get('desc_test_mode_atol_fiscal')
						],
						'login_atol_fiscal' => [
								'type' => 'text',
								'title' => $this->language->get('entry_login_atol_fiscal'),
								'desc' => $this->language->get('desc_login_atol_fiscal')
						],
						'pass_atol_fiscal' => [
								'type' => 'text',
								'title' => $this->language->get('entry_pass_atol_fiscal'),
								'desc' => $this->language->get('desc_pass_atol_fiscal')
						],
						'kkt_atol_fiscal' => [
								'type' => 'text',
								'title' => $this->language->get('entry_kkt_atol_fiscal'),
								'desc' => $this->language->get('desc_kkt_atol_fiscal')
						],
						'inn_atol_fiscal' => [
								'type' => 'text',
								'title' => $this->language->get('entry_inn_atol_fiscal'),
								'desc' => $this->language->get('desc_inn_atol_fiscal')
						],
						'tax_system_atol_fiscal' => [
								'type' => 'select',
								'title' => $this->language->get('entry_tax_system_atol_fiscal'),
								'desc' => $this->language->get('desc_tax_system_atol_fiscal'),
								'default' => 'osn',
								'options' => [
										'osn' => $this->language->get('option_osn'),
										'usn_income' => $this->language->get('option_usn_income'),
										'usn_income_outcome' => $this->language->get('option_usn_income_outcome'),
										'envd' => $this->language->get('option_envd'),
										'esn' => $this->language->get('option_esn'),
										'patent' => $this->language->get('option_patent')
								]
						],
						// Общие настройки фискализации
						'email_fiscal' => [
								'type' => 'text',
								'title' => $this->language->get('entry_email_fiscal'),
								'desc' => $this->language->get('desc_email_fiscal')
						],
						'payment_type_delivery_fiscal' => [
								'type' => 'select',
								'title' => $this->language->get('entry_payment_type_delivery_fiscal'),
								'desc' => $this->language->get('desc_payment_type_delivery_fiscal'),
								'default' => 'full_prepayment',
								'options' => [
										'full_prepayment' => $this->language->get('option_full_prepayment'),
										'prepayment' => $this->language->get('option_prepayment'),
										'advance' => $this->language->get('option_advance'),
										'full_payment' => $this->language->get('option_full_payment'),
										'partial_payment' => $this->language->get('option_partial_payment'),
										'credit' => $this->language->get('option_credit'),
										'credit_payment' => $this->language->get('option_credit_payment')
								]
						],
						'section_subtitle_fiscal' => [
								'type' => 'separator',
								'title' => $this->language->get('entry_section_subtitle_fiscal')
						],
						'measure_fiscal' => [
								'type' => 'select',
								'title' => $this->language->get('entry_measure_fiscal'),
								'desc' => $this->language->get('desc_measure_fiscal'),
								'default' => '0',
								'options' => [
										'0' => $this->language->get('option_piece_unit'),
										'10' => $this->language->get('option_gram'),
										'11' => $this->language->get('option_kilogram'),
										'12' => $this->language->get('option_ton'),
										'20' => $this->language->get('option_centimeter'),
										'21' => $this->language->get('option_decimeter'),
										'22' => $this->language->get('option_meter'),
										'30' => $this->language->get('option_sq_cm'),
										'31' => $this->language->get('option_sq_dm'),
										'32' => $this->language->get('option_sq_m'),
										'40' => $this->language->get('option_ml'),
										'41' => $this->language->get('option_liter'),
										'42' => $this->language->get('option_cubic_m'),
										'50' => $this->language->get('option_kwh'),
										'51' => $this->language->get('option_gcal'),
										'70' => $this->language->get('option_day'),
										'71' => $this->language->get('option_hour'),
										'72' => $this->language->get('option_minute'),
										'73' => $this->language->get('option_second'),
										'80' => $this->language->get('option_kb'),
										'81' => $this->language->get('option_mb'),
										'82' => $this->language->get('option_gb'),
										'83' => $this->language->get('option_tb'),
										'255' => $this->language->get('option_other_units')
								]
						],
						'tax_type_fiscal' => [
								'type' => 'select',
								'title' => $this->language->get('entry_tax_type_fiscal'),
								'desc' => $this->language->get('desc_tax_type_fiscal'),
								'default' => 'none',
								'options' => [
										'none' => $this->language->get('option_no_vat'),
										'vat0' => $this->language->get('option_vat0'),
										'vat10' => $this->language->get('option_vat10'),
										'vat110' => $this->language->get('option_vat110'),
										'vat20' => $this->language->get('option_vat20'),
										'vat120' => $this->language->get('option_vat120'),
										'vat5' => $this->language->get('option_vat5'),
										'vat7' => $this->language->get('option_vat7'),
										'vat105' => $this->language->get('option_vat105'),
										'vat107' => $this->language->get('option_vat107')
								]
						],
						'payment_type_fiscal' => [
								'type' => 'select',
								'title' => $this->language->get('entry_payment_type_fiscal'),
								'desc' => $this->language->get('desc_payment_type_fiscal'),
								'default' => 'full_prepayment',
								'options' => [
										'full_prepayment' => $this->language->get('option_full_prepayment'),
										'prepayment' => $this->language->get('option_prepayment'),
										'advance' => $this->language->get('option_advance'),
										'full_payment' => $this->language->get('option_full_payment'),
										'partial_payment' => $this->language->get('option_partial_payment'),
										'credit' => $this->language->get('option_credit'),
										'credit_payment' => $this->language->get('option_credit_payment')
								]
						],
						'payment_subject_fiscal' => [
								'type' => 'select',
								'title' => $this->language->get('entry_payment_subject_fiscal'),
								'desc' => $this->language->get('desc_payment_subject_fiscal'),
								'default' => '1',
								'options' => [
										'1' => $this->language->get('option_goods'),
										'2' => $this->language->get('option_excise_goods'),
										'3' => $this->language->get('option_work'),
										'4' => $this->language->get('option_service'),
										'5' => $this->language->get('option_gambling_bet'),
										'6' => $this->language->get('option_gambling_win'),
										'7' => $this->language->get('option_lottery_bet'),
										'8' => $this->language->get('option_lottery_win'),
										'9' => $this->language->get('option_rights'),
										'10' => $this->language->get('option_payment'),
										'11' => $this->language->get('option_agency_fee'),
										'12' => $this->language->get('option_payout'),
										'13' => $this->language->get('option_other_subject'),
										'14' => $this->language->get('option_property_right'),
										'15' => $this->language->get('option_non_operating_income'),
										'16' => $this->language->get('option_insurance_contributions'),
										'17' => $this->language->get('option_trade_fee'),
										'18' => $this->language->get('option_resort_fee'),
										'19' => $this->language->get('option_deposit'),
										'20' => $this->language->get('option_expense'),
										'21' => $this->language->get('option_pension_ip'),
										'22' => $this->language->get('option_pension'),
										'23' => $this->language->get('option_health_ip'),
										'24' => $this->language->get('option_health'),
										'25' => $this->language->get('option_social'),
										'26' => $this->language->get('option_casino_payment'),
										'27' => $this->language->get('option_cash_issuance_bank_agent'),
										'30' => $this->language->get('option_excise_goods_no_code'),
										'31' => $this->language->get('option_excise_goods_with_code'),
										'32' => $this->language->get('option_goods_no_code'),
										'33' => $this->language->get('option_goods_with_code')
								]
						]
				];
		}


		/**
     * getPreparedOrderStatuses method
     *
     * Gets an array of order statuses prepared for display
     *
     * @return array
     */
    private function getPreparedOrderStatuses()
    {
        $this->load->model('localisation/order_status');

        $order_statuses = $this->model_localisation_order_status->getOrderStatuses();

        return array_combine(
            array_column($order_statuses, 'order_status_id'),
            array_column($order_statuses, 'name')
        );
    }


		/**
		 * showPaymentSettingsPage method
		 *
		 * This method prepares the breadcrumbs and other necessary data for the view,
		 * and then it renders the payment settings page.
		 *
		 * @param array $data The data array that will be passed to the view.
		 * @return void
		 */
		private function showPaymentSettingsPage(array $data): void
		{
				// Добавляем скрипт
				$this->document->addScript('/extension/vtbpay/admin/view/javascript/payment/vtbpay.js');

				$data['breadcrumbs'] = [];

				$data['breadcrumbs'][] = [
						'text' => $this->language->get('text_home'),
						'href' => $this->url->link('common/dashboard', 'user_token=' . $this->session->data['user_token'])
				];

				$data['breadcrumbs'][] = [
						'text' => $this->language->get('text_extension'),
						'href' => $this->url->link('marketplace/extension', 'user_token=' . $this->session->data['user_token'] . '&type=payment')
				];

				$data['breadcrumbs'][] = [
						'text' => $this->language->get('heading_title'),
						'href' => $this->url->link('extension/vtbpay/payment/vtbpay', 'user_token=' . $this->session->data['user_token'])
				];

				$data['save'] = $this->url->link('extension/vtbpay/payment/vtbpay.save', 'user_token=' . $this->session->data['user_token']);
				$data['back'] = $this->url->link('marketplace/extension', 'user_token=' . $this->session->data['user_token'] . '&type=payment');

				$data['header'] = $this->load->controller('common/header');
				$data['column_left'] = $this->load->controller('common/column_left');
				$data['footer'] = $this->load->controller('common/footer');

				$this->response->setOutput($this->load->view('extension/vtbpay/payment/vtbpay', $data));
		}


		/**
		 * Save function
		 *
		 * This method handles the submission of the VTB Pay configuration form in the OpenCart admin.
		 * It checks for the necessary permissions, validates required fields, and if everything is correct,
		 * saves the provided data using the setting model. It then returns a JSON response indicating success
		 * or any errors encountered.
		 *
		 * @return void
		 */
		public function save(): void
		{
				$this->load->language('extension/vtbpay/payment/vtbpay');

				$json = [];

				if (!$this->user->hasPermission('modify', 'extension/vtbpay/payment/vtbpay')) {
						$json['error']['warning'] = $this->language->get('error_permission');
				}

				// Объявляем обязательные поля с условиями
        $validation_rules = [
            'client_id' => true, // Всегда обязательно
            'client_secret' => true, // Всегда обязательно
            'login_atol_fiscal' => $this->request->post['payment_vtbpay_ofd_fiscal'] === 'atol',
            'pass_atol_fiscal' => $this->request->post['payment_vtbpay_ofd_fiscal'] === 'atol',
            'kkt_atol_fiscal' => $this->request->post['payment_vtbpay_ofd_fiscal'] === 'atol',
            'inn_atol_fiscal' => $this->request->post['payment_vtbpay_ofd_fiscal'] === 'atol'
        ];

				// Проверка всех правил
        foreach ($validation_rules as $field => $is_required) {
            if ($is_required && empty($this->request->post['payment_vtbpay_' . $field])) {
                $json['error']['payment_vtbpay_' . $field] = $this->language->get('error_' . $field);
            }
        }

				if (empty($json)) {
						$this->load->model('setting/setting');
						$this->model_setting_setting->editSetting('payment_vtbpay', $this->request->post);
						$json['success'] = $this->language->get('text_success');
				}
				elseif (empty($json['error']['warning'])) {
						$json['error']['warning'] = $this->language->get('text_required_fields_error');
				}

				// Устанавливает результата валидации
				$this->response->addHeader('Content-Type: application/json');
				$this->response->setOutput(json_encode($json));
		}
}
