<?php

namespace Vtbpay\Traits;

use \Vtbpay\Classes\Api\AtolApi,
    \Vtbpay\Classes\Exception\VtbPayException;

trait Atol
{
    /**
     * Обработчик события beforeGetPaymentStatusWebhook
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onBeforeGetPaymentStatusAtol(array $vars)
    {
        // Если есть uuid, значит к вебхуку обращается АТОЛ
        $uuid = $vars['php_input']['uuid'] ?? '';
        if (!empty($uuid)) {
            $this->webhookHandlerAtol($uuid);
            die('OK');
        }
    }


    /**
     * Обработчик события paymentPaid
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentPaidAtol(array $vars)
    {
        if (
            $this->config->get('payment_vtbpay_enable_fiscal') &&
            $this->config->get('payment_vtbpay_ofd_fiscal') === 'atol'
        ) $this->sendReceiptAtol();
    }


    /**
     * Обработчик события paymentRefunded
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentRefundedAtol(array $vars)
    {
        if (
            $this->config->get('payment_vtbpay_enable_fiscal') &&
            $this->config->get('payment_vtbpay_ofd_fiscal') === 'atol'
        ) $this->sendReceiptAtol(true);
    }


    /**
     * Создаем новый экземпляр класса AtolApi с заданными конфигурациями платёжной системы.
     *
     * @return AtolApi
     */
    private function getAtolApi(): AtolApi
    {
        return new AtolApi(
            $this->config->get('payment_vtbpay_login_atol_fiscal'),
            $this->config->get('payment_vtbpay_pass_atol_fiscal'),
            $this->config->get('payment_vtbpay_inn_atol_fiscal'),
            $this->config->get('payment_vtbpay_tax_system_atol_fiscal'),
            $this->config->get('payment_vtbpay_kkt_atol_fiscal'),
            $this->config->get('payment_vtbpay_email_fiscal'),
            (bool) $this->config->get('payment_vtbpay_test_mode_atol_fiscal')
        );
    }


    /**
     * Проверяет статус формирования чека в АТОЛ и оповещает при
     *
     * @param string $uuid Полученные данные
     *
     * @return void
     */
    private function webhookHandlerAtol(string $uuid): void
    {
        $response = $this->getAtolApi()->getReceiptInfo($uuid);

        if ($response['status'] === 'fail'){
            $this->sendReceiptErrorEmail(sprintf(
                  $this->language->get('error_atol_check'),
                  $this->order['order_id']
            ));
        }
        elseif ($response['status'] === 'wait') {
            $this->sendReceiptErrorEmail(sprintf(
                  $this->language->get('wait_atol_check'),
                  $this->order['order_id']
            ));
        }
    }


    /**
     * Отправляет письмо об ошибке при выдаче чека.
     *
     * Использует настройки почты из конфигурации OpenCart (SMTP или стандартный mail).
     *
     * @param string $message Текст сообщения об ошибке (HTML).
     * @return void
     */
    private function sendReceiptErrorEmail($message): void
    {
        // Получаем название магазина и email администратора из настроек
        $store_name = $this->config->get('config_name') ?: 'OpenCart 3';
        $admin_email = $this->config->get('config_email') ?: "noreply@{$_SERVER['HTTP_HOST']}";

        // Формируем тему письма
        $subject = $this->language->get('receipt_issuance_error') . ': ' . $store_name;

        // Инициализируем класс Mail
        $mail = new \Mail($this->config->get('config_mail_engine')); // Почтовый движок (например, mail или smtp)

        // Устанавливаем настройки почты из конфигурации
        $mail->parameter = $this->config->get('config_mail_parameter');
        $mail->smtp_hostname = $this->config->get('config_mail_smtp_hostname');
        $mail->smtp_username = $this->config->get('config_mail_smtp_username');
        $mail->smtp_password = html_entity_decode($this->config->get('config_mail_smtp_password'), ENT_QUOTES, 'UTF-8');
        $mail->smtp_port = $this->config->get('config_mail_smtp_port');
        $mail->smtp_timeout = $this->config->get('config_mail_smtp_timeout');

        // Устанавливаем параметры письма
        $mail->setTo($this->config->get('payment_vtbpay_email_fiscal')); // Email получателя
        $mail->setFrom($admin_email);     // Email отправителя
        $mail->setSender(html_entity_decode($store_name, ENT_QUOTES, 'UTF-8')); // Имя отправителя
        $mail->setSubject(html_entity_decode($subject, ENT_QUOTES, 'UTF-8'));   // Тема письма
        $mail->setHtml($message);        // Тело письма в формате HTML

        // Отправляем письмо
        $mail->send();
    }


    /**
     * Отправляет в АТОЛ чек прихода или возврата на печать и возвращает результат.
     *
     * @param bool $is_return Данный чек "Возврат прихода" или "Приход"
     *
     * @return array
     */
    private function sendReceiptAtol(bool $is_return = false): array
    {
        $order_id = $this->order['order_id'];

        try {
            $email = $this->getCustomerEmail();

            $items = $this->preparingItemsForAtol(
                $this->getItems()
            );

            $total = self::parseAmount($this->order['total']);

            $callback_url = $this->url->link('extension/payment/vtbpay/webhook');

            $response = $this->getAtolApi()->sendReceipt(
                $order_id,
                $items,
                $total,
                $email,
                $callback_url,
                $is_return
            );

            return $response;

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $this->executeErrorScenario(
                $e,
                $order_id,
                __FUNCTION__
            );

            $this->sendReceiptErrorEmail(sprintf(
                  $this->language->get('error_atol_check'),
                  $order_id
            ));

            return [];
        }
    }


    /**
     * Подготавливает элементы для АТОЛ.
     *
     * @param array $items Элементы для подготовки.
     *
     * @return array Массив подготовленных элементов.
     */
    private function preparingItemsForAtol(array $items): array
    {
        return array_map(function ($item) {
            return [
                'name' => $item['name'], // Лейбл
                'price' => $item['price'], // Цена товара за 1шт
                'sum' => $item['amount'], // Итоговая цена (цена * кол-во)
                'quantity' => $item['quantity'], // Кол-во позиций товара (сколько шт)
                'measure' => $item['measure'], // Единица измерения
                'vat' => [
                    'type' => $item['taxParams']['taxType'] // Налоговая ставка (НДС)
                ],
                'payment_method' => $item['paymentType'], // Способ расчета
                'payment_object' => $item['paymentSubject'], // Предмет расчета
            ];
        }, $items);
    }
}
