<?php

require_once(DIR_SYSTEM . 'library/vtbpay/vendor/autoload.php');

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Common\EventDispatcher,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Vtbpay\Traits\Atol,
    \Symfony\Component\Dotenv\Dotenv;

class ControllerExtensionPaymentVtbPay extends Controller
{
    use Atol;

    const CMS_NAME = 'OpenCart 3';
    const EXTENSION_VERSION = '1.8.7';

    private array $order;
    private VtbPayLogger $logger;
    private EventDispatcher $event_dispatcher;


    /**
     * ControllerExtensionPaymentVtbPay constructor
     *
     * Set up the logging functionality for the VtbPay transactions.
     *
     * @param object $registry The registry of the OpenCart system
     */
    public function __construct($registry)
    {
        parent::__construct($registry);

        $env_path = DIR_SYSTEM . 'library/vtbpay/config/.env';
       if (file_exists($env_path)) (new Dotenv())->load($env_path);

        $this->event_dispatcher = new EventDispatcher($this); // Инициализация менеджера событий

        $this->setVtbpayLogger();
        $this->load->language('extension/payment/vtbpay');
    }


    /**
     * Создаем новый экземпляр класса VtbApi с заданными конфигурациями платёжной системы.
     *
     * @return VtbApi
     */
    private function getVtbApi(): VtbApi
    {
        return new VtbApi(
            $this->config->get('payment_vtbpay_client_id'),
            $this->config->get('payment_vtbpay_client_secret'),
            (bool) $this->config->get('payment_vtbpay_test_mode'),
            $this->config->get('payment_vtbpay_merchant_authorization') ?: ''
        );
    }


    /**
     * Index function
     *
     * Prepares data for the checkout view, and returns the rendered HTML.
     *
     * @return string Rendered HTML of the checkout view
     */
    public function index()
    {
        $data['button_confirm'] = $this->language->get('button_confirm');
        $data['action'] = $this->url->link('extension/payment/vtbpay/confirm', '', true);
        $data['logo'] = '/catalog/view/theme/default/image/extension/payment/vtbpay.png';
        $data['css'] = '/catalog/view/theme/default/stylesheet/extension/payment/vtbpay.css';
        $data['js'] = '/catalog/view/theme/default/js/extension/payment/vtbpay.js';
        $data['desc'] = $this->config->get('payment_vtbpay_desc') ?: '';
        return $this->load->view('extension/payment/vtbpay', $data);
    }


    /**
     * Confirm function
     *
     * Initiates a payment transaction with VTB Pay and redirects the user to the VTB Pay payment page.
     * Logs all VTB Pay transactions.
     *
     * @return void
     */
    public function confirm()
    {
        $order_id = $this->session->data['order_id'] ?? null;
        $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            if (!$order_id) {
                throw new \Exception($this->language->get('error_no_order_id'));
            }

            $this->setOrder($order_id);

            if (empty($pay_url = $this->getPayUrl())) {
                throw new \Exception($this->language->get('error_pay_url_not_available'));
            }

            $this->response->redirect($pay_url);

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $this->executeErrorScenario(
                $e,
                $order_id,
                __FUNCTION__,
                'redirect'
            );
        }
    }


    /**
     * Получает URL-адрес платежа для перенаправления.
     *
     * @return string URL-адрес платежа.
     */
    private function getPayUrl(): string
    {
        $order_id = $this->order['order_id'];
        $email = $this->getCustomerEmail();
        $total = self::parseAmount($this->order['total']);
        $return_url = $this->url->link('extension/payment/vtbpay/return'); // Get return URL
        $two_stage = (bool) $this->config->get('payment_vtbpay_two_stage');
        $enable_fiscal = (bool) $this->config->get('payment_vtbpay_enable_fiscal');
        $ofd_fiscal = $this->config->get('payment_vtbpay_ofd_fiscal') ?: '';

        $items = [];
        if (
            !$two_stage &&
            $enable_fiscal &&
            $ofd_fiscal === '1ofd'
        ) $items = $this->getItems();

        $this->model_checkout_order->addOrderHistory(
            $order_id,
            1,
            'orderId: ' . $order_id,
            true
        );

        $response = $this->getVtbApi()->getOrderLink(
            $order_id,
            $email,
            time(),
            $total,
            $return_url,
            $two_stage,
            $items,
            self::CMS_NAME,
            self::EXTENSION_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает массив товаров заказа и доставку.
     *
     * @return array Массив товаров.
     */
    private function getItems(): array
    {
        $this->load->model('catalog/product');

        // Настройки по умолчанию для фискальных параметров
        $fiscal_settings = [
            'payment_vtbpay_measure_fiscal' => 0,
            'payment_vtbpay_tax_type_fiscal' => 'none',
            'payment_vtbpay_payment_type_fiscal' => 'full_prepayment',
            'payment_vtbpay_payment_subject_fiscal' => 1
        ];
        $items = [];
        $key = 1;
        // Добавление в массив товаров
        foreach ($this->order['products'] as $item) {
            $product = $this->model_catalog_product->getProduct($item['product_id']);

            $product_fiscal = [];
            // Получение фискальных параметров для каждого продукта
            foreach ($fiscal_settings as $name => $default) {
                $product_fiscal[$name] = $product[$name] ?: '-';
                if ($product_fiscal[$name] === '-') $product_fiscal[$name] = $this->config->get($name) ?: $default;
            }

            $items[] = [
                'positionId' => $key,
                'name' => $item['name'],
                'code' => (string) $item['product_id'],
                'price' => self::parseAmount($item['price']),
                'measure' => (int) $product_fiscal['payment_vtbpay_measure_fiscal'],
                'quantity' => (int) $item['quantity'],
                'taxParams' => [
                    'taxType' => $product_fiscal['payment_vtbpay_tax_type_fiscal']
                ],
                'paymentType' => $product_fiscal['payment_vtbpay_payment_type_fiscal'],
                'paymentSubject' => (int) $product_fiscal['payment_vtbpay_payment_subject_fiscal'],
                'amount' => self::parseAmount($item['total'])
            ];

            $key++;
        }

        // Добавление в массив доставки
        foreach ($this->order['totals'] as $total) {
            if ($total['code'] === 'shipping' && $total['value'] > 0) {

                $shipping_amount = self::parseAmount($total['value']);
                $items[] = [
                    'positionId' => $key,
                    'name' => $total['title'],
                    'price' => $shipping_amount,
                    'measure' => 0,
                    'quantity' => 1,
                    'taxParams' => [
                        'taxType' => $this->config->get('payment_vtbpay_tax_type_fiscal') ?: 'none'
                    ],
                    'paymentType' => $this->config->get('payment_vtbpay_payment_type_delivery_fiscal') ?: 'full_prepayment',
                    'paymentSubject' => 4,
                    'amount' => $shipping_amount
                ];

                $key++;
            }
        }

        return $items;
    }


    /**
     * Return function
     *
     * Processes the callback from VTB Pay, updates the order status based on the callback data,
     * and prepares and displays a success or failure message to the user.
     *
     * @return void
     */
    public function return()
    {
        $order_id = $this->request->get['orderId'] ?? null;
        $this->logger->setOption('additionalCommonText', 'return-' . rand(1111, 9999));

        // Logging $_REQUEST
        $this->logger->debug(
            'redirect > ' . __FUNCTION__ . ': ', [
            'request_data' => $_REQUEST
        ]);

        try {
            if (empty($order_id)) {
                throw new \Exception($this->language->get('error_no_order_id'));
            }

            $this->setOrder($order_id);

            $payment_status = $this->getPaymentStatus();
            $this->changeStatus($payment_status);

            $this->cart->clear();
            $this->response->redirect($this->url->link('checkout/success'));

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $this->executeErrorScenario(
                $e,
                $order_id,
                __FUNCTION__,
                'redirect'
            );
        }
    }


    /**
     * Обработчик callback, вызываемый платёжкой при изменение статуса платежа.
     *
     * @return void
     */
    public function webhook(): void
    {
        $php_input = json_decode(file_get_contents('php://input'), true) ?: null;
        $this->logger->setOption('additionalCommonText', 'webhook-' . rand(1111, 9999));

        // Logging php input
        $this->logger->debug(
            'callback > ' . __FUNCTION__ . ': ', [
            'php_input' => $php_input
        ]);

        $order_id = $php_input['object']['orderId'] ?? '';
        $external_id = $php_input['external_id'] ?? ''; // Только для Атола

        if (!empty($external_id)) {
            $order_id = explode('-', $external_id)[0];
        }

        try {
            if (empty($order_id)) {
                throw new \Exception($this->language->get('error_no_order_id'));
            }

            $this->setOrder($order_id);

            $this->event_dispatcher->dispatch('beforeGetPaymentStatusWebhook', [
                'php_input' => $php_input
            ]);

            $payment_status = $this->getPaymentStatus();
            $this->changeStatus($payment_status);

            $this->response->setOutput('OK');

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $this->executeErrorScenario(
                $e,
                $order_id,
                __FUNCTION__,
                'show'
            );
        }
    }


    /**
     * Получени информации о платеже и зменение статуса заказа
     *
     * @param string $payment_status Статус оплаты
     *
     * @return void
     */
    private function changeStatus(string $payment_status): void
    {
        $current_order_status = $this->order['order_status_id'];
        $order_id = $this->order['order_id'];
        $available_statuses = [
            'PAID' => [
                'status' => $this->config->get('payment_vtbpay_successful_transaction_status'),
                'fnc_first' => function() {
                    $this->event_dispatcher->dispatch('paymentPaid');
                }
            ],
            'PENDING' => [
                'status' => '1'
            ],
            'REFUNDED' => [
                'status' => '13',
                'fnc_first' => function() {
                    $this->event_dispatcher->dispatch('paymentRefunded');
                }
            ]
        ];
        $order_status_data = $available_statuses[$payment_status] ?? [
            'status' => $this->config->get('payment_vtbpay_failed_transaction_status'),
            'fnc_second' => function() {
                throw new \Exception($this->language->get('error_payment_not_paid'));
            }
        ];

        if (!empty($order_status_data)) {
            if ($current_order_status !== $order_status_data['status']) {
                $this->model_checkout_order->addOrderHistory(
                    $order_id,
                    $order_status_data['status'],
                    $payment_status,
                    true
                );

                if (isset($order_status_data['fnc_first'])) $order_status_data['fnc_first']();
            }
            if (isset($order_status_data['fnc_second'])) $order_status_data['fnc_second']();
        }
    }


    /**
     * Получение статуса заказа из VTB API.
     *
     * @return string Значение статуса заказа, если оно существует, или пустая строка в противном случае.
     */
    private function getPaymentStatus(): string
    {
        $order_id = $this->order['order_id'];

        $response = $this->getVtbApi()->getOrderInfo($order_id);

        return $response['object']['status']['value'] ?? '';
    }


    /**
     * Инициализируем свойство order результатом выполнения функции getOrder.
     *
     * @param int $order_id Идентификатор заказа.
     *
     * @return self|Exception Объект заказа, если он существует, или Exception в противном случае.
     */
    private function setOrder(int $order_id): self
    {
        $this->load->model('checkout/order');

        $this->order = $this->model_checkout_order->getOrder($order_id);
        if (!$this->order) {
            throw new \Exception($this->language->get('error_order_not_found'));
        }

        $this->order['products'] = $this->model_checkout_order->getOrderProducts($order_id);
        $this->order['totals'] = $this->model_checkout_order->getOrderTotals($order_id);

        return $this;
    }


    /**
     * Получает email клиента, иначе выкидывает ошибку
     */
    private function getCustomerEmail(): string
    {
        // Get Email
        $email = $this->order['email'] ?: $this->config->get('payment_vtbpay_email_fiscal');
        if (!self::validateEmail($email)) {
            throw new VtbPayException($this->language->get('error_email_not_valid'), [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Преобразует строку в число с плавающей запятой.
     *
     * @param mixed $amount Сумма для преобразования.
     * @return float Преобразованное значение суммы.
     */
    private static function parseAmount($amount): float
    {
        if (is_string($amount)) {
            $amount = str_replace([' ', ','], ['', '.'], $amount);
        }

        return floatval(number_format($amount, 2, '.', ''));
    }


    /**
     * Сценарий, выполняемый после отлова исключения
     *
     * @param \Exception|VtbPayException $e   Объект исключения.
     * @param int $order_id                   Идентификатор заказа.
     * @param string $caller_func_name        Название функции, котоорая вызывает обработчик исключений.
     * @param bool $end_action                Действие которое нужно выполнить в конце.
     *
     * @return void
     */
    private function executeErrorScenario(
        $e,
        $order_id,
        $caller_func_name = '',
        $end_action = ''
    ): void {
        $context = [
            'file_exception' => $e->getFile(),
            'line_exception' => $e->getLine(),
        ];
        if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

        $this->logger->error(sprintf(
            $caller_func_name . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
            $e->getMessage(),
            $order_id ?: ''
        ), $context);

        if ($end_action === 'redirect') {
            $this->session->data['error'] = $e->getMessage();
            $this->response->redirect($this->url->link('checkout/failure', '', true));
        }
        elseif ($end_action === 'show') {
            $this->response->setOutput($e->getMessage());
        }
    }


    /**
     * Инициализация и настройка объекта класса VtbPayLogger.
     *
     * Эта функция инициализирует и настраивает логгер, используемый плагином VtbPay для ведения журнала.
     *
     * @return void
     */
    private function setVtbpayLogger(): void
    {
        // Ключи массива которые нужно замаскировать
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $logging = (bool) $this->config->get('payment_vtbpay_logging');
        $oc_log = new Log('vtbpay-' . date('d-m-Y') . '.log');

        $this->logger = VtbPayLogger::getInstance()
                        ->setOption('showCurrentDate', false)
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitive_data_keys)
                        ->setCustomRecording(function($message) use ($oc_log){
                            $oc_log->write($message);
                        }, VtbPayLogger::LOG_LEVEL_ERROR)
                        ->setCustomRecording(function($message) use ($oc_log, $logging){
                            if ($logging) $oc_log->write($message);
                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
