<?php
require_once dirname(__DIR__, 3) . '/vtbpay/vendor/autoload.php';

// Checking PHP Version
if (version_compare(PHP_VERSION, '7.4.0', '<')) {
    die('Payment module "Payment system VTB" requires PHP version 7.4.0 or higher.');
}

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;

class nc_payment_system_vtbpay extends nc_payment_system
{

    const CMS_NAME = 'NetCat';
    const PLUGIN_VERSION = '1.4.3';

    /**
     * Indicates whether the payment system is automatic or not.
     *
     * @var bool
     */
    protected $automatic = TRUE;

    /**
     * List of accepted currencies.
     *
     * @var array
     */
    protected $accepted_currencies = ['RUB', 'RUR'];

    /**
     * Mapping of currency aliases to their canonical names.
     *
     * @var array
     */
    protected $currency_map = ['RUR' => 'RUB'];

    /**
     * Settings of the site in the payment system.
     * Includes client ID, client secret, merchant authorization, and various URLs.
     *
     * @var array
     */
    protected $settings = [
        'client_id' => '',
        'client_secret' => '',
        'merchant_authorization' => '',
        'test_mode' => 0,
        'logging' => 0,
        'success_url' => '',
        'fail_url' => '',
        'enable_fiscal' => 0,
        'email_fiscal' => '',
        'payment_type_delivery_fiscal' => 'full_prepayment', // full_prepayment - Предоплата 100%
        'measure_fiscal' => 0, // 0 - Применяется для предметов расчета, которые могут быть реализованы поштучно или единицами
        'tax_type_fiscal' => 'none', // none - Без НДС
        'payment_type_fiscal' => 'full_prepayment', // full_prepayment - Предоплата 100%
        'payment_subject_fiscal' => 1, // 1 - Товар
    ];

    /**
     * Parameters to be sent to the payment system.
     *
     * @var array
     */
    protected $request_parameters = [];

    /**
     * Parameters received from the payment system.
     *
     * @var array
     */
    protected $callback_response = [
        'orderId' => null
    ];

    /**
     * URL for making payments.
     *
     * @var string|null
     */
    protected $pay_url = null;

    /**
     * URL for receiving responses from the payment system.
     *
     * @var string|null
     */
    protected $response_url = null;

    /**
     * Logger object.
     *
     * @var VtbPayLogger
     */
    private $logger;


    /**
     * The class constructor. Adds listeners for the specified events
     * related to payment requests and callbacks.
     */
    public function __construct()
    {
        parent::__construct();

        $env_path = dirname(__DIR__, 3) . '/vtbpay/config/.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);

        $this->setVtbpayLogger();

        $event = nc_core::get_object()->event;

        $event->add_listener(
            nc_payment_system::EVENT_AFTER_PAY_REQUEST, [
            $this,
            'on_after_pay_request'
        ]);
        $event->add_listener(
            nc_payment_system::EVENT_AFTER_PAY_CALLBACK, [
            $this,
            'on_after_pay_callback'
        ]);
        $event->add_listener(
            nc_payment_system::EVENT_ON_PAY_CALLBACK_ERROR, [
            $this,
            'on_pay_callback_error'
        ]);
    }


    /**
     * Redirects to the payment page after a payment request is created
     *
     * @param nc_payment_system_vtbpay $this_obj Object of the current class
     * @param nc_payment_invoice $invoice The invoice for the payment
     */
    public function on_after_pay_request(
        nc_payment_system_vtbpay $this_obj,
        nc_payment_invoice $invoice = null
    ) {
        if (!empty($this->pay_url)) {
            header('Location: ' . $this->pay_url);
        }
    }


    /**
     * Redirects to the appropriate URL based on the payment callback response
     *
     * @param nc_payment_system_vtbpay $this_obj Object of the current class
     * @param nc_payment_invoice $invoice The invoice for the payment
     */
    public function on_after_pay_callback(
        nc_payment_system_vtbpay $this_obj,
        nc_payment_invoice $invoice = null
    ) {
        if (!empty($this->response_url)) {
            header('Location: ' . $this->response_url);
        }
    }


    /**
     * Redirects to the error URL.
     *
     * @param nc_payment_system_vtbpay $this_obj Object of the current class
     * @param nc_payment_invoice $invoice The invoice for the payment
     */
    public function on_pay_callback_error(
        nc_payment_system_vtbpay $this_obj,
        nc_payment_invoice $invoice = null
    ) {
        if (!empty($invoice)) $this->on_payment_failure($invoice);
        header('Location: ' . $this->get_setting('fail_url') ?: '/');
    }


    /**
     * Processes the payment request and initiates the payment procedure
     *
     * @param nc_payment_invoice $invoice The invoice for the payment
     */
    public function execute_payment_request(nc_payment_invoice $invoice)
    {
        $invoice_id = $invoice->get_id();
        $order_id = $invoice->get('order_id');

        $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            if ($invoice->get('order_source') !== 'netshop') {
                throw new \Exception('Cannot process invoice with ' . __CLASS__ . ': order_source is not supported');
            }

            $nc_core = nc_Core::get_object();
            $netshop = $nc_core->modules->netshop;

            $order = $netshop->load_order($order_id);
            if (!$order) {
                throw new \Exception("Order with id={$order_id} not found");
            }

            // Return url
            $site = $nc_core->catalogue->get_url_by_id($nc_core->catalogue->id());
            $return_url = $site .
                          nc_module_path('payment') .
                          'callback.php?type=result&paySystem=' . __CLASS__ .
                          '&invoice_id=' . $invoice_id;

            // Total
            $total_sum = (float) $order->get_totals();

            // Customer email
            $customer_email = $this->get_customer_email($invoice);

            // Get payment link
            $pay_url = $this->get_pay_url(
                $order_id,
                $customer_email,
                $total_sum,
                $return_url
            );

            if (!empty($pay_url)) {
                $invoice->set('last_response', "Payment URL: {$pay_url}");
                $this->pay_url = $pay_url;
            } else {
                $msg = "Failed to get payment URL. Order id: {$order_id}";
                $invoice->set('last_response', $msg)
                        ->set('status', nc_payment_invoice::STATUS_CALLBACK_ERROR);
                $this->add_error($msg);
            }

            $invoice->save();

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);
            // Log the caught exception for debugging purposes.
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s;',
                $e->getMessage(),
                $order_id ?: ''
            ), $context);

            $this->add_error($e->getMessage());
        }
    }


    /**
     * Retrieves the payment URL for redirection.
     *
     * @param string $order_id Order ID
     * @param string $email Customer email
     * @param float $total Amount to pay
     * @param string $return_url Return url
     *
     * @return string The payment URL.
     * @throws \Exception Throws an exception if the payment URL cannot be obtained.
     */
    private function get_pay_url(
        string $order_id,
        string $email,
        float $total,
        string $return_url
    ) {
        $netshop = nc_netshop::get_instance();
        $order = $netshop->load_order($order_id);

        $enable_fiscal = $this->get_setting('enable_fiscal') ?: false;
        $items = [];
        if ($enable_fiscal) $items = $this->get_items($order);

        $response = $this->getVtbApi()->getOrderLink(
            $order_id,
            $email,
            time(),
            $total,
            $return_url,
            false,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает email клиента, иначе выкидывает ошибку
     */
    private function get_customer_email(object $invoice): string
    {
        // Get Email
        $email = $invoice->get('customer_email') ?:
                 nc_payment_register::get_default_customer_email() ?:
                 $this->get_setting('email_fiscal');
        if (!self::validate_email($email)) {
            throw new VtbPayException('The email address provided has not been validated.', [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validate_email(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function get_items($order)
    {
        $items = [];
        $order_items = $order->get_items();

        foreach ($order_items as $key => $item) {
            $items[] = [
                'positionId' => ($key + 1),
                'name' => $item['Name'], // FullName
                'code' => $item['Article'] ?: '',
                'price' => floatval($item['ItemPrice']),
                'measure' => (int) $this->get_setting('measure_fiscal') ?: 0,
                'quantity' => (int) $item['Qty'],
                'taxParams' => [
                    'taxType' => $this->get_setting('tax_type_fiscal') ?: 'none'
                ],
                'paymentType' => $this->get_setting('payment_type_fiscal') ?: 'full_prepayment',
                'paymentSubject' => (int) $this->get_setting('payment_subject_fiscal') ?: 1,
                'amount' => floatval($item['ItemPrice'] * $item['Qty'])
            ];
        }
        $delivery_cost = floatval($order['DeliveryCost'] ?? 0);
        // Стоимость доставки
        if ($delivery_cost > 0) {
            $items[] = [
                'positionId' => ($key + 2),
                'name' => 'Доставка',
                'price' => $delivery_cost,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->get_setting('tax_type_fiscal') ?: 'none'
                ],
                'paymentType' => $this->get_setting('payment_type_delivery_fiscal') ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $delivery_cost
            ];
        }
        return $items;
    }


    /**
     * Validates that all necessary parameters for making a payment request are present
     * Throws an error if any required parameters are missing
     */
    public function validate_payment_request_parameters()
    {
        if (!$this->get_setting('client_id')) {
            $this->add_error('The "client_id" parameter is required.');
        }

        if (!$this->get_setting('client_secret')) {
            $this->add_error('The "client_secret" parameter is required.');
        }
    }


    /**
     * Validates the response from the payment callback
     *
     * @param nc_payment_invoice $invoice The invoice associated with the payment
     */
    public function validate_payment_callback_response(nc_payment_invoice $invoice = null)
    {
        if (!$invoice) {
            $this->add_error(NETCAT_MODULE_PAYMENT_ERROR_INVOICE_NOT_FOUND);
        }

        if (empty($this->get_response_value('orderId'))) {
            $this->add_error(NETCAT_MODULE_PAYMENT_ORDER_ID_IS_NULL);
        }
    }


    /**
     * Handles the response after the payment operation is complete
     *
     * @param nc_payment_invoice $invoice The invoice associated with the payment
     */
    protected function on_response(nc_payment_invoice $invoice = null)
    {
        $order_id = $this->get_response_value('orderId');

        $this->logger->setOption('additionalCommonText', 'return-' . rand(1111, 9999));

        try {
            $payment_status = $this->get_payment_status($order_id);

            if ($payment_status !== 'PAID') {
                throw new \Exception('Payment status is not PAID.');
            }

            $this->on_payment_success($invoice);

            $this->response_url = $this->get_setting('success_url') ?: '/';

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);
            // Log the caught exception for debugging purposes.
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s;',
                $e->getMessage(),
                $order_id ?: ''
            ), $context);

            // На данном этапе уже нет смысла вызывать $this->add_error()
            $this->on_payment_failure($invoice);
            $this->response_url = $this->get_setting('fail_url') ?: '/';
        }
    }


    /**
     * Method requests the payment status from the VtbApi
     * for a specific order and returns it, throwing an exception if no status is received.
     *
     * @param string $order_id Order ID
     *
     * @return string Returns payment status string.
     * @throws \Exception Throws exception when no information about payment status.
     */
    private function get_payment_status(string $order_id)
    {
        $response = $this->getVtbApi()->getOrderInfo($order_id);

        return $response['object']['status']['value'] ?? '';
    }


    /**
     * Loads the invoice during the payment callback
     *
     * @return bool|nc_payment_invoice Returns the loaded invoice if it exists, otherwise false
     */
    public function load_invoice_on_callback()
    {
        return $this->load_invoice($this->get_response_value('invoice_id'));
    }


    /**
     * Logs an error and adds it to the list of errors
     *
     * @param string $string The error message to be logged and added
     */
    protected function add_error($string)
    {
        parent::add_error($string);
    }


    /**
     * Устанавливает объект логирования для модуля.
     */
    private function setVtbpayLogger()
    {
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true);
        $logger = VtbPayLogger::getInstance();
        $dir = $_SERVER['DOCUMENT_ROOT'] . '/netcat_files/logs/';

        $this->logger = $logger
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitive_data_keys)
            ->setLogFilePath(
                $dir . 'vtbpay-' . date('d-m-Y') . '.log'
            )->setCustomRecording(function($message) use ($logger) {
                $logging = $this->get_setting('logging') ?: false;
                if ($logging) $logger->writeToFile($message);
            }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }


    /**
     * Получает объект для работы с API ВТБ.
     *
     * @return VtbApi Объект для работы с API ВТБ.
     */
    public function getVtbApi()
    {
        return new VtbApi(
            $this->get_setting('client_id'),
            $this->get_setting('client_secret'),
            (bool) $this->get_setting('test_mode'),
            $this->get_setting('merchant_authorization') ?: ''
        );
    }
}
