<?php
require_once 'lib/vendor/autoload.php';

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;

class Pactioner extends Actioner
{
    const CMS_NAME = 'Moguta';
    const PLUGIN_VERSION = '1.3.6';

    /**
     * The name of the plugin.
     * @var string $pluginName
     */
    private static $pluginName = 'vtbpay-payment';

    /**
     * Logger object.
     * @var VtbPayLogger
     */
    private $logger;

    /**
     * Payment parameters
     *
     * @var array
     */
    private $paymentParams;


    /**
     * Конструктор класса.
     */
    public function __construct($lang = false)
    {
        parent::__construct($lang);

        $env_path = __DIR__ . '/config/.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);

        $this->setVtbpayLogger();
    }


    /**
     * getVtbApi
     *
     * Initialize and return an instance of the VtbApi with provided payment parameters.
     *
     * @return VtbApi Instance of VtbApi with provided payment parameters.
     */
    private function getVtbApi()
    {
        $paymentParams = $this->paymentParams;
        return new VtbApi(
            $paymentParams['client_id'],
            $paymentParams['client_secret'],
            filter_var($paymentParams['test_mode'], FILTER_VALIDATE_BOOLEAN),
            $paymentParams['merchant_authorization'] ?? ''
        );
    }


    /**
     * getPayLink
     *
     * Generate a payment link for a given order. Handles exceptions and logs payment events.
     *
     * @return string|bool The generated payment link or false if an error occurred.
     */
    public function getPayLink()
    {
        $paymentId = $_POST['paymentId'];
        $mgBaseDir = $_POST['mgBaseDir'];
        $orderNumber = str_replace('№ ', '', $_POST['orderNumber']);

        $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            if (empty($orderNumber)) {
                throw new \Exception($this->lang['VTBPAY_ORDER_NUMBER_IS_EMPTY']);
            }

            $order = $this->getOrderByNumber($orderNumber);
            if (!$order) {
                throw new \Exception($this->lang['VTBPAY_ORDER_NOT_FOUND']);
            }

            $paymentId = $paymentId ?: $order['payment_id'];

            $total = (float) $this->getOrderTotal($order);

            $returnUrl = $mgBaseDir . '/ajaxrequest?mguniqueurl=action/notification&pluginHandler=' .
                         self::$pluginName . '&payment=' .
                         $paymentId;

            $this->setPaymentParams($paymentId);

            $result = $this->getPayUrl(
                $order,
                $total,
                $returnUrl
            );

            return $this->data['result'] = $result;

        } catch (Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine()
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Log the caught exception for debugging purposes.
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s;',
                $e->getMessage(),
                $orderNumber
            ), $context);

            return false;
        }
    }


    /**
     * Получает URL-адрес платежа для перенаправления.
     *
     * @return string URL-адрес платежа.
     */

    private function getPayUrl($order, $total, $returnUrl)
    {
        $paymentParams = $this->paymentParams;

        $enableFiscal = (bool) ($paymentParams['enable_fiscal'] ?? false);
        $items = [];
        if ($enableFiscal) $items = $this->getItems($order);

        $email = $this->getCustomerEmail($order);

        $response = $this->getVtbApi()->getOrderLink(
            $order['number'],
            $email,
            time(),
            $total,
            $returnUrl,
            false,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        $orderCode = $response['object']['orderCode'] ?? '';
        if (!empty($orderCode)) $this->addTransaction($order['id'], $orderCode);

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает email клиента, иначе выкидывает ошибку
     */
    private function getCustomerEmail($order): string
    {
        // Get Email
        $email = $order['user_email'] ?: $this->paymentParams['email_fiscal'] ?? '';
        if (!self::validateEmail($email)) {
            throw new VtbPayException($this->lang['VTBPAY_EMAIL_NOT_VALID'], [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($localPart, $domainPart) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $localLength = strlen($localPart);
        if ($localLength < 1 || $localLength > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domainLength = strlen($domainPart);
        if ($domainLength < 1 || $domainLength > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowedTlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowedTlds) && !empty($allowedTlds[0])) {
            // Извлечение TLD из доменной части
            $domainParts = explode('.', $domainPart);
            $tld = strtolower(end($domainParts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowedTlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * notification
     *
     * Receives a notification of a payment event, validates it and processes accordingly.
     * Redirects to success or failure URLs based on the payment status.
     *
     * @return void
     */
    public function notification()
    {
        $paymentId = $_GET['payment'];
        $orderNumber = $_GET['orderId'];
        $this->data['result'] = $orderNumber;

        $this->logger->setOption('additionalCommonText', 'return-' . rand(1111, 9999));

        try {
            if (empty($orderNumber)) {
                throw new \Exception($this->lang['VTBPAY_ORDER_NUMBER_IS_EMPTY']);
            }

            $order = $this->getOrderByNumber($orderNumber);
            if (!$order) {
                throw new \Exception($this->lang['VTBPAY_ORDER_NOT_FOUND']);
            }

            $operationId = $this->getOperationId($order['id']);
            if (!$operationId) {
                throw new \Exception($this->lang['VTBPAY_OPERATION_NOT_FOUND']);
            }

            $this->setPaymentParams($paymentId);

            $paymentStatusData = $this->getPaymentStatusData($orderNumber);

            if ($paymentStatusData['object']['status']['value'] !== 'PAID') {
                throw new \Exception($this->lang['VTBPAY_PAYMENT_NOT_FOUND']);
            }

            $payment = new Controllers_Payment();
            $payment->actionWhenPayment([
                'paymentOrderId' => $order['id'],
                'paymentAmount' => $paymentStatusData['object']['amount']['value'],
                'paymentID' => $paymentId
            ]);

            $redirectUrlGetParams = '&payStatus=success&id=' . $paymentId .
                '&orderId=' . $orderNumber;

        } catch (Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Log the caught exception for debugging purposes.
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s;',
                $e->getMessage(),
                $orderNumber ?: ''
            ), $context);

            $redirectUrlGetParams = '&payStatus=fail';
            if (
                !empty($paymentId) &&
                !empty($orderNumber)
            ) {
                $redirectUrlGetParams = '&payStatus=fail&id=' . $paymentId .
                    '&orderId=' . $orderNumber;
            }
        }

        MG::redirect('/payment?plugin_name=' . self::$pluginName . $redirectUrlGetParams);
    }


    /**
     * getPaymentStatus() method requests the payment status from the VtbApi
     * for a specific order and returns it, throwing an exception if no status is received.
     *
     * @param string $orderNumber Order number
     * @return array Returns payment status data
     */
    public function getPaymentStatusData($orderNumber)
    {
        $response = $this->getVtbApi()->getOrderInfo($orderNumber);

        return $response ?: [];
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */

    private function getItems($order): array
    {
        $getAllItems = unserialize(stripslashes($order['order_content']));
        $paymentParams = $this->paymentParams;

        $items = [];
        foreach ($getAllItems as $key => $item) {
            $items[] = [
                'positionId' => ($key + 1),
                'name' => $item['name'],
                'code' => $item['code'] ?? (string) $item['id'],
                'price' => floatval($item['fulPrice']),
                'measure' => (int) $paymentParams['measure_fiscal'] ?? 0,
                'quantity' => (int) $item['count'],
                'taxParams' => [
                    'taxType' => $paymentParams['tax_type_fiscal'] ?? 'none'
                ],
                'paymentType' => $paymentParams['payment_type_fiscal'] ?? 'full_prepayment',
                'paymentSubject' => (int) $paymentParams['payment_subject_fiscal'] ?? 1,
                'amount' => floatval($item['fulPrice'] * $item['count'])
            ];
        }

        $deliveryCost = floatval($order['delivery_cost'] ?? 0);
        // Стоимость доставки
        if ($deliveryCost > 0) {
            $items[] = [
                'positionId' => ($key + 2),
                'name' => 'Доставка',
                'price' => $deliveryCost,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $paymentParams['tax_type_fiscal'] ?? 'none'
                ],
                'paymentType' => $paymentParams['payment_type_delivery_fiscal'] ?? 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $deliveryCost
            ];
        }

        return $items;
    }


    /**
     * getPaymentData
     *
     * Fetches the payment information from the database using the provided payment ID.
     *
     * @param int $paymentId The ID of the payment to fetch.
     * @return array An associative array of the payment data.
     */
    private function getPaymentData($paymentId)
    {
        $dbRes = DB::query("SELECT *
									          FROM `{$this->prefix}payment`
									          WHERE `id` = '{$paymentId}'");
        return DB::fetchArray($dbRes);
    }


    /**
     * getOrder
     *
     * Fetches an order from the database using the provided order ID.
     *
     * @param int $orderId The ID of the order to fetch.
     * @return array An associative array of the order data.
     */
    private function getOrder($orderId)
    {
        $dbRes = DB::query("SELECT *
									          FROM `{$this->prefix}order`
									          WHERE `id` = '{$orderId}'");
        return DB::fetchAssoc($dbRes);
    }


    /**
     * getOrderByNumber
     *
     * Fetches an order from the database using the provided order number.
     *
     * @param int $orderNumber The number of the order to fetch.
     * @return array An associative array of the order data.
     */
    private function getOrderByNumber($orderNumber)
    {
        $dbRes = DB::query("SELECT *
									          FROM `{$this->prefix}order`
									          WHERE `number` = '{$orderNumber}'");
        return DB::fetchAssoc($dbRes);
    }


    /**
     * setPaymentParams
     *
     * Sets the class-level payment parameters variable using the provided payment ID.
     *
     * @param int $paymentId The ID of the payment.
     * @return void
     */
    private function setPaymentParams($paymentId)
    {
        $this->paymentParams = Models_Payment::getPaymentParams(self::$pluginName, true);
    }


    /**
     * getOrderTotal
     *
     * Calculates and returns the total cost of an order, including delivery cost if available.
     *
     * @param array $order The order to calculate the total for.
     * @return float The total cost of the order.
     */
    private function getOrderTotal($order)
    {
        if (
            isset($order['delivery_cost']) &&
            $order['delivery_cost'] > 0
        ) {
            $summ = $order['summ'] + $order['delivery_cost'];
        }
        else {
            $summ = $order['summ'];
        }

        return $summ;
    }


    /**
     * addTransaction
     *
     * Adds a new transaction to the database with the provided order ID and operation ID.
     *
     * @param int $orderNumber The number of the order associated with the transaction.
     * @param int $operationId The ID of the operation associated with the transaction.
     * @return void
     */
    private function addTransaction($id, $operationId)
    {
        $sql = "INSERT INTO `{$this->prefix}vtbpay_transactions`
								VALUES ('{$id}', '{$operationId}', CURRENT_TIMESTAMP);";
        DB::query($sql);
    }


    /**
     * getOperationId
     *
     * Fetches the operation ID associated with the provided order ID from the database.
     *
     * @param int $orderNumber The number of the order to fetch the operation for.
     * @return string The operation ID associated with the order.
     */
    private function getOperationId($id)
    {
        $dbRes = DB::query("SELECT *
			          FROM `{$this->prefix}vtbpay_transactions`
			          WHERE `id` = '{$id}'
								ORDER BY `date` DESC");
        $result = DB::fetchAssoc($dbRes);

        return $result['transaction_id'];
    }


    /**
     * Устанавливает объект логирования для модуля.
     */
    private function setVtbpayLogger()
    {
        $sensitiveDataKeys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $this->logger = VtbPayLogger::getInstance()
            ->setOption('showCurrentDate', false)
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitiveDataKeys)
            ->setCustomRecording(function($message) {
                MG::loger($message,'append','vtbpay');
            }, VtbPayLogger::LOG_LEVEL_ERROR)
            ->setCustomRecording(function($message) {
                $logging = (bool) ($this->paymentParams['logging'] ?? false);
                if ($logging) MG::loger($message,'append','vtbpay');
            }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
