<?php

namespace Vtb\VtbPay\Model;

use Exception,
    Magento\Payment\Model\Method\AbstractMethod,
    Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface,
    Magento\Sales\Model\Order,
    Magento\Framework\App\Config\ScopeConfigInterface,
    Magento\Quote\Api\Data\CartInterface,
    Magento\Framework\Exception\LocalizedException,
    Magento\Framework\Model\Context,
    Magento\Framework\Registry,
    Magento\Framework\Api\ExtensionAttributesFactory,
    Magento\Framework\Api\AttributeValueFactory,
    Magento\Payment\Helper\Data,
    Magento\Payment\Model\Method\Logger,
    Magento\Checkout\Model\Session,
    Magento\Sales\Model\OrderFactory,
    Magento\Framework\UrlInterface,
    Vtb\VtbPay\Gateway\Http\Client\Api\VtbApi,
    Vtb\VtbPay\Gateway\Http\Client\Exception\VtbPayException,
    Symfony\Component\Dotenv\Dotenv;


/**
 * VtbPay Payment Method
 *
 * This class extends Magento's AbstractMethod to provide VtbPay as a payment method.
 * It defines several properties and methods required by the Magento payment method system, including code, form block type,
 * and info block type.
 * It also interacts with the VtbApi to get and provide order information to and from the VtbPay system.
 *
 * @package Vtb\VtbPay\Model
 */
class VtbPay extends AbstractMethod
{
    /**
     * These boolean variables define the capabilities of the VtbPay payment method.
     * It declares the method as a gateway method, capable of order, authorize, capture, and use for internal transactions
     * and checkout, but not capable of refunds or multi-shipping transactions.
     */
    protected $_isGateway = true;
    protected $_isOffline = false;
    protected $_canOrder = true;
    protected $_canAuthorize = true;
    protected $_canCapture = true;
    protected $_canRefund = false;
    protected $_canUseInternal = true;
    protected $_canUseCheckout = true;
    protected $_canUseForMultishipping = false;
    protected $_isInitializeNeeded = true;

    /**
     * @var string Payment method code and XML path for VtbPay configuration data.
     */
    const PAYMENT_METHOD_VTBPAY_CODE = 'vtb_vtbpay';
    const PAYMENT_METHOD_VTBPAY_XML_PATH = 'payment/vtb_vtbpay/';
    const CMS_NAME = 'Magento 2';
    const PLUGIN_VERSION = '1.5.4';

    /**
     * @var string Payment method code for this module.
     */
    protected $_code = self::PAYMENT_METHOD_VTBPAY_CODE;

    /**
     * @var string Path to the template for the VtbPay payment information block.
     */
    protected $_infoBlockType = \Magento\Payment\Block\Info\Instructions::class;

    /**
     * @var \Vtb\VtbPay\Model\VtbApi Reference to the VtbApi object that performs interactions with the VtbPay system.
     */
    public $vtbApi;

    /**
     * @var Session
     */
    protected $checkoutSession;

    /**
     * @var OrderFactory
     */
    protected $orderFactory;

    /**
     * @var BuilderInterface
     */
    protected $transactionBuilder;

    /**
     * @var UrlInterface
     */
    protected $_urlBuilder;

    protected $_scopeConfig;


    /**
     * The constructor sets up the necessary dependencies for the class.
     *
     * @param Context $context Application context object
     * @param Registry $registry Registry object
     * @param ExtensionAttributesFactory $extensionFactory Extension attributes factory object
     * @param AttributeValueFactory $customAttributeFactory Custom attribute factory object
     * @param Data $paymentData Payment helper object
     * @param ScopeConfigInterface $scopeConfig Scope configuration object
     * @param Logger $logger Logger object
     * @param Session $checkoutSession Checkout session object
     * @param OrderFactory $orderFactory Order factory object
     * @param BuilderInterface $transactionBuilder Transaction builder object
     * @param UrlInterface $urlBuilder URL builder object
     */
    public function __construct(
        Context $context,
        Registry $registry,
        ExtensionAttributesFactory $extensionFactory,
        AttributeValueFactory $customAttributeFactory,
        Data $paymentData,
        ScopeConfigInterface $scopeConfig,
        Logger $logger,
        Session $checkoutSession,
        OrderFactory $orderFactory,
        BuilderInterface $transactionBuilder,
        UrlInterface $urlBuilder
    ) {
        $this->_checkoutSession = $checkoutSession;
        $this->_orderFactory = $orderFactory;
        $this->_transactionBuilder = $transactionBuilder;
        $this->_urlBuilder = $urlBuilder;
        $this->_scopeConfig = $scopeConfig;

        $env_path = dirname(__DIR__) . '/config/.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);

        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger
        );
    }


    /**
     * setVtbApi() initializes the VtbApi object with configuration data.
     */
    public function setVtbApi()
    {
        $arFields = [
            'client_id',
            'client_secret',
            'test_mode',
            'merchant_authorization'
        ];

        $arFieldsData = [];

        foreach ($arFields as $arField) {
            $arFieldsData[$arField] = $this->getPaymentSettings($arField);
        }

        $this->vtbApi = new VtbApi(
            $arFieldsData['client_id'],
            $arFieldsData['client_secret'],
            (bool) $arFieldsData['test_mode'],
            $arFieldsData['merchant_authorization'] ?: ''
        );
    }


    /**
     * initialize() method sets the initial order status and state.
     *
     * @param string $paymentAction Payment action string
     * @param \Magento\Framework\DataObject $stateObject State object
     * @return PaymentMethod Returns this class instance.
     */
    public function initialize($paymentAction, $stateObject): PaymentMethod
    {
        $stateObject->setStatus('new');
        $stateObject->setState(Order::STATE_PENDING_PAYMENT);
        $stateObject->setIsNotified(false);

        return $this;
    }


    /**
     * getCheckoutRedirect() method requests a checkout URL from the VtbApi
     * and returns it, throwing an exception if no URL is received.
     *
     * @param Order $order Order object
     * @return string Returns checkout URL string.
     * @throws LocalizedException Throws exception when failed to get payment URL.
     */
    public function getCheckoutRedirect($order)
    {
        $orderId = $order->getId();

        $items = [];
        $twoStage = (bool) ($this->getPaymentSettings('two_stage') ?: false);
        $fiscalEnable = (bool) ($this->getPaymentSettings('fiscal_enable') ?: false);
        if (!$twoStage && $fiscalEnable) $items = $this->getItems($order);

        $email = $this->getCustomerEmail($order);

        $response = $this->vtbApi->getOrderLink(
            $orderId,
            $email,
            time(),
            $order->getGrandTotal(),
            $this->getReturnUrl(),
            $twoStage,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получение Email покупателя
     *
     * @param Order $order Order object
     *
     * @return string Email клиента
     *
     * @throws VtbPayException If the invoice number is invalid
     */
    private function getCustomerEmail($order): string
    {
        $fiscalEmail = $this->getPaymentSettings('fiscal_email');
        $email = $order->getCustomerEmail() ?: $fiscalEmail;
        if (!self::validateEmail($email)) {
            throw new VtbPayException(__('The email address provided has not been validated.'), [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Получает массив товаров заказа.
     *
     * @param Order $order Order object
     * @return array Массив товаров.
     */
    private function getItems($order): array
    {
        // Настройки по умолчанию для фискальных параметров
        $fiscalSettings = [
            'fiscal_measure' => 0,
            'fiscal_tax' => 'none',
            'fiscal_payment_method' => 'full_prepayment',
            'fiscal_subject' => 1
        ];
        $items = [];

        foreach ($order->getAllItems() as $key => $item) {
            if (isset($item) && $item->getPrice() > 0) {
                $product = $item->getProduct();

                $productFiscal = [];
                // Получение фискальных параметров для каждого продукта
                foreach ($fiscalSettings as $name => $default) {
                    $productFiscal[$name] = $product->getData('payment_vtbpay_' . $name) ?: '-';
                    if ($productFiscal[$name] === '-') $productFiscal[$name] = $this->getPaymentSettings($name) ?: $default;
                }

                $items[] = [
                    'positionId' => ($key + 1),
                    'name' => $item->getName(),
                    'code' => $item->getSku() ?: (string) $item->getProductId(), // 13/06/24 на случай если у товара нет артикула передать ID в виде строки
                    'price' => floatval($item->getPrice()),
                    'measure' => (int) $productFiscal['fiscal_measure'],
                    'quantity' => (int) $item->getQtyOrdered(),
                    'taxParams' => [
                        'taxType' => $productFiscal['fiscal_tax']
                    ],
                    'paymentType' => $productFiscal['fiscal_payment_method'],
                    'paymentSubject' => (int) $productFiscal['fiscal_subject'],
                    'amount' => floatval($item->getPrice() * $item->getQtyOrdered())
                ];
            }
        }

        // Стоимость доставки
        $shippingAmount = floatval($order->getShippingAmount() ?: 0);
        if ($shippingAmount > 0) {
            $fiscalTax = $this->getPaymentSettings('fiscal_tax');
            $fiscalDeliveryPaymentMethod = $this->getPaymentSettings('fiscal_delivery_payment_method');

            $items[] = [
                'positionId' => ($key + 2),
                'name' => 'Доставка',
                'price' => $shippingAmount,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $fiscalTax ?: 'none'
                ],
                'paymentType' => $fiscalDeliveryPaymentMethod ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $shippingAmount
            ];
        }

        return $items;
    }


    /**
     * getReturnUrl() method returns the URL that the VtbPay gateway should redirect
     * the customer to after they complete their payment.
     *
     * @return string Returns URL string.
     */
    protected function getReturnUrl()
    {
        // You might need to adjust the exact URL depending on your module structure.
        return $this->_urlBuilder->getUrl('vtbpay/payment/handleResponse', [
            '_secure' => true,          // Устанавливаем HTTPS
            '_query' =>  [              // Добавляем GET-параметр request_type
                'request_type' => 'return'
            ]
        ]);
    }


    /**
     * getPaymentStatus() method requests the payment status from the VtbApi
     * for a specific order and returns it, throwing an exception if no status is received.
     *
     * @param Order $order Order object
     * @return string Returns payment status string.
     * @throws LocalizedException Throws exception when no information about payment status.
     */
    public function getPaymentStatus($order)
    {
        $orderId = $order->getId();

        $response = $this->vtbApi->getOrderInfo($orderId);

        return $response['object']['status']['value'] ?? '';
    }


    /**
     * isAvailable() method checks if the payment method is available for use
     * based on whether or not it is active in the configuration.
     *
     * @param CartInterface|null $quote Cart quote object
     * @return bool Returns true if the payment method is active, otherwise false.
     */
    public function isAvailable(CartInterface $quote = null): bool
    {
        if (!$this->isActive($quote ? $quote->getStoreId() : null)) {
            return false;
        }
        return true;
    }


    /**
     * getInstructions() method fetches the instruction text configured for the payment method.
     *
     * @return string Returns instruction text string.
     */
    public function getInstructions()
    {
        $instructions = $this->getConfigData('instructions');
        return !empty($instructions) ? trim($instructions) : '';
    }


    /**
     * Получает настройки платёжной системы
     *
     * @param string $settingsName Название настройки
     * @return mixed
     */
    public function getPaymentSettings($settingsName)
    {
        return $this->_scopeConfig->getValue(
            self::PAYMENT_METHOD_VTBPAY_XML_PATH . 'payment_vtbpay_' . $settingsName,
            ScopeConfigInterface::SCOPE_TYPE_DEFAULT
        );
    }
}
