<?php

namespace Vtb\VtbPay\Controller\Payment;

use Exception,
    Magento\Checkout\Model\Session,
    Magento\Framework\App\Action\Action,
    Magento\Framework\App\Action\Context,
    Magento\Framework\Controller\Result\JsonFactory,
    Magento\Framework\Exception\LocalizedException,
    Magento\Framework\Message\ManagerInterface,
    Magento\Sales\Api\OrderRepositoryInterface,
    Magento\Sales\Model\Order,
    Vtb\VtbPay\Helper\VtbPayLogger,
    Vtb\VtbPay\Gateway\Http\Client\Exception\VtbPayException;


/**
 * Class Redirect
 *
 * This class is used to handle the redirection of the customer to the VTB payment system.
 * After a customer confirms their order and selects VTB payment method, they are redirected to this controller.
 * This controller then fetches the order based on the checkout session and retrieves the redirect URL from the payment method.
 * The order's status is updated to pending payment, and the customer is then redirected to the VTB payment system for payment.
 * If an error occurs, the customer's quote is restored, an error message is displayed, and the customer is redirected back to the checkout page.
 *
 * @package Vtb\VtbPay\Controller\Payment
 */
class Redirect extends Action
{
    /**
     * @var Session
     */
    protected $checkoutSession;

    /**
     * @var ManagerInterface
     */
    protected $messageManager;

    /**
     * @var JsonFactory
     */
    protected $jsonFactory;

    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;

    /**
     * @var VtbPayLogger
     */
    protected $vtbPayLogger;


    /**
     * Redirect constructor.
     *
     * @param Context $context
     * @param Session $checkoutSession
     * @param ManagerInterface $messageManager
     * @param JsonFactory $jsonFactory
     * @param OrderRepositoryInterface $orderRepository
     * @param VtbPayLogger $vtbPayLogger
     */
    public function __construct(
        Context $context,
        Session $checkoutSession,
        ManagerInterface $messageManager,
        JsonFactory $jsonFactory,
        OrderRepositoryInterface $orderRepository,
        VtbPayLogger $vtbPayLogger
    ) {
        parent::__construct($context);

        $this->checkoutSession = $checkoutSession;
        $this->messageManager = $messageManager;
        $this->jsonFactory = $jsonFactory;
        $this->orderRepository = $orderRepository;
        $this->vtbPayLogger = $vtbPayLogger;
    }


    /**
     * Executes when a user clicks on the "proceed to payment" button.
     *
     * This method is responsible for handling the redirection of the customer to the VTB payment system.
     * It fetches the order, retrieves the redirect URL from the payment method, updates the order status to pending payment,
     * and redirects the customer to the VTB payment system. If an error occurs, the customer is redirected back to the checkout page.
     *
     * @return ResponseInterface|Json|ResultInterface
     */
    public function execute()
    {
        $session = $this->checkoutSession;
        /**
         * @var Order
         */
        $order = $session->getLastRealOrder();
        $orderId = $order->getId();

        $this->vtbPayLogger->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            if (empty($orderId)) {
                throw new LocalizedException(__('This order does not exist.'));
            }

            /**
             * @var VtbPay
             */
            $vtbpay = $order->getPayment()->getMethodInstance();
            $vtbpay->setVtbApi();
            $url = $vtbpay->getCheckoutRedirect($order);

            $order->setStatus(Order::STATE_PENDING_PAYMENT);
            $order->setState(Order::STATE_PENDING_PAYMENT);
            $this->orderRepository->save($order);

            $result = $this->jsonFactory->create();

            return $result->setData([
                'url' => $url,
            ]);
        } catch (Exception | LocalizedException | VtbPayException $e) {

            $session->restoreQuote();

            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->vtbPayLogger->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s',
                $e->getMessage(),
                $orderId ?: ''
            ), $context);

            $this->messageManager->addErrorMessage($e->getMessage());

            $resultRedirect = $this->resultRedirectFactory->create();
            $resultRedirect->setUrl($this->_redirect->getRefererUrl());

            return $resultRedirect;
        }
    }
}
