<?php

namespace Vtb\VtbPay\Controller\Payment;

use Exception,
    Magento\Framework\App\Action\Action,
    Magento\Framework\App\Action\Context,
    Magento\Framework\App\CsrfAwareActionInterface,
    Magento\Framework\App\RequestInterface,
    Magento\Framework\App\Request\InvalidRequestException,
    Magento\Framework\App\Config\ScopeConfigInterface,
    Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface as TransactionBuilder,
    Magento\Customer\Model\Session as CustomerSession,
    Magento\Sales\Api\OrderRepositoryInterface,
    Magento\Framework\Exception\LocalizedException,
    Magento\Sales\Model\Order\Payment\Transaction,
    Magento\Sales\Model\Order,
    Magento\Framework\Controller\Result\JsonFactory,
    Vtb\VtbPay\Model\VtbPay,
    Vtb\VtbPay\Helper\VtbPayLogger,
    Vtb\VtbPay\Gateway\Http\Client\Exception\VtbPayException;

class HandleResponse extends Action implements CsrfAwareActionInterface
{
    /**
     * @var ScopeConfigInterface $scopeConfig
     * Интерфейс для получения настроек конфигурации из области видимости.
     */
    protected $scopeConfig;

    /**
     * @var TransactionBuilder $transactionBuilder
     * Построитель для создания транзакций оплаты.
     */
    protected $transactionBuilder;

    /**
     * @var OrderRepositoryInterface $orderRepository
     * Репозиторий для работы с заказами (CRUD операции).
     */
    protected $orderRepository;

    /**
     * @var CustomerSession $customerSession
     * Сессия покупателя для работы с текущим пользователем.
     */
    protected $customerSession;

    /**
     * @var VtbPayLogger $vtbPayLogger
     * Логгер для записи операций VtbPay.
     */
    protected $vtbPayLogger;

    /**
     * @var JsonFactory $jsonFactory
     * Фабрика для создания JSON-ответов.
     */
    protected $jsonFactory;

    /**
     * @var Order $order
     * Текущий объект заказа, который будет использоваться в методах.
     */
    private $order;

    /**
     * @var VtbPay $vtbpay
     * Инстанс платёжного метода VtbPay.
     */
    private $vtbpay;

    /**
     * Конструктор
     *
     * @param Context $context
     * @param ScopeConfigInterface $scopeConfig
     * @param TransactionBuilder $transactionBuilder
     * @param OrderRepositoryInterface $orderRepository
     * @param CustomerSession $customerSession
     * @param JsonFactory $jsonFactory
     * @param VtbPayLogger $vtbPayLogger
     */
    public function __construct(
        Context $context,
        ScopeConfigInterface $scopeConfig,
        TransactionBuilder $transactionBuilder,
        OrderRepositoryInterface $orderRepository,
        CustomerSession $customerSession,
        JsonFactory $jsonFactory,
        VtbPayLogger $vtbPayLogger
    ) {
        parent::__construct($context);
        $this->scopeConfig = $scopeConfig;
        $this->transactionBuilder = $transactionBuilder;
        $this->orderRepository = $orderRepository;
        $this->customerSession = $customerSession;
        $this->jsonFactory = $jsonFactory;
        $this->vtbPayLogger = $vtbPayLogger;
    }


    /**
     * @param RequestInterface $request
     *
     * @return bool|null
     */
    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }


    /**
     * @param RequestInterface $request
     *
     * @return InvalidRequestException|null
     */
    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }


    /**
     * Основной метод для обработки webhook или возврата после оплаты.
     *
     * @return \Magento\Framework\Controller\Result\Redirect|\Magento\Framework\Controller\Result\Json
     */
    public function execute()
    {
        // Получаем тип запроса: 'return' или 'hook'
        $mode = $this->getRequest()->getParam('request_type');

        if (!in_array($mode, ['return', 'webhook'])) return;

        // Настройка логгера
        $this->vtbPayLogger->logger->setOption('additionalCommonText', $mode . '-' . rand(1111, 9999));

        // Получение данных запроса в зависимости от типа
        $requestData = ($mode === 'return') ? $this->getRequest()->getParams() : json_decode(file_get_contents('php://input'), true);
        $this->vtbPayLogger->logger->debug(
            __FUNCTION__ . ' > ' . $mode . ': ', [
            'requestData' => $requestData
        ]);

        // Извлечение orderId
        $orderId = $requestData['orderId'] ?? $requestData['object']['orderId'] ?? '';

        try {
            if (empty($orderId)) {
                throw new LocalizedException(__('Order ID not found.'));
            }

            // Загрузка заказа
            $this->order = $this->orderRepository->get($orderId);
            if (!$this->order) {
                throw new LocalizedException(__('This order does not exist.'));
            }

            // Получение платёжного метода
            $payment = $this->order->getPayment();
            $this->vtbpay = $payment->getMethodInstance();
            if ($this->vtbpay->getCode() !== VtbPay::PAYMENT_METHOD_VTBPAY_CODE) {
                throw new LocalizedException(__('Unknown payment method.'));
            }

            $this->vtbpay->setVtbApi();

            // Получение статуса оплаты
            $paymentStatus = $this->vtbpay->getPaymentStatus($this->order);

            // Проверка и смена статуса заказа
            $this->changePaymentStatus($paymentStatus);

            // Обработка редиректа после успешной оплаты
            if ($mode === 'return') {
                return $this->handleReturnRedirect();
            }

            // Возвращаем JSON ответ для webhook
            return $this->jsonFactory->create()->setData(['status' => 'OK']);

        } catch (Exception | LocalizedException | VtbPayException $e) {
            // Обработка ошибок
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Логирование ошибки
            $this->vtbPayLogger->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s',
                $e->getMessage(),
                $orderId ?: ''
            ), $context);

            if ($mode === 'return') {
                $this->messageManager->addErrorMessage(__('An error occurred.') . ' ' . $e->getMessage());
                return $this->resultRedirectFactory->create()->setPath('checkout/cart');
            }

            return $this->jsonFactory->create()->setData(['error' => $e->getMessage()]);
        }
    }


    /**
     * Смена статуса заказа в зависимости от статуса оплаты.
     *
     * @param string $paymentStatus Статус оплаты
     * @return void
     * @throws LocalizedException
     */
    private function changePaymentStatus(string $paymentStatus)
    {
        $availableStatuses = [
            'PAID' => function() {
                $this->completeOrder();
            },
            'PENDING' => function() {
                $this->holdOrder();
            },
            'REFUNDED' => function() {
                $this->refundOrder();
            }
        ];

        $orderStatusFunction = $availableStatuses[$paymentStatus] ?? function() {
            $this->cancelOrder();
        };

        $orderStatusFunction();
        $this->orderRepository->save($this->order);
    }


    /**
     * Завершает заказ и создаёт транзакцию.
     */
    private function completeOrder()
    {
        if ($this->order->getStatus() !== $this->vtbpay->getPaymentSettings('transaction_paid_status')) {
            $this->order->setTotalPaid($this->order->getGrandTotal());
            $this->order->setBaseTotalPaid($this->order->getBaseGrandTotal());

            $this->order->setState(\Magento\Sales\Model\Order::STATE_COMPLETE)
                        ->setStatus($this->vtbpay->getPaymentSettings('transaction_paid_status'));

            $this->order->addStatusHistoryComment(__('Payment was successful for order #') . $this->order->getId());
            $this->messageManager->addSuccessMessage(__('Payment completed successfully.'));

            // Создаём транзакцию
            $this->createTransaction();
        }
    }


    /**
     * Ставит заказ в ожидание.
     */
    private function holdOrder()
    {
        if ($this->order->getStatus() !== $this->vtbpay->getPaymentSettings('transaction_holded_status')) {
            $this->order->hold();
            $this->order->setStatus($this->vtbpay->getPaymentSettings('transaction_holded_status'));
            $this->order->addStatusHistoryComment(__('Payment received but not confirmed.'));
            $this->messageManager->addNoticeMessage(__('Payment completed successfully.'));
        }
    }


    /**
     * Возвращает заказ.
     */
    private function refundOrder()
    {
        if ($this->order->getStatus() !== $this->vtbpay->getPaymentSettings('transaction_refunded_status')) {
            $this->order->setBaseTotalRefunded($this->order->getBaseGrandTotal());
            $this->order->setTotalRefunded($this->order->getGrandTotal());

            $this->order->setState(\Magento\Sales\Model\Order::STATE_CLOSED)
                        ->setStatus($this->vtbpay->getPaymentSettings('transaction_refunded_status'));

            $this->order->addStatusHistoryComment(__('Refund of payment.'));
            $this->messageManager->addSuccessMessage(__('The payment was refunded.'));
        }
    }


    /**
     * Отменяет заказ.
     */
    private function cancelOrder()
    {
        if ($this->order->getStatus() !== $this->vtbpay->getPaymentSettings('transaction_canceled_status')) {
            $this->order->cancel();
            $this->order->setStatus($this->vtbpay->getPaymentSettings('transaction_canceled_status'));
            $this->order->addStatusHistoryComment(__('Payment not paid.'));
            $this->messageManager->addErrorMessage(__('Payment not paid. Your order has been canceled.'));
        }
    }


    /**
     * Обработка редиректа после успешной оплаты.
     *
     * @return \Magento\Framework\Controller\Result\Redirect
     */
    private function handleReturnRedirect()
    {
       $pathRedirect = $this->customerSession->isLoggedIn()
           ? 'sales/order/view/order_id/' . $this->order->getId()
           : 'sales/guest/form/';
       return $this->resultRedirectFactory->create()->setPath($pathRedirect);
    }

    /**
     * Создаёт транзакцию типа "capture" для данного заказа и оплаты.
     *
     * @return void
     */
    private function createTransaction()
    {
        $payment = $this->order->getPayment();
        $transaction = $this->transactionBuilder
            ->setPayment($payment)
            ->setOrder($this->order)
            ->setTransactionId($this->order->getIncrementId())
            ->setAdditionalInformation(
                [Transaction::RAW_DETAILS => (array) $payment->getAdditionalInformation()]
            )
            ->setFailSafe(true)
            ->build(Transaction::TYPE_CAPTURE);

        // Добавляем комментарий к заказу о том, что транзакция была захвачена
        $payment->addTransactionCommentsToOrder(
            $transaction,
            __('The payment transaction has been captured.')
        );
    }
}
