<?php

$newBasePaymentHandler = dirname(__FILE__, 3) . '/handlers/mspaymenthandler.class.php';
$oldBasePaymentHandler = dirname(__FILE__, 3) . '/model/minishop2/mspaymenthandler.class.php';

if (!class_exists('msPaymentInterface')) {
    if (file_exists($newBasePaymentHandler)) {
        require_once $newBasePaymentHandler;
    } else {
        require_once $oldBasePaymentHandler;
    }
}

require_once MODX_BASE_PATH . '/mspVtbPay/vendor/autoload.php';

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;

class VtbPay extends msPaymentHandler implements msPaymentInterface
{
    const CMS_NAME = 'MODX Revolution';
    const PLUGIN_VERSION = '1.5.5';

    private msOrder $order;
    public VtbPayLogger $logger;

    /**
     * VtbPay constructor.
     *
     * This method initiates the VtbPay object by merging the default configuration with the user defined configurations.
     *
     * @param xPDOObject $object The xPDOObject from the modx system.
     * @param array $config User defined configuration array.
     */
    public function __construct(xPDOObject $object, $config = [])
    {
        parent::__construct($object, $config);

        $env_path = MODX_BASE_PATH . '/mspVtbPay/config/.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);

        $resultConfig  = [];
        $settings = [
            'client_id',
            'client_secret',
            'merchant_authorization',
            'test_mode',
            'logging',
            'two_stage',
            'success_id',
            'failure_id',
            'enable_fiscal',
            'email_fiscal',
            'payment_type_delivery_fiscal',
            'measure_fiscal',
            'tax_type_fiscal',
            'payment_type_fiscal',
            'payment_subject_fiscal'
        ];

        foreach ($settings as $setting) {
            $resultConfig[$setting] = $this->modx->getOption('ms2_payment_vtbpay_' . $setting);
        }

        $this->config = array_merge($resultConfig, $config);

        $this->setVtbpayLogger();
    }


    /**
     * Send method.
     *
     * This method is used to create a new order and send it to the payment gateway.
     * It then logs the response and returns the payment URL to the user.
     *
     * @param msOrder $order An instance of the order that needs to be sent to the gateway.
     * @return array|string Either the URL of the payment gateway or an error message.
     */
    public function send(msOrder $order)
    {
        $this->order = $order;

        $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            $statusNew = $this->modx->getOption('ms2_status_new', null, 1) ?: 1;
            if ($this->order->get('status') > $statusNew) {
                throw new \Exception($this->modx->lexicon('ms2_err_status_wrong'));
            }

            $payUrl = $this->getPayUrl();
            $this->setOrderProperty('payment_link', $payUrl);
            return $this->success('', [
                'redirect' => $payUrl,
                'msorder' => $this->order->get('id')
            ]);

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Handle exception and log error
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $this->order->get('id') ?: ''
            ), $context);

            // Set parameters and context
            $params['msorder'] = (int) $this->order->get('id');

            // Get failure url page
            if ($id = $this->modx->getOption('ms2_payment_vtbpay_failure_id', null, 0)) {
                $failure = $this->modx->makeUrl(
                    $id,
                    $this->order->get('context'),
                    $params,
                    'full'
                );
            }

            return $this->error($e->getMessage(), [
                'redirect' => $failure
            ]);
        }
    }


    /**
     * Получает URL-адрес платежа для перенаправления.
     *
     * @return string URL-адрес платежа.
     */
    private function getPayUrl(): string
    {
        $email = $this->getCustomerEmail();
        $cost = str_replace(',', '.', $this->order->get('cost'));

        // Get return url
        $siteUrl = $this->modx->getOption('site_url');
        $assetsUrl = $this->modx->getOption('assets_url') . 'components/minishop2/';
        $returnUrl = $siteUrl . substr($assetsUrl, 1) . 'payment/vtbpay.php?mode=return';

        // настройка в админке двустадийной оплаты
        $twoStage = (bool) ($this->config['two_stage'] ?? false);
        $enableFiscal = (bool) ($this->config['enable_fiscal'] ?? false);

        $items = [];
        if (!$twoStage && $enableFiscal) $items = $this->getItems();

        $response = $this->getVtbApi()->getOrderLink(
            $this->order->get('id'),
            $email,
            time(),
            $cost,
            $returnUrl,
            $twoStage,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает email клиента, иначе выкидывает ошибку
     */
    private function getCustomerEmail()
    {
        $profile = $this->order->getOne('UserProfile');
        $email = $profile->get('email') ?: $this->config['email_fiscal'] ?? '';
        if (!self::validateEmail($email)) {
            throw new VtbPayException($this->modx->lexicon('ms2_payment_vtbpay_email_not_valid'), [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($localPart, $domainPart) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $localLength = strlen($localPart);
        if ($localLength < 1 || $localLength > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domainLength = strlen($domainPart);
        if ($domainLength < 1 || $domainLength > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowedTlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowedTlds) && !empty($allowedTlds[0])) {
            // Извлечение TLD из доменной части
            $domainParts = explode('.', $domainPart);
            $tld = strtolower(end($domainParts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowedTlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Get order property.
     *
     * This method is used to retrieve a specific property of an order.
     *
     * @param msOrder    $order The order for which the property needs to be fetched.
     * @param string     $name Name of the property to be fetched.
     * @param mixed|null $default The default value to return if the property doesn't exist.
     * @return mixed|null The property value if found, else the default value.
     */
    private function getOrderProperty(msOrder $order, $name, $default = null)
    {
        $props = $order->get('properties');

        return $props['payments']['vtbpay'][$name] ?? $default;
    }


    /**
     * Set order property.
     *
     * This method is used to set a property for an order.
     * It handles both single and multiple property setting.
     *
     * @param string|array $name The name of the property or an array of properties to be set.
     * @param mixed|null   $value The value to be set for the property.
     */
    private function setOrderProperty($name, $value = null)
    {
        $newProperties = [];
        if (is_array($name)) {
            $newProperties = $name;
        } else {
            $newProperties[$name] = $value;
        }

        $orderProperties = $this->order->get('properties');

        if (isset($orderProperties['payments']['vtbpay'])) {
            $orderProperties['payments']['vtbpay'] = array_merge(
                $orderProperties['payments']['vtbpay'],
                $newProperties
            );
        }
        else {
            if (!is_array($orderProperties)) {
                $orderProperties = [];
            }
            $orderProperties['payments']['vtbpay'] = $newProperties;
        }

        $this->order->set('properties', $orderProperties);
        $this->order->save();
    }


    /**
     * Get payment link.
     *
     * This method is used to retrieve the payment link for an order.
     * Returns a direct link for continue payment process of existing order
     *
     * @param msOrder $order The order for which the payment link needs to be fetched.
     * @return string The URL of the payment gateway.
     */
    public function getPaymentLink(msOrder $order)
    {
        return $this->getOrderProperty($order, 'payment_link');
    }


    /**
     * Receive method.
     *
     * This method is used to receive the status of a payment from the payment gateway.
     * Depending on the response, it changes the status of the order accordingly.
     *
     * @param msOrder $order The order for which the payment status needs to be fetched.
     * @param array $params Any additional parameters needed.
     *
     * @return bool True if the payment status is 'PAID', False otherwise.
     */
    public function receive(msOrder $order)
    {
        $this->order = $order;

        try {
            $paymentStatus = $this->getPaymentStatus();
            $this->changePaymentStatus($paymentStatus);

            if (in_array($paymentStatus, ['PAID', 'PENDING', 'RECONCLIED'])) return true;

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Handle exception and log error
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $this->order->get('id') ?: ''
            ), $context);

            $status = $context['order_status'] ?? $this->modx->getOption('ms2_status_canceled', null, 4) ?: 4;

            // Set status "cancelled"
            $this->ms2->changeOrderStatus($this->order->get('id'), $status);
        }

        return false;
    }


    /**
     * Получение статуса заказа из VTB API.
     *
     * @return string Значение статуса заказа, если оно существует, или пустая строка в противном случае.
     */
    private function getPaymentStatus(): string
    {
        // Get order information from VtbApi.
        $response = $this->getVtbApi()->getOrderInfo($this->order->get('id'));

        return $response['object']['status']['value'] ?? '';
    }


    /**
     * Устанавливаем статус заказа
     *
     * @param string $paymentStatus Статус оплаты.
     *
     * @return void
     */
    private function changePaymentStatus(string $paymentStatus): void
    {
        $currentOrderStatus = $this->order->get('status');
        $paidStatus = (int) $this->modx->getOption('ms2_status_paid', null, 2) ?: 2;
        $availableStatuses = [
            'PAID' => [
                'order_status' => $paidStatus
            ],
            'PENDING' => [
                'order_status' => (int) $this->modx->getObject('msOrderStatus', ['rank' => 7777])->id
            ],
            'REFUNDED' => [
                'order_status' => (int) $this->modx->getObject('msOrderStatus', ['rank' => 8888])->id
            ]
        ];

        $orderStatusData = $availableStatuses[$paymentStatus] ?? [];

        if (!empty($orderStatusData)) {
            if ($currentOrderStatus !== $orderStatusData['order_status']) {
                $this->ms2->changeOrderStatus(
                    $this->order->get('id'),
                    $orderStatusData['order_status']
                );
            }
        }
        else {
            throw new \Exception($this->modx->lexicon('ms2_payment_vtbpay_unsuccessful_payment_message'));
        }
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems(): array
    {
        $items = [];
        $key = 1;

        foreach ($this->order->getMany('Products') as $item) {
            $items[] = [
                'positionId' => $key,
                'name' => $item->get('name'),
                'code' => (string) $item->get('product_id'),
                'price' => floatval($item->get('price')),
                'measure' => (int) $this->config['measure_fiscal'] ?: 0,
                'quantity' => (int) $item->get('count'),
                'taxParams' => [
                    'taxType' => $this->config['tax_type_fiscal'] ?: 'none'
                ],
                'paymentType' => $this->config['payment_type_fiscal'] ?: 'full_prepayment',
                'paymentSubject' => (int) $this->config['payment_subject_fiscal'] ?: 1,
                'amount' => floatval($item->get('cost'))
            ];

            $key++;
        }

        $deliveryCost = $this->order->get('delivery_cost');
        if ($deliveryCost > 0) {
            $items[] = [
                'positionId' => $key,
                'name' => $this->modx->lexicon('ms2_payment_vtbpay_delivery'),
                'price' => floatval($deliveryCost),
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->config['tax_type_fiscal'] ?: 'none'
                ],
                'paymentType' => $this->config['payment_type_delivery_fiscal'] ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => floatval($deliveryCost)
            ];
        }

        return $items;
    }


    /**
     * Create and return an instance of the VtbApi
     *
     * @return VtbApi An instance of the VtbApi class.
     */
    private function getVtbApi(): VtbApi
    {
        return new VtbApi(
            $this->config['client_id'],
            $this->config['client_secret'],
            (bool) $this->config['test_mode'],
            $this->config['merchant_authorization'] ?: ''
        );
    }


    /**
     * Инициализация и настройка объекта класса VtbPayLogger.
     *
     * Эта функция инициализирует и настраивает логгер, используемый плагином VtbPay для ведения журнала.
     *
     * @return void
     */
    private function setVtbpayLogger(): void
    {
        $modx = $this->modx;
        $modx->setLogTarget([
           'target' => 'FILE',
           'options' => [
               'filename' => 'vtbpay-' . date('d-m-Y') . '.log'
            ]
        ]);

        $logging = (bool) ($this->config['logging'] ?? false);
        $sensitiveDataKeys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $this->logger = VtbPayLogger::getInstance()
                        ->setOption('showCurrentDate', false)
                        ->setOption('showLogLevel', false)
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitiveDataKeys)
                        ->setCustomRecording(function($message) use ($modx) {
                            $type = modX::LOG_LEVEL_ERROR;
                            $modx->setLogLevel($type);
                            $modx->log(
                                $type,
                                '[miniShop2:VtbPay] ' . $message
                            );
                        }, VtbPayLogger::LOG_LEVEL_ERROR)
                        ->setCustomRecording(function($message) use ($modx, $logging) {
                            $type = modX::LOG_LEVEL_INFO;
                            $modx->setLogLevel($type);
                            if ($logging) $modx->log(
                                $type,
                                '[miniShop2:VtbPay] ' . $message
                            );
                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
