<?php

namespace Vtbpay\Traits;

use \Vtbpay\Classes\Api\AtolApi,
    \Vtbpay\Classes\Exception\VtbPayException;

trait Atol
{
    /**
     * Обработчик события paymentRequestReceived
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentRequestReceivedAtol(array $vars)
    {
        // Если есть uuid, значит к вебхуку обращается АТОЛ
        $uuid = $vars['request_data']['uuid'] ?? '';
        if ($vars['mode'] === 'webhook' && !empty($uuid)) {
            $this->webhookHandlerAtol($uuid, $vars['order_id']);
            die('ok');
        }
    }


    /**
     * Обработчик события paymentPaid
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentPaidAtol(array $vars)
    {
        if (
            $this->pm_config['enable_fiscal'] &&
            $this->pm_config['ofd_fiscal'] === 'atol'
        ) $this->sendReceiptAtol();
    }


    /**
     * Обработчик события paymentRefunded
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentRefundedAtol(array $vars)
    {
        if (
            $this->pm_config['enable_fiscal'] &&
            $this->pm_config['ofd_fiscal'] === 'atol'
        ) $this->sendReceiptAtol(true);
    }


    /**
     * Создаем новый экземпляр класса AtolApi с заданными конфигурациями платёжной системы.
     *
     * @return AtolApi
     */
    private function getAtolApi(): AtolApi
    {
        return new AtolApi(
            $this->pm_config['login_atol_fiscal'],
            $this->pm_config['pass_atol_fiscal'],
            $this->pm_config['inn_atol_fiscal'],
            $this->pm_config['tax_system_atol_fiscal'],
            $this->pm_config['kkt_atol_fiscal'],
            $this->pm_config['email_fiscal'],
            (bool) $this->pm_config['test_mode_atol_fiscal']
        );
    }


    /**
     * Проверяет статус формирования чека в АТОЛ и оповещает при ошибке
     *
     * @param string $uuid Полученные данные
     * @param string $order_id Идентификатор заказа
     *
     * @return void
     */
    private function webhookHandlerAtol(
        string $uuid,
        string $order_id
    ): void {
        $response = $this->getAtolApi()->getReceiptInfo($uuid);

        if ($response['status'] === 'fail'){
            $this->sendReceiptErrorEmail(sprintf(
                _JSHOP_CFG_VTBPAY_ERROR_ATOL_CHECK,
                $order_id
            ));
        }
        elseif ($response['status'] === 'wait') {
            $this->sendReceiptErrorEmail(sprintf(
                _JSHOP_CFG_VTBPAY_WAIT_ATOL_CHECK,
                $order_id
            ));
        }
    }


    /**
     * Отправляет в АТОЛ чек прихода или возврата на печать и возвращает результат.
     *
     * @param bool $is_return Данный чек "Возврат прихода" или "Приход"
     *
     * @return array
     */
    private function sendReceiptAtol(bool $is_return = false): array
    {
        $order_id = $this->order->order_id;

        try {
            $email = $this->getCustomerEmail();

            $items = $this->preparingItemsForAtol(
                $this->getItems()
            );

            $order_total = self::parseAmount($this->order->order_total);

            $callback_url = \JURI::root() . 'index.php?option=com_jshopping&controller=checkout&task=step7&js_paymentclass=' . __CLASS__ .
                            '&act=notify&request_type=webhook&no_lang=1';

            $response = $this->getAtolApi()->sendReceipt(
                $order_id,
                $items,
                $order_total,
                $email,
                $callback_url,
                $is_return
            );

            return $response;

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s', $e->getMessage()
            ), $context);

            $this->sendReceiptErrorEmail(sprintf(
                _JSHOP_CFG_VTBPAY_ERROR_ATOL_CHECK,
                $order_id
            ));

            return [];
        }
    }


    /**
     * Подготавливает элементы для АТОЛ.
     *
     * @param array $items Элементы для подготовки.
     *
     * @return array Массив подготовленных элементов.
     */
    private function preparingItemsForAtol(array $items): array
    {
        return array_map(function ($item) {
            return [
                'name' => $item['name'], // Лейбл
                'price' => $item['price'], // Цена товара за 1шт
                'sum' => $item['amount'], // Итоговая цена (цена * кол-во)
                'quantity' => $item['quantity'], // Кол-во позиций товара (сколько шт)
                'measure' => $item['measure'], // Единица измерения
                'vat' => [
                    'type' => $item['taxParams']['taxType'] // Налоговая ставка (НДС)
                ],
                'payment_method' => $item['paymentType'], // Способ расчета
                'payment_object' => $item['paymentSubject'], // Предмет расчета
            ];
        }, $items);
    }


    /**
     * Отправляет письмо об ошибке при выдаче чека в VirtueMart.
     *
     * Использует настройки почты из конфигурации Joomla.
     *
     * @param string $message Текст сообщения об ошибке (HTML).
     * @return void
     */
    private function sendReceiptErrorEmail($message): void
    {
        // Получаем конфигурацию Joomla
        $config = \JFactory::getConfig();
        // Название магазина и email администратора из настроек Joomla
        $store_name = $config->get('sitename', self::CMS_NAME);
        $admin_email = $config->get('mailfrom', "noreply@{$_SERVER['HTTP_HOST']}");

        // Получаем email для отправки из настроек VirtueMart (например, поле в конфигурации)
        $recipient_email = $this->pm_config['email_fiscal'] ?: $admin_email;

        // Формируем тему письма
        $subject = _JSHOP_CFG_VTBPAY_RECEIPT_ISSUANCE_ERROR . ': ' . $store_name;

        // Создаём объект почты
        $mailer = \JFactory::getMailer();

        // Устанавливаем параметры письма
        $mailer->setSender([$admin_email, $store_name]); // Email и имя отправителя
        $mailer->addRecipient($recipient_email);         // Email получателя
        $mailer->setSubject($subject);                   // Тема письма
        $mailer->isHtml(true);                           // Устанавливаем HTML-формат
        $mailer->setBody($message);                      // Текст сообщения
        $mailer->Send();
    }
}
