<?php
/**
* @package JoomShopping for Joomla!
* @subpackage payment
* @author vtb
*/
(defined('_JEXEC') || PHP_SAPI == 'cli') or die('Restricted access');

// Checking PHP Version
if (version_compare(PHP_VERSION, '7.4.0', '<')) {
    die('Payment module "Payment system VTB" requires PHP version 7.4.0 or higher.');
}

require_once __DIR__ . '/vendor/autoload.php';

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Common\EventDispatcher,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Vtbpay\Traits\Atol,
    \Symfony\Component\Dotenv\Dotenv;

class pm_vtbpay extends PaymentRoot
{
    use Atol;

    const CMS_NAME = 'Joomla! JoomShopping';
    const PLUGIN_VERSION = '1.5.10';

    /**
     * @var VtbPayLogger Объект для логирования.
     */
    private VtbPayLogger $logger;
    private array $pm_config;
    private object $order;
    private EventDispatcher $event_dispatcher;


    /**
     * The constructor initializes the payment root, language file, logger and stages.
     */
    public function __construct()
    {
        JSFactory::loadExtLanguageFile('pm_vtbpay');
        JLog::addLogger(
            ['text_file' => 'com_jshopping.pm_vtbpay-' . date('d-m-Y') . '.php'],
            JLog::ALL,
            ['pm_vtbpay']
        );

        $this->initDotenv();
        $this->event_dispatcher = new EventDispatcher($this);
    }


    /**
     * Init Dotenv global settings
     */
    private function initDotenv()
    {
        try {
            // Получаем настройки из .env в корне
            $env_path = __DIR__ . '/config/.env';
            if (file_exists($env_path)) (new Dotenv())->load($env_path);
        } catch (\Exception $e) {
            $this->pm_config['logging'] = true;
            $this->setVtbpayLogger();
            // Handle exception and log error
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s', $e->getMessage()
            ), [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ]);
            throw $e;
        }
    }


    /**
     * Displays the payment form for the VTB Pay payment method.
     *
     * @param array $params The parameters for the payment form.
     * @param array $pmconfigs The payment method configurations.
     */
    public function showPaymentForm($params, $pmconfigs)
    {
        include dirname(__FILE__) . '/paymentform.php';
    }


    /**
     * Displays the admin form parameters for the VTB Pay payment method plugin.
     *
     * @param array $params The parameters for the admin form.
     */
    public function showAdminFormParams($params)
    {
        $document = JFactory::getDocument();
        $document->addScript(str_replace(JPATH_ROOT, '', __DIR__) . '/assets/js/admin_vtbpay.js');

        $orders = JSFactory::getModel('orders', 'JshoppingModel');

        $settings_definitions = include dirname(__FILE__) . '/config/settings.php';
        include dirname(__FILE__) . '/adminparamsform.php';
    }


    /**
     * Creates a new VtbApi instance with the given payment method configurations.
     *
     * @return VtbApi The new VtbApi instance.
     */
    private function getVtbApi(): VtbApi
    {
        return new VtbApi(
            $this->pm_config['client_id'],
            $this->pm_config['client_secret'],
            (bool) $this->pm_config['test_mode'],
            $this->pm_config['merchant_authorization'] ?? ''
        );
    }


    /**
     * Перенаправление на форму оплаты
     *
     * @param array $pm_config The payment method configurations.
     * @param object $order The order object.
     */
    public function showEndForm($pm_config, $order)
    {
        $this->pm_config = $pm_config;
        $this->order = $order;

        try {
            $this->setVtbpayLogger();
            $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

            return JFactory::getApplication()->redirect(
                $this->getPayUrl()
            );

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $msg = $e->getMessage() . ' Order ID: ' . $this->order->order_id;
            // Handle exception and log error
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s', $msg
            ), $context);

            JFactory::getApplication()->enqueueMessage($msg, 'error');
        }
    }


    /**
     * Получение URL для оплаты заказа.
     *
     * @return string URL для оплаты заказа.
     */
    private function getPayUrl(): string
    {
        $return_url = JUri::base() .
                      'index.php?option=com_jshopping&controller=checkout&task=step7&js_paymentclass=' .
                      __CLASS__ . '&act=finish&request_type=return';

        $email = $this->getCustomerEmail();

        $order_total = self::parseAmount($this->order->order_total);

        $two_stage = (bool) ($this->pm_config['two_stage'] ?? false);
        $enable_fiscal = (bool) ($this->pm_config['enable_fiscal'] ?? false);
        $ofd_fiscal = $this->pm_config['ofd_fiscal'] ?? '';

        $items = [];
        if (
            !$two_stage &&
            $enable_fiscal &&
            $ofd_fiscal === '1ofd'
        ) $items = $this->getItems();

        $response = $this->getVtbApi()->getOrderLink(
            $this->order->order_id,
            $email,
            time(),
            $order_total,
            $return_url,
            $two_stage,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Метод проверки валидности платежа, вызывается при создании платежа, после возврата пользователя на страницу
     * подтверждения заказа, так же вызывается при приходе нотификации от платёжной системы.
     * Выполняет "проверку" транзакций (может проводить обработку запросов на Success Url, Fail Url и Result Url).
     *
     * @param $pm_config Array  Массив настроек модуля оплаты
     * @param $order     Object Объект текущего заказа, по которому происходит оформление
     * @param $act       String Тип запроса (notify, return, cancel) Возвращаем массив с результатом проверки транзакции
     *
     * @return array($rescode, $restext, $transaction, $transaction_data)
     */
    public function checkTransaction($pm_config, $order, $act)
    {
        $this->order = $order;
        if ($this->order->order_created === 0) $this->order->order_created = 1;

        try {
            $payment_status_data = $this->getPaymentStatusData();
            $payment_status = $payment_status_data['object']['status']['value'];

            $available_statuses = [
                'PAID' => [
                    'rescode' => 1,
                    'order_status' => (int) $this->pm_config['transaction_end_status'],
                    'restext' => _JSHOP_CFG_VTBPAY_MSG_SUCCESSFULLY_PAID,
                    'fnc' => function() {
                        $this->event_dispatcher->dispatch('paymentPaid');
                    }
                ],
                'PENDING' => [
                    'rescode' => 0,
                    'order_status' => (int) $this->pm_config['transaction_pending_status'],
                    'restext' => _JSHOP_CFG_VTBPAY_MSG_SUCCESSFULLY_PAID,
                    'fnc' => function() {
                        $cart = JSFactory::getModel('cart', 'jshop');
                        $cart->clear();
                    }
                ],
                'REFUNDED' => [
                    'rescode' => 0,
                    'order_status' => (int) $this->pm_config['transaction_refund_status'],
                    'restext' => _JSHOP_CFG_VTBPAY_MSG_SUCCESSFULLY_REFUNDED,
                    'fnc' => function() {
                        $this->event_dispatcher->dispatch('paymentRefunded');
                    }
                ]
            ];

            $order_status_data = $available_statuses[$payment_status] ?? [
                'rescode' => 0,
                'order_status' => (int) $this->pm_config['transaction_cancel_status'],
                'restext' => _JSHOP_CFG_VTBPAY_MSG_SUCCESSFULLY_CANCELED
            ];

            if (
                $payment_status !== 'REFUNDED' &&
                isset($payment_status_data['object']['transactions']['refunds'])
            ) {
                // Указываем статус возврата
                $order_status_data = $available_statuses['REFUNDED'];
            }
        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $this->order->order_id ?: ''
            ), $context);

            $order_status_data = [
                'rescode' => 0,
                'order_status' => (int) $this->pm_config['transaction_cancel_status'],
                'restext' => $e->getMessage()
            ];
        }

        $this->logger->debug(
            __FUNCTION__ . ' > change_status - INPUT: ', [
            'order_status_data' => $order_status_data
        ]);

        $this->order->order_status = $order_status_data['order_status'];
        $this->order->store();

        if (isset($order_status_data['fnc'])) $order_status_data['fnc']();

        $transactions_payment = $payment_status_data['object']['transactions']['payments'] ?? [];
        $payment_id = end($transactions_payment)[0]['object']['paymentId'] ?? '';

        return [
            $order_status_data['rescode'],
            $order_status_data['restext'],
            $payment_id,
            $transactions_payment
        ];
    }


    /**
     * Вызывается при входящем вебхуке перед методом CheckTransaction().
     * Инициализирует массив с параметрами обработки входящего запроса
     *
     * @return array Массив с параметрами обработки входящего запроса
     */
    private function getPaymentStatusData(): array
    {
        $response = $this->getVtbApi()->getOrderInfo(
            $this->order->order_id
        );

        return $response ?: [];
    }


    /**
     * Вызывается при входящем вебхуке перед методом CheckTransaction().
     * Инициализирует массив с параметрами обработки входящего запроса
     *
     * @param array $pm_config Array Массив настроек способа оплаты
     *
     * @return array Массив с параметрами обработки входящего запроса
     */
    public function getUrlParams($pm_config)
    {
        $this->pm_config = $pm_config;

        $params = [];
        $params['hash'] = '';
        $params['checkHash'] = 0;
        $params['checkReturnParams'] = 1;

        $this->setVtbpayLogger();
        $input = JFactory::getApplication()->input;
        $mode = $input->getString('request_type', null);

        // Logging request data
        if (in_array($mode, ['return', 'webhook'])) {
            $this->logger->setOption('additionalCommonText', $mode . '-' . rand(1111, 9999));
            $request_data = ($mode === 'return') ? $_REQUEST : json_decode(file_get_contents('php://input'), true);
            $this->logger->debug(
                __FUNCTION__ . ' > ' . $mode . ': ',
                ['request_data' => $request_data]
            );

            $params['order_id'] = $input->getString('orderId') ?:
                                  $request_data['object']['orderId'] ??
                                  $request_data['external_id'] ?? '';

            $this->event_dispatcher->dispatch('paymentRequestReceived', [
                'mode' => $mode,
                'request_data' => $request_data,
                'order_id' => $params['order_id']
            ]);
        }

        return $params;
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems()
    {
        $get_all_items = $this->getOrderItems();
        $items = [];
        foreach ($get_all_items as $key => $item) {
            $product_price = self::parseAmount($item['product_item_price']);

            if (!$item['manufacturer_code'] || $item['manufacturer_code'] == '0') {
                $code = (string) $item['product_id'];
            }
            else {
                $code = (string) $item['manufacturer_code'];
            }

            $items[] = [
                'positionId' => ($key + 1),
                'name' => $item['product_name'],
                'code' => $code,
                'price' => floatval($product_price),
                'measure' => (int) $this->pm_config['measure_fiscal'] ?: 0,
                'quantity' => (int) $item['product_quantity'],
                'taxParams' => [
                    'taxType' => $this->pm_config['tax_fiscal'] ?: 'none'
                ],
                'paymentType' => $this->pm_config['payment_fiscal'] ?: 'full_prepayment',
                'paymentSubject' => (int) $this->pm_config['subject_fiscal'] ?: 1,
                'amount' => floatval($product_price * $item['product_quantity'])
            ];
        }

        // Стоимость доставки
        if ($this->order->order_shipping > 0) {
            $delivery_price = self::parseAmount($this->order->order_shipping);
            $items[] = [
                'positionId' => ($key + 2),
                'name' => 'Доставка',
                'price' => $delivery_price,
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->pm_config['tax_fiscal'] ?: 'none'
                ],
                'paymentType' => $this->pm_config['delivery_fiscal'] ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => $delivery_price
            ];
        }

        return $items;
    }


    /**
     * Получает email клиента иначе выкидывает ошибку
     *
     * @param object $order The order object.
     *
     * @return string email клиента.
     */
    private function getCustomerEmail(): string
    {
        $email = $this->order->email ?: $this->pm_config['email_fiscal'] ?? '';
        if (!self::validateEmail($email)) {
            throw new VtbPayException(_JSHOP_CFG_VTBPAY_MSG_EMAIL_NOT_VALID, [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email): bool
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Получает товары заказа из бд по его идентификатору
     *
     * @return array Массив товаров заказа.
     */
    private function getOrderItems()
    {
        $db =& JFactory::getDBO();
        $q = "SELECT * FROM `#__jshopping_order_item` where `order_id`=" . $this->order->order_id;
        $db->setQuery($q);
        return $db->loadAssocList();
    }


    /**
     * Преобразует строку в число с плавающей запятой.
     *
     * @param mixed $amount Сумма для преобразования.
     * @return float Преобразованное значение суммы.
     */
    private static function parseAmount($amount): float
    {
        if (is_string($amount)) {
            $amount = str_replace([' ', ','], ['', '.'], $amount);
        }

        return floatval(number_format($amount, 2, '.', ''));
    }


    /**
     * Устанавливает объект логирования для модуля.
     *
     * @return void
     */
    private function setVtbpayLogger()
    {
        $logging = (bool) $this->pm_config['logging'];
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $this->logger = VtbPayLogger::getInstance()
            ->setOption('showCurrentDate', false)
            ->setOption('showLogLevel', false)
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitive_data_keys)
            ->setCustomRecording(function($message)
            {
                // Логирование ошибки
                JLog::add($message, JLog::ERROR, 'pm_vtbpay');

            }, VtbPayLogger::LOG_LEVEL_ERROR)
            ->setCustomRecording(function($message) use ($logging)
            {
                // Логирование сообщения дебага
                if ($logging) JLog::add($message, JLog::DEBUG, 'pm_vtbpay');

            }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
