<?php
defined('HOSTCMS') || exit('HostCMS: access denied.');

// Checking PHP Version
if (version_compare(PHP_VERSION, '7.4.0', '<')) {
    die('Payment module "Payment system VTB" requires PHP version 7.4.0 or higher.');
}

require_once CMS_FOLDER .
            'modules' .  DIRECTORY_SEPARATOR .
            'vtbpay' .  DIRECTORY_SEPARATOR .
            'vendor'. DIRECTORY_SEPARATOR .
            'autoload.php';

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;


/**
 * Класс Shop_Payment_System_HandlerXX для обработки платежей через VTB
 *
 * Класс, предназначенный для интеграции с платежной системой VTB.
 * Этот обработчик позволяет настроить и обработать платежи через VTB в вашем интернет-магазине.
 *
 */
class Shop_Payment_System_HandlerXX extends Shop_Payment_System_Handler
{
    const CMS_NAME = 'HostCMS';
    const PLUGIN_VERSION = '1.1.7';

    // Client ID мерчанта *
    private $client_id = "";

    // Client Secret мерчанта *
    private $client_secret = "";

    // Merchant-Authorization
    private $merchant_authorization = "";

    /**
     * Тестовый режим
     * TRUE - включить
     * FALSE - выключить
     */
    private $test_mode = TRUE;

    /**
     * Логирование запросов к платежному шлюзу
     * TRUE - включить
     * FALSE - выключить
     */
    private $enable_logging = TRUE;

    /**
     * Автоматический редирект после нажатия на кнопу оплатить
     * TRUE - включить
     * FALSE - выключить
     */
    protected $auto_redirect = TRUE;

    // Дефолтный идентификатор валюты
    protected $currency_id = 1;

    // Дефолтный код валюты
    protected $currency_code = 'RUB';

    // XSL-шаблон для накладной
    protected $XSL_templates_invoice = 'КвитанцияПД4';

    // XSL-шаблон для уведомления
    protected $XSL_templates_notification = 'ОплатаПоФормеПД4НовыйСайт2';

    // Объект для логирования
    public VtbPayLogger $logger;

    //  Ниже настройки для фискализации

    /**
     * Включение / Отключение фискализации
     * Дефолтное значение false - выключено
     * Возможные значения:
     * true - включено
     * false - выключено
     */
    private $fiscal_enable = false;

    /**
     * E-mail
     * Электронная почта для отправки фискального чека, если не указана почта пользователя.
     */
    private $fiscal_email = '';

    /**
     * Способ расчета для услуги доставки.
     * Дефолтное значение full_prepayment - Предоплата 100%
     * Возможные значения:
     * full_prepayment - Предоплата 100%
     * prepayment - Предоплата
     * advance - Аванс
     * full_payment - Полный расчёт
     * partial_payment - Частичный расчёт и кредит
     * credit - Передача в кредит
     * credit_payment - Оплата кредита
     */
    private $fiscal_delivery = 'full_prepayment';

    /**
     * Единицы измерения количества предмета расчета.
     * Дефолтное значение 0 - Применяется для предметов расчета, которые могут быть реализованы поштучно или единицами
     * Возможные значения:
     * 0 - Применяется для предметов расчета, которые могут быть реализованы поштучно или единицами
     * 10 - Грамм
     * 11 - Килограмм
     * 12 - Тонна
     * 20 - Сантиметр
     * 21 - Дециметр
     * 22 - Метр
     * 30 - Квадратный сантиметр
     * 31 - Квадратный дециметр
     * 32 - Квадратный метр
     * 40 - Миллилитр
     * 41 - Литр
     * 42 - Кубический метр
     * 50 - Киловатт час
     * 51 - Гигакалория
     * 70 - Сутки (день)
     * 71 - Час
     * 72 - Минута
     * 73 - Секунда
     * 80 - Килобайт
     * 81 - Мегабайт
     * 82 - Гигабайт
     * 83 - Терабайт
     * 255 - Применяется при использовании иных единиц измерения
     */
    private $fiscal_measure = '0';

    /**
     * Ставка налогообложения.
     * Дефолтное значение none - Без НДС
     * Возможные значения:
     * none - Без НДС
     * vat0 - НДС по ставке 0%
     * vat10 - НДС чека по ставке 10%
     * vat110 - НДС чека по расчетной ставке 10/110
     * vat20 - НДС чека по ставке 20%
     * vat120 - НДС чека по расчетной ставке 20/120
     * vat5 - НДС чека по ставке 5%
     * vat7 - НДС чека по ставке 7%
     * vat105 - НДС чека по расчетной ставке 5/105
     * vat107 - НДС чека по расчетной ставке 7/107
     */
    private $fiscal_tax = 'none';

    /**
     * Способ расчета.
     * Дефолтное значение full_prepayment - Предоплата 100%
     * Возможные значения:
     * full_prepayment - Предоплата 100%
     * prepayment - Предоплата
     * advance - Аванс
     * full_payment - Полный расчёт
     * partial_payment - Частичный расчёт и кредит
     * credit - Передача в кредит
     * credit_payment - Оплата кредита
     */
    private $fiscal_payment_type = 'full_prepayment';

    /**
     * Предмет расчета.
     * Дефолтное значение 1 - Товар
     * Возможные значения:
     * 1 - Товар
     * 2 - Подакцизный товар
     * 3 - Работа
     * 4 - Услуга
     * 5 - Ставка азартной игры
     * 6 - Выигрыш азартной игры
     * 7 - Ставка лотереи
     * 8 - Выигрыш лотереи
     * 9 - Предоставление прав
     * 10 - Платеж
     * 11 - Агентское вознаграждение
     * 12 - Выплата
     * 13 - Иной предмет расчёта
     * 14 - Имущественное право
     * 15 - Внереализационный доход
     * 16 - Страховые взносы
     * 17 - Торговый сбор
     * 18 - Курортный сбор
     * 19 - Залог
     * 20 - Расход
     * 21 - Взносы на ОПС ИП
     * 22 - Взносы ОПС
     * 23 - Взносы на ОМС ИП
     * 24 - Взносы на ОМС
     * 25 - Взносы на ОСС
     * 26 - Платеж казино
     * 27 - Выдача денежных средств банковским платежным агентом
     * 30 - Подакцизный товар, подлежащий маркировке средством идентификации, не имеющий кода маркировки
     * 31 - Подакцизный товар, подлежащий маркировке средством идентификации, имеющий код маркировки
     * 32 - Товар, подлежащий маркировке средством идентификации, не имеющий кода маркировки, за исключением подакцизного товара
     * 33 - Товар, подлежащий маркировке средством идентификации, имеющий код маркировки, за исключением подакцизного товара
     */
    private $fiscal_subject = '1';


    /**
     * Конструктор класса.
     * При создании экземпляра класса, задает значения полям $currency_id и $currency_code,
     * получая их из переданной модели платежной системы.
     *
     * @param Shop_Payment_System_Model $oShop_Payment_System_Model Модель платежной системы
     */
    public function __construct(Shop_Payment_System_Model $oShop_Payment_System_Model)
    {
        parent::__construct($oShop_Payment_System_Model);
        $this->currency_id = $oShop_Payment_System_Model->shop_currency_id;
        $currency = Core_Entity::factory('Shop_Currency')->getById($this->currency_id);
        $this->currency_code = $currency->code;

        $env_path = CMS_FOLDER .
            'modules' .  DIRECTORY_SEPARATOR .
            'vtbpay' .  DIRECTORY_SEPARATOR .
            'config' .  DIRECTORY_SEPARATOR .
            '.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);
    }


    /**
     * Возвращает html-строку с ссылкой для перехода к оплате.
     * Генерирует ссылку на оплату через VTB API и возвращает кнопку оплаты.
     * @return Shop_Payment_System_HandlerXX Возвращает экземпляр текущего объекта
     * @throws Exception Бросает исключение, если обязательные параметры пусты или не установлены
     */
    public function userExecute()
  	{
        $order_id = $this->getShopOrder()->id;
        $this->setVtbpayLogger();
        $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            if (empty($this->client_id)) {
                throw new \Exception(Core::_('Vtbpay.client_id_is_empty'));
            }

            if (empty($this->client_secret)) {
                throw new \Exception(Core::_('Vtbpay.client_secret_is_empty'));
            }

            $pay_url = $this->getPayUrl();
            $this->showPayButton($pay_url);

            return $this;
        } catch (\Exception | VtbPayException $e) {
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __CLASS__ . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $order_id ?: ''
            ), $context);

            throw $e;
        }
  	}


    /**
     * Отображает кнопку оплаты
     */
    private function showPayButton(string $pay_url): void
    {
        $html = '<h2>' . Core::_('Vtbpay.go_to_payment_page') . ' #' . $this->getShopOrder()->id . '</h2>
                 <a href="' . $pay_url . '" class="btn" id="button-confirm">' . Core::_('Vtbpay.pay_for_the_order') . '</a>';
        if ($this->auto_redirect) {
            $html .= '<script type="text/javascript">
                        const paymentButton = document.getElementById("button-confirm");
                        if (paymentButton) {
                            paymentButton.click();
                        }
                      </script>';
        }
        echo $html;
    }


    /**
     * Возвращает ссылку на платежную страницу перехода к оплате.
     * Генерирует ссылку на оплату через VTB API.
     * @throws Exception Бросает исключение, если обязательные параметры пусты или не установлены
     */
    private function getPayUrl()
    {
        $order_id = $this->getShopOrder()->id;

        // Get return url
        $return_url = $this->getShopCartUrl() . '?method=vtbpay&action=return';

        // Get Total
        $total = $this->getSumWithCoeff();

        $email = $this->getCustomerEmail();

        $items = [];
        if ($this->fiscal_enable) $items = $this->getItems();

        $response = $this->getVtbApi()->getOrderLink(
            $order_id,
            $email,
            time(),
            $total,
            $return_url,
            false,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает email клиента, иначе выкидывает ошибку
     */
    private function getCustomerEmail(): string
    {
        // Get Email
        if(
            !isset($this->getShopOrder()->email) ||
            empty($email = $this->getShopOrder()->email)
        ) {
            $email = $this->_orderParams['email'] ?? $this->fiscal_email;
        }
        if (!self::validateEmail($email)) {
            throw new VtbPayException(Core::_('Vtbpay.email_not_valid'), [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems(): array
    {
        $Shop_Order_Items = $this->getShopOrder()->Shop_Order_Items->findAll();

        $items = [];
        foreach ($Shop_Order_Items as $key => $item) {
            if (strpos($item->name, 'Доставка') == false) {
                    $items[] = [
                        'positionId' => ($key + 1),
                        'name' => $item->name,
                        'code' => $item->marking ?: '',
                        'price' => floatval($item->price),
                        'measure' => (int) $this->fiscal_measure ?: 0,
                        'quantity' => (int) $item->quantity,
                        'taxParams' => [
                            'taxType' => $this->fiscal_tax ?: 'none'
                        ],
                        'paymentType' => $this->fiscal_payment_type ?: 'full_prepayment',
                        'paymentSubject' => (int) $this->fiscal_subject ?: 1,
                        'amount' => floatval($item->price * $item->quantity)
                    ];
            } else {
                if ($item->price > 0) {
                $items[] = [
                    'positionId' => ($key + 2),
                    'name' => $item->name,
                    'code' => $item->marking ?: '',
                    'price' => floatval($item->price),
                    'measure' => 0,
                    'quantity' => 1,
                    'taxParams' => [
                        'taxType' => $this->fiscal_tax ?: 'none'
                    ],
                    'paymentType' => $this->fiscal_delivery ?: 'full_prepayment',
                    'paymentSubject' => 4,
                    'amount' => floatval($item->price * $item->quantity)
                ];
            }
            }
        }

        return $items;
    }


    /**
     * Получает и применяет XSL шаблон для счета.
     *
     * @return Shop_Payment_System_HandlerXX Возвращает экземпляр текущего объекта
     */
    public function getInvoice()
    {
        $this->xsl(
          Core_Entity::factory('Xsl')->getByName($this->XSL_templates_invoice)
        );
        return parent::getInvoice();
    }


    /**
     * Получает и применяет XSL шаблон для уведомления.
     *
     * @return Shop_Payment_System_HandlerXX Возвращает экземпляр текущего объекта
     */
    public function getNotification()
    {
        $this->xsl(
          Core_Entity::factory('Xsl')->getByName($this->XSL_templates_notification)
        );
        return parent::getNotification();
    }


    /**
     * Получает экземпляр API VTB.
     * Создает и возвращает новый экземпляр VtbApi, используя данные конфигурации.
     *
     * @return VtbApi Возвращает экземпляр класса VtbApi
     */
    private function getVtbApi(): VtbApi
    {
        return new VtbApi(
            $this->client_id,
            $this->client_secret,
            (bool) $this->test_mode,
            $this->merchant_authorization ?: ''
        );
    }


    /**
     * Возвращает общую сумму товаров заказа с учетом коэффициента.
     * Если валюты заказа и платежной системы различаются, преобразует сумму в валюту платежной системы.
     *
     * @return float Возвращает общую сумму заказа с учетом коэффициента
     */
    public function getSumWithCoeff()
    {
        $sum = 0;

        if ($this->currency_id > 0 && $this->getShopOrder()->shop_currency_id > 0) {
            $sum = Shop_Controller::instance()->getCurrencyCoefficientInShopCurrency(
                $this->getShopOrder()->Shop_Currency,
                Core_Entity::factory('Shop_Currency', $this->currency_id)
            );
        }

        return Shop_Controller::instance()->round($sum * $this->getShopOrder()->getAmount());
    }


    /**
     * Обрабатывает запрос перед отображением контента.
     * Если переданный метод равен 'vtbpay' и указан orderId, начинает обработку платежа.
     *
     * @return void
     */
    public function checkPaymentBeforeContent()
    {
        $method = Core_Array::getGet('method');
        $order_id = intval(Core_Array::getGet('orderId'));

        if (
                $method == 'vtbpay' &&
                !empty($order_id)
        ) {
            $oShop_Order = Core_Entity::factory('Shop_Order')->find($order_id);

            if (!is_null($oShop_Order->id)) {
                Shop_Payment_System_Handler::factory($oShop_Order->Shop_Payment_System)
                    ->shopOrder($oShop_Order)
                    ->paymentProcessing();
            }
        }
    }


    /**
     * Обрабатывает уведомления об оплате заказа или возврат пользователя после оплаты.
     * Определяет, является ли запрос уведомлением об оплате или возвратом пользователя.
     *
     * @return void
     */
    public function paymentProcessing()
    {
        $order_id = $this->getShopOrder()->id;

        $this->setVtbpayLogger();
        $this->logger->setOption('additionalCommonText', 'return-' . rand(1111, 9999));

        try {
            $action = Core_Array::getGet('action');
            if ($action == 'return') {
                if ($this->getPaymentStatus() === 'PAID') {
                    $this->getShopOrder()->system_information = Core::_('Vtbpay.order_successfully_paid') . ' \n';
                    $this->getShopOrder()->paid();
                    // Установка XSL-шаблонов в соответствии с настройками в узле структуры
            				$this->setXSLs();
            				// Отправка писем клиенту и пользователю
            				$this->send();

                    ob_start();
            				$this->shopOrderBeforeAction(clone $this->getShopOrder())->changedOrder('changeStatusPaid');
            				ob_get_clean();

                    $payment_status = 'success';
                } else {
                    $this->getShopOrder()->system_information = Core::_('Vtbpay.payment_error') . ' \n';
                    $this->getShopOrder()->save();
                    $payment_status = 'fail';
                }

                header('location: ' . $this->getShopCartUrl() . "?payment={$payment_status}&order_id={$order_id}");
                exit();
            }
        } catch (\Exception | VtbPayException $e) {
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __CLASS__ . ' > ' . __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $order_id ?: ''
            ), $context);

            throw $e;
        }
    }


    /**
     * Возвращает Url корзины магазина.
     */
    private function getShopCartUrl(): string
    {
        $oSite_Alias = $this->getShopOrder()->Shop->Site->getCurrentAlias();
        $site_alias = !is_null($oSite_Alias) ? $oSite_Alias->name : '';
        $shop_path = $this->getShopOrder()->Shop->Structure->getPath();

        return 'https://' . $site_alias . $shop_path . 'cart/';
    }


    /**
     * Получение статуса оплаты.
     *
     * @return string Значение статуса заказа, если оно существует, или пустая строка в противном случае.
     */
    public function getPaymentStatus()
    {
        $order_id = $this->getShopOrder()->id;

        $response = $this->getVtbApi()->getOrderInfo($order_id);

        return $response['object']['status']['value'] ?? '';
    }


    /**
     * Обрабатывает новый заказ.
     * Вызывает стандартную обработку заказа из родительского класса,
     * устанавливает XSL шаблоны и отправляет уведомления о новом заказе.
     *
     * @return Shop_Payment_System_HandlerXX Возвращает экземпляр текущего объекта
     */
    protected function _processOrder()
    {
        // Вызываем стандартное оформление заказа из родительского класса Shop_Payment_System_Handler
        parent::_processOrder();
        // Установка XSL-шаблонов в соответствии с настройками в узле структуры
        $this->setXSLs();
        // Отправка писем клиенту и пользователю
        $this->send();

        return $this;
    }


    /**
     * Устанавливает объект логирования для модуля.
     *
     * @return VtbPayLogger
     */
    private function setVtbpayLogger(): void
    {
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];
        $logger = VtbPayLogger::getInstance();
        $this->logger = $logger
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitive_data_keys)
            ->setLogFilePath(dirname(__DIR__, 2) . '/logs/vtbpay-' . date('d-m-Y') . '.log')
            ->setCustomRecording(function($message) use ($logger) {
                if ($this->enable_logging) $logger->writeToFile($message);
            }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
