<?php

namespace Drupal\commerce_vtbpayment\Traits;

use \Drupal\commerce_order\Entity\OrderInterface,
    \Drupal\commerce_shipping\Entity\Shipment;

/**
 * Трейт для обновления стоимости доставки в заказе Drupal Commerce.
 */
trait Shipping
{
    /**
     * Обработчик событий afterGetPaymentStatusWebhook и afterGetPaymentStatusReturn.
     *
     * @param array $vars
     *   Данные статуса платежа.
     *
     * @return void
     */
    public function onAfterGetPaymentStatusShipping(array $vars): void
    {
        $input_amount = $vars['payment_status_data']['object']['amount']['value'] ?? 0;

        if ($input_amount > (float) $this->order->getTotalPrice()->getNumber()) {
            $order_code = $vars['payment_status_data']['object']['orderCode'] ?? '';
            $this->updateShippingCost(
                $this->order,
                $order_code,
                (float) $input_amount
            );
        }
    }


    /**
     * Обновляет или добавляет стоимость доставки в заказе.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $order
     *   Объект заказа.
     * @param string $order_code
     *   Код заказа (из API ВТБ).
     * @param float $new_total
     *   Новая итоговая сумма заказа.
     *
     * @return void
     */
    private function updateShippingCost(
        OrderInterface $order,
        string $order_code,
        float $new_total
    ): void {
        $current_order_total = (float) $order->getTotalPrice()->getNumber();
        $difference = $new_total - $current_order_total;

        if ($difference <= 0) {
            return;
        }

        $response = $this->getVtbApi()->getBundle($order_code);
        $delivery_code = $response['order']['deliveryParams']['recipientParams']['deliveryCode'] ?? '';

        if (!empty($delivery_code)) {
            // Удаляем доставки с нулевой стоимостью.
            $this->removeZeroShipments($order);

            // Добавляем новую доставку.
            $this->addShipmentItem(
                $order,
                $delivery_code,
                'Почта РФ',
                $difference
            );

            $order->recalculateTotalPrice();
            $order->save();
        }
    }


    /**
     * Удаляет доставки с нулевой стоимостью.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $order
     *   Объект заказа.
     *
     * @return void
     */
    private function removeZeroShipments(OrderInterface $order): void
    {
        foreach ($order->get('shipments')->referencedEntities() as $shipment) {
            /** @var \Drupal\commerce_shipping\Entity\Shipment $shipment */
            if ((float) $shipment->getAmount()->getNumber() === 0.0) {

                $this->logger->debug(__FUNCTION__ . ' > remove shipment', [
                    'shipment_id' => $shipment->id(),
                ]);

                $shipment->delete();
            }
        }
    }


    /**
     * Добавляет новый элемент доставки.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $order
     *   Объект заказа.
     * @param string $shipping_method
     *   Идентификатор способа доставки (deliveryCode).
     * @param string $shipping_name
     *   Название доставки.
     * @param float $shipping_total
     *   Стоимость доставки.
     *
     * @return void
     */
    private function addShipmentItem(
        OrderInterface $order,
        string $shipping_method,
        string $shipping_name,
        float $shipping_total
    ): void {
        $this->logger->debug(__FUNCTION__ . ' > Shipment create INPUT: ', [
            'shipping_method' => $shipping_method,
            'shipping_name' => $shipping_name,
            'shipping_total' => $shipping_total,
        ]);

        $shipment = Shipment::create([
            'type' => 'default',
            'order_id' => $order->id(),
            'shipping_method' => $shipping_method,
            'amount' => [
                'number' => (string) $shipping_total,
                'currency_code' => $order->getTotalPrice()->getCurrencyCode(),
            ],
            'title' => $shipping_name,
        ]);

        $shipment->save();
        $order->get('shipments')->appendItem($shipment);
    }
}
