<?php

namespace Drupal\commerce_vtbpayment\Traits;

use \Drupal\commerce_vtbpayment\Classes\Api\OrangeDataApi,
    \Drupal\commerce_vtbpayment\Classes\Exception\VtbPayException;

trait OrangeData
{
    /**
     * Обработчик события beforeGetPaymentStatusWebhook
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onBeforeGetPaymentStatusOrangeData(array $vars)
    {
        // Если к вебхуку обращается OrangeData
        $document_id = $vars['php_input']['id'] ?? '';
        if (!empty($document_id)) {
            $this->webhookHandlerOrangeData($document_id);
            die('OK');
        }
    }


    /**
     * Обработчик события paymentPaid
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentPaidOrangeData(array $vars)
    {
        if (
            $this->enable_fiscal &&
            $this->ofd_fiscal === 'orange_data'
        ) $this->sendReceiptOrangeData();
    }


    /**
     * Обработчик события paymentRefunded
     *
     * @param array $vars Переданные переменные
     *
     * @return void
     */
    public function onPaymentRefundedOrangeData(array $vars)
    {
        if (
            $this->enable_fiscal &&
            $this->ofd_fiscal === 'orange_data'
        ) $this->sendReceiptOrangeData(true);
    }


    /**
     * Создаем новый экземпляр класса OrangeDataApi с заданными конфигурациями платёжной системы.
     *
     * @return OrangeDataApi The new OrangeDataApi instance.
     */
    protected function getOrangeDataApi(): OrangeDataApi
    {
        return new OrangeDataApi(
            $this->inn_orange_data_fiscal,
            $this->tax_system_orange_data_fiscal,
            $this->group_orange_data_fiscal,
            $this->signature_key_orange_data_fiscal,
            $this->private_key_orange_data_fiscal,
            $this->client_key_orange_data_fiscal,
            $this->client_crt_orange_data_fiscal,
            $this->ca_cert_orange_data_fiscal,
            $this->cert_password_orange_data_fiscal,
            (bool) $this->test_mode_orange_data_fiscal
        );
    }


    /**
     * Проверяет статус формирования чека в АТОЛ и оповещает при ошибке
     *
     * @param string $document_id Полученные данные
     *
     * @return void
     */
    private function webhookHandlerOrangeData(string $document_id): void
    {
        $response = $this->getOrangeDataApi()->getReceiptInfo($document_id);

        if ($response['status'] === 202) {
            $this->sendReceiptErrorEmail(sprintf(
                  'По заказу №%s процесс формирования чека не закончен. Просьба проверить наличие чека самостоятельно в личном кабинете Orange Data.',
                  $this->order->id()
            ));
        }
        elseif ($response['status'] !== 200) {
            $this->sendReceiptErrorEmail(sprintf(
                  'По заказу №%s возникла ошибка при формировании чека. Просьба сформировать чек самостоятельно в личном кабинете Orange Data.',
                  $this->order->id()
            ));
        }
    }


    /**
     * Отправляет в OrangeData чек прихода или возврата на печать и возвращает результат.
     *
     * @param bool $is_return Данный чек "Возврат прихода" или "Приход"
     *
     * @return array
     */
    private function sendReceiptOrangeData(bool $is_return = false): array
    {
        try {
            $order_id = $this->order->id();

            $email = $this->getCustomerEmail();

            $items = $this->prepareItemsForOrangeData(
                $this->getItems()
            );

            $total = floatval($this->order->getTotalPrice() ? number_format($this->order->getTotalPrice()->getNumber(),2,'.','') : 0);

            $callback_url = $this->getNotifyUrl()->toString();

            return $this->getOrangeDataApi()->sendReceipt(
                $order_id,
                $items,
                $total,
                $email,
                $callback_url,
                $is_return
            );

        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $this->order->id() ?: ''
            ), $context);

            $this->sendReceiptErrorEmail(sprintf(
                'По заказу №%s возникла ошибка при формировании чека. Просьба сформировать чек самостоятельно в личном кабинете Orange Data.',
                $this->order->id()
            ));
            return [];
        }
    }


     /**
      * Сопоставляет внутренние значения с параметрами Orange Data.
      *
      * @param string $setting Тип настройки: 'paymentMethodType', 'tax', 'unitOfMeasurement'
      * @param string $key Ключ настройки
      * @return string Соответствующее значение или пустая строка
      */
    private static function settingsMappingOrangeData(string $setting, string $key): string
    {
        $settings_mapping = [
            'paymentMethodType' => [
                'full_prepayment'   => 1, // Предоплата 100%
                'prepayment'        => 2, // Частичная предоплата
                'advance'           => 3, // Аванс
                'full_payment'      => 4, // Полный расчёт
                'partial_payment'   => 5, // Частичный расчет и кредит
                'credit'            => 6, // Передача в кредит
                'credit_payment'    => 7, // Оплата кредита
            ],
            'tax' => [
                'vat20'   => 1,  // ставка НДС 20%
                'vat10'   => 2,  // ставка НДС 10%
                'vat120'  => 3,  // ставка НДС расч. 20/120
                'vat110'  => 4,  // ставка НДС расч. 10/110
                'vat0'    => 5,  // ставка НДС 0%
                'none'    => 6,  // НДС не облагается
                'vat5'    => 7,  // ставка НДС 5%
                'vat7'    => 8,  // ставка НДС 7%
                'vat105'  => 9,  // ставка НДС расч. 5/105
                'vat107'  => 10, // ставка НДС расч. 7/107
            ],
            'unitOfMeasurement' => [
                0   => 'pcs',    // Штучный товар
                10  => 'g',      // Грамм
                11  => 'kg',     // Килограмм
                12  => 't',      // Тонна
                20  => 'cm',     // Сантиметр
                21  => 'dm',     // Дециметр
                22  => 'm',      // Метр
                30  => 'cm²',    // Квадратный сантиметр
                31  => 'dm²',    // Квадратный дециметр
                32  => 'm²',     // Квадратный метр
                40  => 'ml',     // Миллилитр
                41  => 'l',      // Литр
                42  => 'm³',     // Кубический метр
                50  => 'kWh',    // Киловатт-час
                51  => 'Gcal',   // Гигакалория
                70  => 'd',      // Сутки
                71  => 'h',      // Час
                72  => 'min',    // Минута
                73  => 's',      // Секунда
                80  => 'KB',     // Килобайт
                81  => 'MB',     // Мегабайт
                82  => 'GB',     // Гигабайт
                83  => 'TB',     // Терабайт
                255 => 'other',  // Иная единица измерения
            ],


        ];

        return $settings_mapping[$setting][$key] ?? '';
    }


    /**
     * Подготавливает элементы для АТОЛ.
     *
     * @param array $items Элементы для подготовки.
     *
     * @return array Массив подготовленных элементов.
     */
    private function prepareItemsForOrangeData(array $items): array
    {
        return array_map(function ($item) {
            return [
                'text' => $item['name'], // Лейбл
                'price' => $item['price'], // Цена товара за 1шт
                'totalSum' => $item['amount'], // Итоговая цена (цена * кол-во)
                'quantity' => $item['quantity'], // Кол-во позиций товара (сколько шт)
                'unitOfMeasurement' => self::settingsMappingOrangeData('unitOfMeasurement', $item['measure']), // Единица измерения
                'tax' => self::settingsMappingOrangeData('tax', $item['taxParams']['taxType']), // Налоговая ставка (НДС),
                'paymentMethodType' => self::settingsMappingOrangeData('paymentMethodType', $item['paymentType']), // Способ расчета
                'paymentSubjectType' => $item['paymentSubject'] // Предмет расчета
            ];
        }, $items);
    }
}
