<?php

namespace Drupal\commerce_vtbpayment\Traits;

use \Drupal\file\Entity\File,
    \Drupal\commerce_vtbpayment\Classes\Api\VtbApi,
    \Drupal\commerce_vtbpayment\Classes\Exception\VtbPayException;

trait Common
{
    /**
     * Создаем новый экземпляр класса VtbApi с заданными конфигурациями платёжной системы.
     * @return VtbApi The new VtbApi instance.
     */
    protected function getVtbApi(): VtbApi
    {
        return new VtbApi(
            $this->client_id,
            $this->client_secret,
            (bool) $this->mode,
            $this->merchant_authorization ?: ''
        );
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems(): array
    {
        $order = $this->order ?: $this->entity->getOrder();
        $items = [];
        $key = 1;

        foreach ($order->getItems() as $item) {
            $itemPrice = number_format($item->getUnitPrice()->getNumber(), 2, '.', '');
            $amount = number_format($itemPrice * $item->getQuantity(), 2, '.', '');
            $variation = $item->getPurchasedEntity();
            $code = $variation->getSku() ?: $item->getPurchasedEntityId();

            // Получение массы (веса) товара
            $weight = $variation->get('weight');
            $massUnit = $weight->unit ?? '';
            $massNumber = 0;
            if (in_array($massUnit, ['g', 'kg'], true)) {
                $number = $weight->number ?? 0;
                $massNumber = (int) ($massUnit === 'kg' ? round($number * 1000) : $number);
            }


            $items[] = [
                'positionId' => $key,
                'name' => $item->getTitle(),
                'code' => (string) $code,
                'price' => floatval($itemPrice),
                'measure' => (int) $this->measure_fiscal ?: 0,
                'quantity' => (int)$item->getQuantity(),
                'taxParams' => [
                    'taxType' => $this->tax_type_fiscal ?: 'none'
                ],
                'paymentType' => $this->payment_type_fiscal ?: 'full_prepayment',
                'paymentSubject' => (int) $this->payment_subject_fiscal ?: 1,
                'amount' => floatval($amount),
                'mass' => $massNumber
            ];

            $key++;
        }

        foreach ($order->getAdjustments() as $item) {
            if ($item->getType() === 'shipping') {
                $shippingPrice = floatval($item->getAmount()->getNumber());
                if ($shippingPrice <= 0) continue;

                $items[] = [
                    'positionId' => $key,
                    'name' => $item->getLabel(),
                    'code' => (string) $item->getSourceId() ?: '',
                    'price' => $shippingPrice,
                    'measure' => 0,
                    'quantity' => 1,
                    'taxParams' => [
                        'taxType' => $this->tax_type_fiscal ?: 'none'
                    ],
                    'paymentType' => $this->payment_type_delivery_fiscal ?: 'full_prepayment',
                    'paymentSubject' => 4,
                    'amount' => $shippingPrice
                ];

                $key++;
            }
        }

        return $items;
    }


    /**
     * Получает email клиента, иначе выкидывает ошибку
     */
    private function getCustomerEmail(): string
    {
        $order = $this->order ?: $this->entity->getOrder();
        // Get Email
        $email = $order->getEmail() ?: $this->email_fiscal;
        if (!self::validateEmail($email)) {
            throw new VtbPayException('Указанный адрес электронной почты не прошёл валидацию.', [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Устанавливает свойства настроек
     * @param array $configuration - массив настроек
     */
    private function setConfigurationProps(array $configuration)
    {
        $form_fields = $this->form_fields + [
            'mode' => [
                '#type' => 'radios'
            ]
        ];

        foreach ($form_fields as $field_key => $field_data) {
            switch ($field_data['#type']) {
              case 'textfield':
              case 'select':
                $this->{$field_key} = $configuration[$field_key] ?? '';
                break;
              case 'radios':
                $bool_setting = $configuration[$field_key] ?? false;
                $this->{$field_key} = (bool) ($bool_setting === 'test' || $bool_setting === '1');
                break;
              case 'managed_file':
                $fid_array = $configuration[$field_key] ?? [];
                $fid = $fid_array[0] ?? NULL;
                if ($fid) {
                    $file = File::load($fid);
                    if ($file) {
                        $uri = $file->getFileUri();
                        $this->{$field_key} = \Drupal::service('file_system')->realpath($uri);
                    }
                }
                break;
            }
        }
    }
}
