<?php

namespace Drupal\commerce_vtbpayment\Traits;

use \Drupal\commerce_vtbpayment\Classes\Api\AtolApi,
    \Drupal\commerce_vtbpayment\Classes\Exception\VtbPayException;

trait Atol
{
    /**
     * Обработчик запроса от АТОЛ перед проверкой статуса платежа.
     *
     * @param array $vars
     *   Входящие данные запроса (обычно $_REQUEST или декодированный JSON).
     */
    public function onBeforeGetPaymentStatusAtol(array $vars): void
    {
        $uuid = $vars['php_input']['uuid'] ?? '';
        if (!empty($uuid)) {
            $this->webhookHandlerAtol($uuid);
            exit('OK');
        }
    }


    /**
     * Обработка события об успешной оплате: отправка чека.
     *
     * @param array $vars
     *   Входные данные события.
     */
    public function onPaymentPaidAtol(array $vars): void
    {
        if (
            !empty($this->enable_fiscal) &&
            $this->ofd_fiscal === 'atol'
        ) {
            $this->sendReceiptAtol();
        }
    }


    /**
     * Обработка события возврата платежа: отправка возвратного чека.
     *
     * @param array $vars
     *   Входные данные события.
     */
    public function onPaymentRefundedAtol(array $vars): void
    {
      if (
          !empty($this->enable_fiscal) &&
          $this->ofd_fiscal === 'atol'
      ) {
          $this->sendReceiptAtol(true);
      }
    }


    /**
     * Возвращает инстанс API-клиента Atol.
     *
     * @return \Drupal\commerce_vtbpayment\Classes\Api\AtolApi
     *   Экземпляр API Atol.
     */
    protected function getAtolApi(): AtolApi
    {
        return new AtolApi(
            $this->login_atol_fiscal,
            $this->pass_atol_fiscal,
            $this->inn_atol_fiscal,
            $this->tax_system_atol_fiscal,
            $this->kkt_atol_fiscal,
            $this->email_fiscal,
            (bool) $this->test_mode_atol_fiscal
        );
    }


    /**
     * Обрабатывает результат фискализации из вебхука АТОЛ.
     *
     * @param string $uuid
     *   UUID транзакции чека.
     */
    private function webhookHandlerAtol(string $uuid): void
    {
        $response = $this->getAtolApi()->getReceiptInfo($uuid);
        $order_id = $this->order->id();

        if ($response['status'] === 'fail') {
            $this->sendReceiptErrorEmail("Ошибка при формировании чека для заказа #$order_id. Сформируйте чек вручную в личном кабинете АТОЛ.");
        }
        elseif ($response['status'] === 'wait') {
            $this->sendReceiptErrorEmail("Формирование чека для заказа #$order_id не завершено. Проверьте статус чека в личном кабинете АТОЛ.");
        }
    }


    /**
     * Формирует и отправляет чек в АТОЛ.
     *
     * @param bool $isReturn
     *   TRUE, если это возвратный чек.
     *
     * @return array
     *   Ответ от АТОЛ или пустой массив в случае ошибки.
     */
    private function sendReceiptAtol(bool $isReturn = false): array
    {
        try {
            $order_id = $this->order->id();
            $email = $this->getCustomerEmail();

            $items = $this->prepareItemsForAtol(
                $this->getItems()
            );

            $total = floatval($this->order->getTotalPrice() ? number_format($this->order->getTotalPrice()->getNumber(),2,'.','') : 0);

            $callback_url = $this->getNotifyUrl()->toString();

            return $this->getAtolApi()->sendReceipt(
                $order_id,
                $items,
                $total,
                $email,
                $callback_url,
                $isReturn
            );
        }
        catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $e->getMessage(),
                $this->order->id() ?: ''
            ), $context);

            $this->sendReceiptErrorEmail("Ошибка при формировании чека для заказа #" . $this->order->id());
            return [];
        }
    }


    /**
     * Подготавливает позиции товаров для отправки в АТОЛ.
     *
     * @param array $items
     *   Список товаров.
     *
     * @return array
     *   Отформатированные данные для API АТОЛ.
     */
    private function prepareItemsForAtol(array $items): array
    {
        return array_map(function ($item) {
            return [
                'name' => $item['name'],
                'price' => $item['price'],
                'sum' => $item['amount'],
                'quantity' => $item['quantity'],
                'measure' => $item['measure'],
                'vat' => [
                    'type' => $item['taxParams']['taxType'] ?? 'none',
                ],
                'payment_method' => $item['paymentType'],
                'payment_object' => $item['paymentSubject'],
            ];
        }, $items);
    }
}
