<?php

namespace Drupal\commerce_vtbpayment\PluginForm\OffsiteRedirect;

require dirname(__DIR__, 3) . '/vendor/autoload.php';

use \Drupal\commerce_payment\PluginForm\PaymentOffsiteForm as BasePaymentOffsiteForm,
    \Drupal\Core\Form\FormStateInterface,
    \Drupal\commerce_vtbpayment\Classes\Common\VtbPayLogger,
    \Drupal\commerce_vtbpayment\Classes\Exception\VtbPayException,
    \Drupal\commerce_vtbpayment\Traits\Common,
    \Drupal\commerce\Response\NeedsRedirectException,
    \Symfony\Component\Dotenv\Dotenv;

class VtbPaymentOffsiteForm extends BasePaymentOffsiteForm
{
    use Common;

    const CMS_NAME = 'Drupal';
    const PLUGIN_VERSION = '1.6.5';

    private string $client_id;
    private string $client_secret;
    private string $merchant_authorization;
    private bool $mode;
    private bool $logging;
    private bool $two_stage;
    private string $return_url;

    // Fiscal settings
    private bool $enable_fiscal;
    private string $email_fiscal;
    private string $payment_type_delivery_fiscal;
    private string $measure_fiscal;
    private string $tax_type_fiscal;
    private string $payment_type_fiscal;
    private string $payment_subject_fiscal;

    // Fiscal settings ATOL
    private bool $test_mode_atol_fiscal;
    private string $login_atol_fiscal;
    private string $pass_atol_fiscal;
    private string $kkt_atol_fiscal;
    private string $inn_atol_fiscal;
    private string $tax_system_atol_fiscal;

    // Fiscal settings Orange Data
    private bool $test_mode_orange_data_fiscal;
    private string $signature_key_orange_data_fiscal;
    private string $private_key_orange_data_fiscal;
    private string $client_key_orange_data_fiscal;
    private string $client_crt_orange_data_fiscal;
    private string $ca_cert_orange_data_fiscal;
    private string $cert_password_orange_data_fiscal;
    private string $group_orange_data_fiscal;
    private string $inn_orange_data_fiscal;
    private string $tax_system_orange_data_fiscal;

    // Common
    private array $form_fields;
    private object $logger;

    /**
     * Устанавливает настройки из файла .env
     */
    public function __construct()
    {
        $this->loadEnvFiles([
            dirname(__DIR__, 3) . '/config/.env',        // обязательный
            dirname(__DIR__, 3) . '/config/.env.custom', // опциональный
        ]);

        $this->form_fields = include dirname(__DIR__, 3) . '/config/form_fields.php';

        $this->setVtbpayLogger();
    }


    /**
     * Загружает .env файлы.
     *
     * Первый файл обязателен, остальные — опциональные.
     *
     * @param array $files Пути к .env файлам, в порядке приоритета.
     * @return void
     * @throws \RuntimeException Если первый файл отсутствует.
     */
    private function loadEnvFiles(array $files): void
    {
        $dotenv = new Dotenv();

        foreach ($files as $index => $file) {
            if (is_readable($file)) {
                $dotenv->load($file);
            } elseif ($index === 0) {
                throw new \RuntimeException("Обязательный .env файл не найден: {$file}");
            }
        }
    }


    /**
     * Формирует запрос и перенаправление на платежную страница.
     * @param array $form - массив формы оплаты
     * @param object $form_state - объект формы оплаты
     * @return array|null Массив, содержащий результат и URL перенаправления, или null, если произошла ошибка.
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
        $form = parent::buildConfigurationForm($form, $form_state);
        $payment = $this->entity;
        $order = $payment->getOrder();

        $this->logger->setOption('additionalCommonText', 'payment-' . rand(1111, 9999));

        try {
            $plugin = $payment->getPaymentGateway()->getPlugin();
            $configuration = $plugin->getConfiguration();
            $this->setConfigurationProps($configuration);

            return $this->buildRedirectForm(
                $form,
                $form_state,
                $this->getPayUrl($form),
                [],
                self::REDIRECT_GET
            );
        } catch (\Exception | VtbPayException $e) {
            if ($e instanceof NeedsRedirectException) {
                throw $e;
            }
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $msg = $e->getMessage() ?: 'Ошибка генерации ссылки оплаты.';

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $msg,
                $order->id() ?: ''
            ), $context);
            \Drupal::messenger()->addError($msg);
            return null;
        }
    }


    /**
     * Получает URL-адрес платежа для перенаправления.
     * @param array $form - массив формы оплаты для получения адреса возврата
     * @return string URL-адрес платежа.
     */
    private function getPayUrl($form): string
    {
        $order = $this->entity->getOrder();
        $order_id = $order->id();
        $email = $this->getCustomerEmail();
        $total = floatval($order->getTotalPrice() ? number_format($order->getTotalPrice()->getNumber(),2,'.','') : 0);

        // Get return URL
        $return_url = $form['#return_url'];
        if (
            isset($this->return_url) &&
            !empty($this->return_url) &&
            isset($_ENV['SHOW_RETURN_URL_FIELD']) &&
            $_ENV['SHOW_RETURN_URL_FIELD'] === '1'
        ) {
            $return_url = self::getReturnUrl(
                $order_id,
                $this->return_url
            );
        }

        $items = [];
        if (
            !$this->two_stage &&
            $this->enable_fiscal &&
            $this->ofd_fiscal === '1ofd'
        ) $items = $this->getItems();

        $response = $this->getVtbApi()->getOrderLink(
            $order_id,
            $email,
            time(),
            $total,
            $return_url,
            $this->two_stage,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Подготавливает кастомный Return Url
     * @param string $order_id
     * @param string $return_url
     * @return string
     */
    private static function getReturnUrl(string $order_id, string $return_url): string
    {
        $protocol = (empty($_SERVER['HTTPS']) ? 'http' : 'https');
        $host = $_SERVER['HTTP_HOST'];
        $order_page = str_replace('#ID#', $order_id, $return_url);

        return "{$protocol}://{$host}{$order_page}";
    }


    /**
     * Инициализация и настройка объекта класса VtbPayLogger.
     * Эта функция инициализирует и настраивает логгер, используемый плагином VtbPay для ведения журнала.
     * @return void
     */
    private function setVtbpayLogger(): void
    {
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $logger = VtbPayLogger::getInstance();
        $this->logger = $logger
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitive_data_keys)
            ->setLogFilePath(dirname(__DIR__, 3) . '/logs/vtbpay-' . date('d-m-Y') . '.log')
            ->setCustomRecording(function($message) use ($logger) {
                if ($this->logging) {
                    $logger->writeToFile($message);
                }
            }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
