<?php

namespace Drupal\commerce_vtbpayment\Plugin\Commerce\PaymentGateway;

require dirname(__DIR__, 4) . '/vendor/autoload.php';

use \Drupal\Core\Render\Markup,
    \Drupal\Core\Entity\EntityTypeManagerInterface,
    \Drupal\Core\Form\FormStateInterface,
    \Drupal\Component\Datetime\TimeInterface,
    \Drupal\file\Entity\File,
    \Drupal\commerce_order\Entity\OrderInterface,
    \Drupal\commerce_payment\Exception\PaymentGatewayException,
    \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase,
    \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsNotificationsInterface,
    \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface,
    \Drupal\commerce_payment\Entity\PaymentInterface,
    \Drupal\commerce_payment\PaymentMethodTypeManager,
    \Drupal\commerce_payment\PaymentTypeManager,
    \Drupal\commerce_price\Price,
    \Drupal\commerce_vtbpayment\Classes\Common\VtbPayLogger,
    \Drupal\commerce_vtbpayment\Classes\Common\EventDispatcher,
    \Drupal\commerce_vtbpayment\Classes\Exception\VtbPayException,
    \Drupal\commerce_vtbpayment\Traits\Atol,
    \Drupal\commerce_vtbpayment\Traits\OrangeData,
    \Drupal\commerce_vtbpayment\Traits\Shipping,
    \Drupal\commerce_vtbpayment\Traits\Common,
    \Symfony\Component\HttpFoundation\Request,
    \Symfony\Component\HttpFoundation\Response,
    \Symfony\Component\Dotenv\Dotenv;

/**
 * Provides the Off-site Vtb payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "commerce_vtbpayment",
 *   label = "VtbPay",
 *   display_label = "VtbPay",
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_vtbpayment\PluginForm\OffsiteRedirect\VtbPaymentOffsiteForm",
 *   },
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "MIR", "mastercard", "visa",
 *   },
 *   requires_billing_information = FALSE,
 * )
 */
class Vtbpayment extends OffsitePaymentGatewayBase implements SupportsNotificationsInterface, SupportsRefundsInterface
{
    use Atol, OrangeData, Shipping, Common;

    private string $client_id;
    private string $client_secret;
    private string $merchant_authorization;
    private bool $mode;
    private bool $logging;
    private bool $two_stage;

    // Fiscal settings
    private bool $enable_fiscal;
    private string $email_fiscal;
    private string $payment_type_delivery_fiscal;
    private string $measure_fiscal;
    private string $tax_type_fiscal;
    private string $payment_type_fiscal;
    private string $payment_subject_fiscal;

    // Fiscal settings ATOL
    private bool $test_mode_atol_fiscal;
    private string $login_atol_fiscal;
    private string $pass_atol_fiscal;
    private string $kkt_atol_fiscal;
    private string $inn_atol_fiscal;
    private string $tax_system_atol_fiscal;

    // Fiscal settings Orange Data
    private bool $test_mode_orange_data_fiscal;
    private string $signature_key_orange_data_fiscal;
    private string $private_key_orange_data_fiscal;
    private string $client_key_orange_data_fiscal;
    private string $client_crt_orange_data_fiscal;
    private string $ca_cert_orange_data_fiscal;
    private string $cert_password_orange_data_fiscal;
    private string $group_orange_data_fiscal;
    private string $inn_orange_data_fiscal;
    private string $tax_system_orange_data_fiscal;

    // Common
    private OrderInterface $order;
    private array $form_fields;
    private object $logger;
    private EventDispatcher $event_dispatcher;


    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityTypeManagerInterface $entity_type_manager,
        PaymentTypeManager $payment_type_manager,
        PaymentMethodTypeManager $payment_method_type_manager,
        TimeInterface $time
    ) {
        parent::__construct(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $entity_type_manager,
            $payment_type_manager,
            $payment_method_type_manager,
            $time
        );

        $this->loadEnvFiles([
            dirname(__DIR__, 4) . '/config/.env',        // обязательный
            dirname(__DIR__, 4) . '/config/.env.custom', // опциональный
        ]);

        $this->event_dispatcher = new EventDispatcher($this); // Инициализация менеджера событий

        $this->form_fields = include dirname(__DIR__, 4) . '/config/form_fields.php';

        $this->setConfigurationProps($configuration);

        $this->setVtbpayLogger();
    }


    /**
     * Загружает .env файлы.
     *
     * Первый файл обязателен, остальные — опциональные.
     *
     * @param array $files Пути к .env файлам, в порядке приоритета.
     * @return void
     * @throws \RuntimeException Если первый файл отсутствует.
     */
    private function loadEnvFiles(array $files): void
    {
        $dotenv = new Dotenv();

        foreach ($files as $index => $file) {
            if (is_readable($file)) {
                $dotenv->load($file);
            } elseif ($index === 0) {
                throw new \RuntimeException("Обязательный .env файл не найден: {$file}");
            }
        }
    }


    /**
     * Дефолтные значения формы настроек модуля.
     * @return array - возвращает дефолные значения
     */
    public function defaultConfiguration()
    {
        $form_fields = include dirname(__DIR__, 4) . '/config/form_fields.php';
        $defaults = array_map(fn($item) => $item['#default_value'] ?? '', $form_fields);
        return $defaults + parent::defaultConfiguration();
    }


    /**
     * Создание формы настроек модуля.
     * @param array $form - массив параметров настроек
     * @param object $form_state - объект формы настроек
     * @return void
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $private_path = \Drupal::service('file_system')->realpath('private://');
        if (!$private_path) {
            \Drupal::messenger()->addWarning('Не задан путь к приватным файлам. Файлы сертификатов будут сохранены в публичной директории (небезопасно).');
        }
        $form = parent::buildConfigurationForm($form, $form_state);
        $form['#attached']['library'][] = 'commerce_vtbpayment/admin_js';
        return $form + $this->form_fields;
    }


    /**
     * Сохранение параметров настроек модуля.
     * @param array $form - массив параметров настроек
     * @param object $form_state - объект формы настроек
     * @return void
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        parent::submitConfigurationForm($form, $form_state);

        if (!$form_state->getErrors()) {
            $values = $form_state->getValue($form['#parents']);
            foreach ($this->form_fields as $field_key => $field_data) {
                switch ($field_data['#type']) {
                  case 'textfield':
                  case 'select':
                    $this->configuration[$field_key] = $values[$field_key] ?? '';
                    break;
                  case 'radios':
                    $this->configuration[$field_key] = $values[$field_key] ?? false;
                    break;
                  case 'managed_file':
                    $fid_array = $values[$field_key] ?? [];
                    $fid = $fid_array[0] ?? NULL;
                    if ($fid) {
                        $file = File::load($fid);
                        if ($file) {
                            $file->setPermanent();
                            $file->save();
                            \Drupal::service('file.usage')->add($file, 'commerce_vtbpayment', 'config', 1);
                            $this->configuration[$field_key] = [$fid];
                        }
                    }
                    break;
                }
            }
        }
    }


    /**
     * Обработчик cancel, вызываемый при переходе на страницу cancel_url, после попытки оплаты.
     * Показывает сообщение что заказ был отклонен
     * @param object $order - объект заказа
     * @param object $request - объект запроса к странице возврата
     * @return void
     */
    public function onCancel(OrderInterface $order, Request $request)
    {
        \Drupal::messenger()->addMessage($this->t(
            'Вы отменили оплату заказа с помощью @gateway,
             но можете возобновить процесс оформления заказа здесь, когда будете готовы.', [
            '@gateway' => $this->getDisplayLabel(),
        ]));
        throw new PaymentGatewayException('Payment cancelled by user');
    }


    /**
     * Обработчик return, вызываемый при переходе на страницу return_url, после попытки оплаты.
     * Добавляет информацию о платеже в заказ
     * @param object $order - объект заказа
     * @param object $request - объект запроса к странице возврата
     * @return void
     */
    public function onReturn(OrderInterface $order, Request $request)
    {
        $this->order = $order;
        $this->logger->setOption('additionalCommonText', 'return-' . rand(1111, 9999));
        // Logging $_REQUEST
        $this->logger->debug(
            __FUNCTION__ . ' > return: ', [
            'request_data' => $_REQUEST
        ]);

        try {
            $payment_status_data = $this->getPaymentStatusData();

            $this->event_dispatcher->dispatch('afterGetPaymentStatusReturn', [
                'payment_status_data' => $payment_status_data
            ]);

            $payment_status = $payment_status_data['object']['status']['value'] ?? '';

            $result_change_status = $this->changeStatus($payment_status);

            \Drupal::messenger()->addMessage($this->t(
                $result_change_status['message'], [
                '@orderid' => $this->order->id()
            ]));
        } catch (\Exception | VtbPayException $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            $msg = $e->getMessage() ?: 'Ошибка проверки статуса оплаты.';

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $msg,
                $this->order->id() ?: ''
            ), $context);

            throw new PaymentGatewayException($e->getMessage());
        }
    }


    /**
     * Обработчик notify, вызываемый в качестве вебхука, после попытки оплаты.
     * Может использоваться при возврате платежа
     * @param object $request - объект запроса к вебхуку
     * @return string добавляет статус ответа
     */
    public function onNotify(Request $request)
    {
        $this->logger->setOption('additionalCommonText', 'webhook-' . rand(1111, 9999));

        if ($request->getContentTypeFormat() === 'json') {
            $php_input = json_decode($request->getContent(), true);
        } else {
            $php_input = json_decode(file_get_contents('php://input'), true); // fallback
        }

        // Логируем весь входящий запрос.
        $this->logger->debug(__FUNCTION__ . ' > notify: ', [
            'php_input' => $php_input,
        ]);

        try {
            // Получаем идентификатор заказа из запроса.
            $order_id = explode(
                '-',
                $php_input['object']['orderId'] ?? $php_input['external_id'] ?? $php_input['id']
            )[0];

            if (empty($order_id)) {
                throw new \InvalidArgumentException('Не передан параметр orderId или orderNumber в уведомлении.');
            }

            /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
            $order = $this->entityTypeManager
                ->getStorage('commerce_order')
                ->load($order_id);

            if (!$order) {
                throw new VtbPayException('Заказ не найден: ', [
                    'order_id' => $order_id
                ]);
            }

            $this->order = $order;

            $this->event_dispatcher->dispatch('beforeGetPaymentStatusWebhook', [
                'php_input' => $php_input
            ]);

            // Запрашиваем статус оплаты с сервера банка.
            $payment_status_data = $this->getPaymentStatusData();

            $this->event_dispatcher->dispatch('afterGetPaymentStatusWebhook', [
                'payment_status_data' => $payment_status_data
            ]);

            $payment_status = $payment_status_data['object']['status']['value'] ?? '';

            $result_change_status = $this->changeStatus($payment_status);

            if (in_array($result_change_status['state'], ['completed', 'pending', 'refunded'])) {
                // Возвращаем 200 OK как ответ.
                return new Response('OK', 200);
            }
            else {
                // Возвращаем 500 ERROR как ответ.
                return new Response('ERROR', 500);
            }
        } catch (\Exception | VtbPayException $e) {
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) {
                $context = array_merge($e->getContext(), $context);
            }

            $msg = $e->getMessage() ?: 'Ошибка при обработке уведомления.';

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > Exception: %s', $msg
            ), $context);

            die($e->getMessage());
        }
    }


    /**
     * Получение статуса заказа из VTB API.
     * @return array Данные статуса оплаты
     */
    private function getPaymentStatusData(): array
    {
        $response = $this->getVtbApi()->getOrderInfo($this->order->id());

        return $response ?? [];
    }


    /**
     * Меняет статус заказа и обновляет/создаёт сущность commerce_payment.
     *
     * @param string $payment_status - Новый статус платежа (например, 'PAID', 'PENDING', 'REFUNDED').
     *
     * @return array
     *   Массив с ключами: 'state' — новый статус заказа, 'message' — текстовое сообщение для пользователя.
     */
    private function changeStatus(string $payment_status): array
    {
        $available_statuses = [
            'PAID' => [
                'state' => 'completed',
                'message' => 'Ваш платеж прошел успешно для заказа @orderid',
                'fnc_first' => function() {
                    $this->event_dispatcher->dispatch('paymentPaid');
                }
            ],
            'PENDING' => [
                'state' => 'pending',
                'message' => 'Ваш платеж для заказа @orderid ожидает обработки',
            ],
            'REFUNDED' => [
                'state' => 'refunded',
                'message' => 'Платеж по заказу @orderid был полностью возвращен',
                'fnc_first' => function() {
                    $this->event_dispatcher->dispatch('paymentRefunded');
                }
            ]
        ];

        $order_status_data = $available_statuses[$payment_status] ?? [
            'state' => 'canceled',
            'message' => 'Платеж по заказу @orderid был отменён'
        ];

        if ($this->order->getState()->getId() !== $order_status_data['state']) {
            $storage = $this->entityTypeManager->getStorage('commerce_payment');
            // Пытаемся найти существующий платеж по order_id.
            $existing_payments = $storage->loadByProperties(['order_id' => $this->order->id()]);
            $payment = reset($existing_payments);

            if ($payment) {
                // Обновляем существующий платеж.
                $payment->set('state', $order_status_data['state']);
                $payment->set('remote_state', $payment_status);
                $payment->set('amount', $this->order->getBalance());
                $payment->save();
            }
            else {
                // Создаём новый платеж.
                $payment_data = [
                    'state' => $order_status_data['state'],
                    'amount' => $this->order->getBalance(),
                    'payment_gateway' => $this->parentEntity->id(),
                    'order_id' => $this->order->id(),
                    'remote_state' => $payment_status,
                ];

                $storage->create($payment_data)->save();
            }

            if (isset($order_status_data['fnc_first'])) $order_status_data['fnc_first']();
        }

        return $order_status_data;
    }


    /**
     * Проверяет, поддерживает ли платёжный шлюз операции возврата.
     *
     * @return bool
     *   TRUE, если шлюз поддерживает возвраты, иначе FALSE.
     */
    public function supportsRefunds() {
        return true; // Включаем поддержку возвратов
    }


    /**
     * Выполняет возврат платежа через API ВТБ и обновляет сущность commerce_payment.
     *
     * @param \Drupal\commerce_payment\Entity\PaymentInterface $payment
     *   Объект платежа, по которому нужно выполнить возврат.
     * @param \Drupal\commerce_price\Price|null $amount
     *   Сумма возврата. Если NULL, используется сумма всего платежа.
     *
     * @throws \Drupal\commerce_payment\Exception\PaymentGatewayException
     *   В случае ошибки при обращении к API ВТБ или обновлении данных платежа.
     */
    public function refundPayment(PaymentInterface $payment, Price $amount = NULL)
    {
        $this->order = $payment->getOrder();
        $refund_amount  = $amount ?: $payment->getAmount();
        $this->logger->setOption('additionalCommonText', 'refund-' . rand(1111, 9999));

        try {
            // вызываем API ВТБ на возврат
            $response = $this->getVtbApi()->setRefunds(
                $this->order->id(),
                $refund_amount->getNumber()
            );

            $payment->setRefundedAmount($refund_amount);
            $payment->setState('refunded');
            $payment->save();

            $this->event_dispatcher->dispatch('paymentRefunded', [
                'items' => $items
            ]);
        } catch (\Exception | VtbPayException | PaymentGatewayException $e) {
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) {
                $context = array_merge($e->getContext(), $context);
            }

            $msg = $e->getMessage() ?: 'Ошибка при возврате оплаты.';

            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
                $msg,
                $this->order->id() ?: ''
            ), $context);

            throw new PaymentGatewayException($e->getMessage());
        }
    }


    /**
     * Отправляет email при ошибке фискализации.
     *
     * @param string $message - Текст сообщения.
     */
    private function sendReceiptErrorEmail(string $message): void
    {
        $siteName = \Drupal::config('system.site')->get('name') ?? 'Drupal';
        $email = $this->email_fiscal;
        $subject = "Ошибка формирования чека: $siteName";

        $mailManager = \Drupal::service('plugin.manager.mail');
        $params = [
            'subject' => $subject,
            'message' => Markup::create(nl2br(htmlspecialchars($message))),
        ];

        $mailManager->mail(
            'commerce_vtbpayment',
            'fiscal_receipt_error',
            $email,
            \Drupal::currentUser()->getPreferredLangcode(),
            $params,
            NULL,
            TRUE
        );

        $this->logger->debug(
            __FUNCTION__ . ' > mail: ', [
            'email_fiscal' => $email,
            'subject' => $subject,
            'message' => $message
        ]);
    }


    /**
     * Инициализация и настройка объекта класса VtbPayLogger.
     * Эта функция инициализирует и настраивает логгер, используемый плагином VtbPay для ведения журнала.
     * @return void
     */
    private function setVtbpayLogger(): void
    {
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];

        $logger = VtbPayLogger::getInstance();
        $this->logger = $logger
            ->setOption('showBacktrace', true)
            ->setOption('sensitiveDataKeys', $sensitive_data_keys)
            ->setLogFilePath(dirname(__DIR__, 4) . '/logs/vtbpay-' . date('d-m-Y') . '.log')
            ->setCustomRecording(function($message) use ($logger) {
                if ($this->logging) {
                    $logger->writeToFile($message);
                }
            }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
