<?php

namespace Drupal\commerce_vtbpayment\Classes\Common;

use Drupal\commerce_vtbpayment\Classes\Exception\VtbPayException;

/**
 * Класс EventDispatcher реализует регистрацию и вызов обработчиков событий,
 * определённых в конфигурационном файле.
 *
 * Конфигурационный файл должен находиться в папке conf и возвращать массив вида:
 * [
 *     'beforeInit' => ['checkAuth', 'preparePayment'],
 *     'afterInit'  => ['logAuth', 'finalizePayment'],
 * ]
 *
 * Обратите внимание, что методы-обработчики должны быть публичными, иначе
 * они могут существовать, но не быть вызываемыми.
 *
 * @package Vtbpay\Classes\Common
 */
class EventDispatcher
{
    /**
     * Массив слушателей событий.
     *
     * Ключ — имя события, значение — массив имен методов-обработчиков.
     *
     * @var array<string, array<string>>
     */
    protected array $listeners = [];

    /**
     * Объект, для которого регистрируются обработчики.
     *
     * @var object
     */
    protected object $target;

    /**
     * Конструктор.
     *
     * @param object $target Объект, содержащий методы-обработчики событий.
     */
    public function __construct(object $target)
    {
        $this->target = $target;
        $this->registerEventListeners();
    }

    /**
     * Регистрирует обработчики событий, загружая конфигурацию из файла.
     *
     * Путь к файлу конфигурации рассчитывается как:
     * dirname(__DIR__, 3) . '/conf/event_listeners.php'
     *
     * Конфигурация должна быть массивом, где ключи — имена событий,
     * а значения — массивы имен методов-обработчиков.
     *
     * @throws VtbPayException Если конфигурация имеет неверный формат.
     *
     * @return void
     */
    protected function registerEventListeners(): void
    {
        // Формирование пути к файлу конфигурации (например, /conf/event_listeners.php)
        $configFile = dirname(__DIR__, 3) . '/config/event_listeners.php';
        if (!file_exists($configFile)) {
            throw new VtbPayException('Не найден файл конфигурации событий.', [
                'configFile' => $configFile
            ]);
        }

        $listeners = include $configFile;
        if (!is_array($listeners)) {
            throw new VtbPayException('Конфигурационный файл event_listeners должен возвращать массив.', [
                'listeners' => $listeners
            ]);
        }

        foreach ($listeners as $event => $methods) {
            if (!is_array($methods)) {
                throw new VtbPayException('Значение для события должно быть массивом.', [
                    'event' => $event
                ]);
            }
            foreach ($methods as $method) {
                // Проверяем, что метод существует и является вызываемым
                if (method_exists($this->target, $method) && is_callable([$this->target, $method])) {
                    $this->listeners[$event][] = $method;
                }
            }
        }
    }

    /**
     * Вызывает все обработчики для указанного события.
     *
     * @param string $event Имя события, для которого вызываются обработчики.
     * @param array $vars Массив параметров, передаваемых в обработчики.
     *
     * @return void
     */
    public function dispatch(string $event, array $vars = []): void
    {
        if (!empty($this->listeners[$event])) {
            foreach ($this->listeners[$event] as $method) {
                $this->target->$method($vars);
            }
        }
    }
}
