(function ($, Drupal, once) {
  const prefix = '#edit-configuration-commerce-vtbpayment-';

  // Список доступных ОФД с их постфиксом и обязательными полями
  const ofdList = {
    atol: {
      postfix: '_atol_fiscal',
      requiredFields: [
          'login-atol-fiscal',
          'pass-atol-fiscal',
          'kkt-atol-fiscal',
          'inn-atol-fiscal'
      ]
    },
    orange_data: {
      postfix: '_orange_data_fiscal',
      requiredFields: [
          'test-mode-orange-data-fiscal',
          'signature-key-orange-data-fiscal',
          // 'private-key-orange-data-fiscal',
          // 'client-key-orange-data-fiscal',
          // 'client-crt-orange-data-fiscal',
          // 'ca-cert-orange-data-fiscal',
          'cert-password-orange-data-fiscal',
          'group-orange-data-fiscal',
          'inn-orange-data-fiscal',
          'tax-system-orange-data-fiscal'
      ]
    }
  };

  /**
   * Устанавливает или снимает флаг "обязательное" для полей, связанных с ОФД.
   * @param {string} ofdCode - Код ОФД (например, 'atol').
   * @param {boolean} isRequired - Установить поля как обязательные или нет.
   */
  function setRequiredFields(ofdCode, isRequired) {
    const fields = ofdList[ofdCode]?.requiredFields || [];

    fields.forEach(field => {
      const fieldSelector = prefix + field;
      const $field = $(fieldSelector);
      $field.prop('required', isRequired);

      const $label = $(`label[for="${$field.attr('id')}"]`);
      $label.toggleClass('js-form-required form-required', isRequired);
    });
  }

  /**
   * Показывает или скрывает все элементы, связанные с указанным ОФД.
   * @param {string} ofdCode - Код ОФД.
   * @param {boolean} isVisible - Показать или скрыть элементы.
   */
  function toggleFieldVisibilityByPostfix(ofdCode, isVisible) {
    const postfix = ofdList[ofdCode]?.postfix;
    if (!postfix) return;

    // Скрыть/показать поля (input и select)
    $(`input[name*="${postfix}"], select[name*="${postfix}"]`).each(function () {
      const type = $(this).attr('type');

      if (type === 'radio') {
        $(this).closest('fieldset.js-form-item').toggle(isVisible);
      } else if (type === 'file') {
        $(this)
          .closest('div.js-form-item')       // сначала до js-form-item
          .parent('[id^="ajax-wrapper"]')    // затем до родителя с id, начинающимся на ajax-wrapper
          .toggle(isVisible);
      } else {
        // для type="text" и select
        $(this).closest('div.js-form-item').toggle(isVisible);
      }
    });
  }

  /**
   * Обработчик смены ОФД в выпадающем списке.
   */
  function handleOfdChange() {
    const selectedOfd = $(`${prefix}ofd-fiscal`).val();

    Object.keys(ofdList).forEach(ofdCode => {
      const isCurrent = ofdCode === selectedOfd;
      toggleFieldVisibilityByPostfix(ofdCode, isCurrent);
      setRequiredFields(ofdCode, isCurrent);
    });
  }

  /**
   * Проверяет, включён ли тестовый режим одновременно для платёжной системы и ОФД.
   * Если только для одного компонента — выводит предупреждение.
   */
  function checkTestModeMismatch() {
    const selectedOfd = $(`${prefix}ofd-fiscal`).val();
    if (!selectedOfd || !ofdList[selectedOfd]) return;

    // Тестовый режим платёжной системы: radio с name="...mode"
    const psMode = $('input[name="configuration[commerce_vtbpayment][mode]"]:checked').val();
    const isTestPS = psMode === 'test';

    // Тестовый режим ОФД: radio с name="...test_mode_${ofd}_fiscal"
    const ofdRadioName = `configuration[commerce_vtbpayment][test_mode_${selectedOfd}_fiscal]`;
    const ofdMode = $(`input[name="${ofdRadioName}"]:checked`).val();
    const isTestOfd = ofdMode === '1';

    // Проверка несовпадения
    if ((isTestPS || isTestOfd) && !(isTestPS && isTestOfd)) {
      alert('Обратите внимание: включен тестовый режим только у одного компонента — платежной системы или ОФД.');
    }
  }

  /**
   * Инициализирует JS-логику формы при загрузке страницы.
   * @param {object} context - Контекст Drupal (DOM).
   */
  function init(context) {
    let $fiscalDropdown = $(`${prefix}ofd-fiscal`, context);
    if (!$fiscalDropdown.length) {
      $fiscalDropdown = $(`${prefix}ofd-fiscal`);
    }

    let $saveButton = $('input[name="op"]', context);
    if (!$saveButton.length) {
      $saveButton = $('input[name="op"]');
    }

    // Прекращаем выполнение, если выпадающий список ОФД не найден
    if (!$fiscalDropdown.length) return;

    // Назначаем обработчики, используя Drupal.once
    once('vtbpayOfdChange', $fiscalDropdown).forEach(el => {
      $(el).on('change', handleOfdChange);
    });

    once('vtbpaySaveClick', $saveButton).forEach(el => {
      $(el).on('click', checkTestModeMismatch);
    });

    // Первичная инициализация
    handleOfdChange();
  }

  /**
   * Drupal behavior, срабатывает при загрузке админ-формы настроек платёжной системы.
   */
  Drupal.behaviors.vtbpayAdminForm = {
    attach: function (context) {
      init(context);
    }
  };

})(jQuery, Drupal, window.once);
