<?php
// Preventing direct access to the script, because it must be included by the "include" directive.
defined('BOOTSTRAP') or die('Access denied');

use \Tygh\Enum\OrderStatuses,
    \Tygh\Registry,
    \Src\Classes\Api\VtbApi,
    \Src\Classes\Common\VtbPayLogger,
    \Src\Classes\Exception\VtbPayException,
    \Src\Traits\VtbPay,
    \Symfony\Component\Dotenv\Dotenv;


class VtbPayments
{
    use VtbPay;

    const CMS_NAME = 'CS-Cart';
    const PLUGIN_VERSION = '1.5.11';

    private string $order_id;
    private string $order_id_and_amount;
    private array $order_info;
    private array $processor_params;
    public string $mode;
    private VtbPayLogger $logger;


    public function __construct(
        string $order_id,
        array $order_info,
        array $processor_data,
        string $mode
    ) {
        $this->mode = $mode ?: '';

        // Получает настройки из .env в корне
        $env_path = dirname(__DIR__) . '/config/.env';
        if (file_exists($env_path)) (new Dotenv())->load($env_path);

        $this->setVtbpayLogger();

        $this->initProperties(
            $order_id,
            $order_info,
            $processor_data
        );
    }


    /**
     * Инициализация свойств, обязательных для обработки платежа
     *
     * @param string  $order_id        The order ID.
     * @param array   $order_info      Order information, including order details.
     * @param array   $processor_data  Processor data, including payment parameters.
     *
     * @return void
     */
    private function initProperties(
        string $order_id,
        array $order_info,
        array $processor_data
    ) {
        try {
            // Устанавливаем order_id_and_amount
            $this->setOrderIdAndAmount();

            // Устанавливаем order_id
            $this->setOrderId($order_id, $order_info);

            $this->setOrderInfo($order_info);

            $this->setProcessorParams($processor_data);

            // Logging request data
            if (in_array($this->mode, ['return', 'webhook'])) $this->logger->debug(
                __FUNCTION__ . ' > ' . $this->mode . ': ',
                ($this->mode === 'return') ? ['request_data' => $_REQUEST] : ['php_input' => json_decode(file_get_contents('php://input'), true)]
            );

        } catch (\Exception $e) {
            // Handle exception and log error
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay exception : %s;',
                $e->getMessage()
            ));

            throw $e;
        }
    }


    /**
     * Получаем и устанавливаем свойство order_id_and_amount из $_REQUEST или из php://input.
     *
     * @return void
     */
    private function setOrderIdAndAmount(): void
    {
        if ($this->mode === 'return') {
            $this->order_id_and_amount = $_REQUEST['orderId'] ?? '';
        }
        elseif ($this->mode === 'webhook') {
            $this->order_id_and_amount = json_decode(file_get_contents('php://input'), true)['object']['orderId'] ?? '';
        }
    }


    /**
     * Получаем и устанавливаем свойство order_id.
     *
     * @param string  $order_id    The order ID.
     * @param array   $order_info  Order information, including order details.
     *
     * @return void
     */
    private function setOrderId(string $order_id, array $order_info): void
    {
        $this->order_id = $order_id ?: '';
        if (empty($this->order_id)) {
            if (!empty($this->order_id_and_amount)) {
                $this->order_id = explode('-', $this->order_id_and_amount)[0] ?? '';
            }
            elseif (!empty($order_info)) {
                $this->order_id = $order_info['order_id'] ?? '';
            }
        }

        if (empty($this->order_id)) {
            throw new \Exception(__('vtbpayments_order_id_not_found'));
        }
    }


    /**
     * Получаем и устанавливаем свойство order_info.
     *
     * @param array $order_info  Order information, including order details.
     *
     * @return void
     * @throws Exception
     */
    private function setOrderInfo(array $order_info): void
    {
        $this->order_info = $order_info ?: fn_get_order_info($this->order_id);

        // Retrieve order information based on the order ID.
        if (empty($this->order_info)) {
            throw new \Exception(__('vtbpayments_order_not_found'));
        }
    }


    /**
     * Получаем и устанавливаем свойство processor_params.
     *
     * @param array $processor_data  Processor data, including payment parameters.
     *
     * @return void
     * @throws Exception
     */
    private function setProcessorParams(array $processor_data): void
    {
        $processor_data = $processor_data ?: fn_get_processor_data($this->order_info['payment_id']);
        $this->processor_params = $processor_data['processor_params'] ?? [];

        if (empty($this->processor_params)) {
            throw new \Exception(__('vtbpayments_empty_processor_params'));
        }
    }


    /**
     * Handle the payment response.
     *
     * Handle the payment response logic, including error handling.
     * This function checks and processes the payment response based on the provided $mode.
     * If an error occurs during response handling, an exception is thrown with an error message.
     *
     * @return void
     * @throws Exception If an error occurs during response handling, an exception is thrown with an error message.
     */
    public function handleResponse(): void
    {
        try {
            $this->deferConcurrentRequests();

            fn_update_order_payment_info($this->order_id, [
                'vtbpay_query_running' => 1
            ]);

            $this->changePaymentStatus();

            fn_update_order_payment_info($this->order_id, [
                'vtbpay_query_running' => 0
            ]);

            if ($this->mode === 'return') {
                $current_order_status = $this->getCurrentOrderStatus();

                // Если статус ошибки или возврата,
                // то очищаем корзину и редиректим на главную
                switch ($current_order_status) {
                  case $this->processor_params['statuses']['failed']:
                  case 'E':
                      $this->unpleasantEnding($this->order_info['payment_info']['reason_text']);
                      break;
                  default:
                      fn_order_placement_routines('route', $this->order_id, false);
                      break;
                }
            }
            elseif ($this->mode === 'webhook') {
                die('ok');
            }

        } catch (\Exception | VtbPayException $e) {
            fn_update_order_payment_info($this->order_id, [
                'vtbpay_query_running' => 0
            ]);
            $this->executeErrorScenario($e);
        }
    }


    /**
     * Приостанавливает выполнение одновременных запросов
     *
     * @return void
     */
    private function deferConcurrentRequests(): void
    {
        // Если над заказом уже работает запрос, то останавливанм текущий
        $maxAttempts = 5;
        for ($i = 0; $i < $maxAttempts; $i++) {
            $this->setOrderInfo([]);
            if (!($this->order_info['payment_info']['vtbpay_query_running'] ?? false)) {
                break;
            }
            sleep(1);
        }
    }


    /**
     * Получение данные для закрытия платежа
     *
     * @return void
     */
    private function changePaymentStatus(): void
    {
        // Получаем актуальный статус платежа
        $payment_status_data = $this->getPaymentStatusData();
        $payment_status = $payment_status_data['object']['status']['value'] ?? '';
        $paid_status = $this->processor_params['statuses']['paid'];
        $failed_status = $this->processor_params['statuses']['failed'];

        $available_statuses = [
            'PAID' => [
                'order_status' => $paid_status,
                'reason_text' => __('approved')
            ],
            'RECONCLIED' => [
                'order_status' => $paid_status,
                'reason_text' => __('approved')
            ],
            'PENDING' => [
                'order_status' => 'A',
                'reason_text' => __('vtbpayments_payment_received_but_not_confirmed')
            ],
            'REFUNDED' => [
                'order_status' => 'E',
                'reason_text' => __('vtbpayments_payment_refunded')
            ]
        ];

        $order_status_data = $available_statuses[$payment_status] ?? [];

        if (
            $payment_status !== 'REFUNDED' &&
            isset($payment_status_data['object']['transactions']['refunds'])
        ) {
            // Указываем статус возврата
            $context['order_status'] = 'E';
            $context['reason_text'] = __('vtbpayments_payment_refunded');

            throw new VtbPayException($context['reason_text'], $context);
        }

        $current_order_status = $this->getCurrentOrderStatus();

        if (!empty($order_status_data)) {
            if ($current_order_status !== $order_status_data['order_status']) {
                $this->setOrUpdateOrderStatus([
                    'order_status' => $order_status_data['order_status'],
                    'reason_text' => $order_status_data['reason_text']
                ]);

                if (isset($order_status_data['fnc'])) $order_status_data['fnc']();
            }
        }
        elseif ($current_order_status !== $failed_status) {
            $this->setOrUpdateOrderStatus([
                'order_status' => $failed_status,
                'reason_text' => __('vtbpayments_payment_failed')
            ]);

            // Logging unsuccessful payment
            $this->logger->error(
                __FUNCTION__ . ' > getOrderInfo. Payment not paid: ', [
                'order_id_and_amount' => $this->order_id_and_amount
            ]);
        }
    }


    /**
     * Initiate a payment request.
     *
     * Initiate a payment request, including error handling.
     * This function prepares and sends a payment request based on the provided order details and processor parameters.
     * If an error occurs during payment initiation, an exception is thrown with an error message.
     *
     * @return void
     * @throws Exception If an error occurs during payment initiation, an exception is thrown with an error message.
     */
    public function sendForPayment(): void
    {
        try {
            fn_create_payment_form(
                $this->getPayUrl(),
                [],
                'VtbPayments',
                true,
                'GET'
            ); // Create a payment form and redirect the user to the payment page.
        } catch (\Exception | VtbPayException $e) {
            $this->executeErrorScenario($e);
        }
    }


    /**
     * Сценарий, выполняемый после отлова исключения
     *
     * @return void
     */
    private function executeErrorScenario($e): void
    {
        // Handle exception and log error
        $context = [];
        if (method_exists($e, 'getContext')) $context = $e->getContext();

        // Handle exception and log error
        $this->logger->error(sprintf(
            __FUNCTION__ . ' > VtbPay exception : %s; Order id: %s;',
            $e->getMessage(),
            $this->order_id ?: ''
        ), $context);

        // Set payment response for failure.
        $pp_response['order_status'] = $context['order_status'] ?? OrderStatuses::FAILED;
        $pp_response['reason_text'] = $context['reason_text'] ?? $e->getMessage();

        // Finish payment and perform order placement routines.
        $this->setOrUpdateOrderStatus($pp_response);

        if ($this->mode === 'return') {
            $this->unpleasantEnding($pp_response['reason_text']);
        }
        elseif ($this->mode === 'webhook') {
            die('error');
        }
    }


    /**
     * Очищаем корзину, устанавливаем текст ошибки и редиректим на главную
     *
     * @param string $reason_text Текст ошибки
     *
     * @return void
     */
    private function unpleasantEnding(string $reason_text): void
    {
        fn_clear_cart(Tygh::$app['session']['cart']);
        fn_set_notification('E', __('error'), $reason_text);
        fn_redirect(fn_url('', 'C', 'http'));
    }


    /**
     * Устанавливаем или обновляем итоговый статус заказа
     *
     * @param array $pp_response Данный заказ для обновления
     *
     * @return void
     */
    private function setOrUpdateOrderStatus(array $pp_response): void
    {
        $current_order_status = $this->getCurrentOrderStatus();
        $this->logger->debug(
            __FUNCTION__ . ' - INPUT: ', [
            'order_status' => $current_order_status,
            'order_id' => $this->order_id,
            'pp_response' => $pp_response
        ]);
        if (empty($current_order_status)) {
            fn_finish_payment($this->order_id, $pp_response);
        }
        else {
            fn_change_order_status(
                $this->order_id,
                $pp_response['order_status'],
                $current_order_status
            );
            fn_update_order_payment_info($this->order_id, $pp_response);
        }
    }


    /**
     * Получает текущий статус заказа
     *
     * @return string
     */
    private function getCurrentOrderStatus(): string
    {
          $this->setOrderInfo([]);
          return $this->order_info['payment_info']['order_status'] ?? '';
    }


    /**
     * Инициализация и настройка объекта класса VtbPayLogger.
     *
     * Эта функция инициализирует и настраивает логгер, используемый плагином VtbPay для ведения журнала.
     *
     * @return void
     */
    private function setVtbpayLogger(): void
    {
        $sensitive_data_keys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];
        $mode_key = $this->mode . '-' . rand(1111, 9999);
        $logger = VtbPayLogger::getInstance();
        $this->logger = $logger
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitive_data_keys)
                        ->setOption('additionalCommonText', $mode_key)
                        ->setLogFilePath(
                            fn_get_files_dir_path() . 'VtbPayments-' . date('d-m-Y') . '.log'
                        )->setCustomRecording(function($message) use ($logger) {
                            $logging = $this->processor_params['logging'] ?? false;
                            if ($logging) $logger->writeToFile($message);
                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}


$vtb_pay = new VtbPayments(
    $order_id ?? '',
    $order_info ?? [],
    $processor_data ?? [],
    $mode ?? ''
);

if (defined('PAYMENT_NOTIFICATION')) {
    // Handle payment response based on the provided mode.
    $vtb_pay->handleResponse();
}
else {
    // Initiate a payment request using the provided order ID, order information, and processor data.
    $vtb_pay->sendForPayment();
}
