<?php

namespace Src\Traits;

use \Src\Classes\Api\VtbApi,
    \Src\Classes\Exception\VtbPayException;

trait VtbPay
{
    /**
     * Create and return an instance of the VtbApi class for VtbPayments integration.
     *
     * @return VtbApi An instance of the VtbApi class.
     */
    private function getVtbApi(): VtbApi
    {
        $processor_params = $this->processor_params;
        return new VtbApi(
            $processor_params['client_id_vtbpay'],
            $processor_params['client_secret_vtbpay'],
            (bool) $processor_params['test_mode_vtbpay'],
            $processor_params['client_merchant_vtbpay'] ?: ''
        );
    }


    /**
     * Получает URL-адрес платежа для перенаправления.
     *
     * @return string URL-адрес платежа.
     */
    private function getPayUrl(): string
    {
        // Email клиента
        $email = $this->getCustomerEmail();
        // Calculate the payment amount based on order information and processor parameters.
        $amount = $this->getAmount();
        // Generate the return URL for payment notification.
        $return_url = $this->getReturnUrl();
        // Since in this CMS the order identifier does not change when the order composition changes,
        // therefore, to create a new identifier, we associate the identifier with the price.
        $this->order_id_and_amount = $this->order_id . '-' . $amount;

        // Двухстадийный платеж (настройка в админке)
        $two_stage_payment = (bool) ($this->processor_params['two_stage_payment'] ?? false);

        $items = [];
        if (!$two_stage_payment) $items = $this->getItems();

        // Указываем индекс почты как добавочный текст к названию заказа
        $appended_order_name = '';
        if (
            isset($this->processor_params['send_zip_code']) &&
            $this->processor_params['send_zip_code']
        ) {
            $appended_order_name = '. Почтовый индекс: ' . $this->order_info['s_zipcode'] ?? '';
        }

        // Request a payment link from the VtbApi.
        $response = $this->getVtbApi()->getOrderLink(
            $this->order_id_and_amount,
            $this->order_info['email'],
            $this->order_info['timestamp'],
            $amount,
            $return_url,
            $two_stage_payment,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION,
            $appended_order_name
        );

        fn_update_order_payment_info($this->order_id, [
            'transaction_id' => $response['object']['orderCode']
        ]); // Update the order's payment information.

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получение Email покупателя
     *
     * @return string Email клиента
     *
     * @throws VtbPayException Если email пустой
     */
    private function getCustomerEmail(): string
    {
        $email = $this->order_info['email'] ?: $this->processor_params['email_fiscal'] ?: '';
        if (!self::validate_email($email)) {
            throw new VtbPayException(__('vtbpayments_email_not_found'), [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validate_email(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($local_part, $domain_part) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $local_length = strlen($local_part);
        if ($local_length < 1 || $local_length > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domain_length = strlen($domain_part);
        if ($domain_length < 1 || $domain_length > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowed_tlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowed_tlds) && !empty($allowed_tlds[0])) {
            // Извлечение TLD из доменной части
            $domain_parts = explode('.', $domain_part);
            $tld = strtolower(end($domain_parts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowed_tlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Calculate the payment amount based on order information and processor parameters.
     *
     * @return float The calculated payment amount.
     */
    private function getAmount(): float
    {
        // Check discount
        $total = fn_format_price_by_currency(
            $this->order_info['total'],
            CART_PRIMARY_CURRENCY,
            $this->order_info['secondary_currency']
        );
        $discount = $total - $this->order_info['subtotal'];
        if ($discount === $this->order_info['subtotal_discount']) {
            return $this->order_info['subtotal'];
        }
        return $total;
    }


    /**
     * Generate and return the return URL for the payment.
     *
     * @return string The return URL for payment notification.
     */
    private function getReturnUrl(): string
    {
        $protocol = empty($_SERVER['HTTPS']) ? 'http' : 'https';
        $host = $_SERVER['HTTP_HOST'];

        return "{$protocol}://{$host}/index.php?dispatch=payment_notification.return&payment=vtbpayments";
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems(): array
    {
        $items = [];
        $key = 1;
        $fiscal_settings = [
            'measure_fiscal' => 0,
            'tax_type_fiscal' => 'none',
            'payment_type_fiscal' => 'full_prepayment',
            'payment_subject_fiscal' => 1
        ];

        foreach ($this->order_info['products'] as $item) {

            $product_settings_fiscal = fn_vtbpayments_get_product_settings_fiscal($item['product_id']);
            $product_fiscal = [];
            // Получение фискальных параметров для каждого продукта
            foreach ($fiscal_settings as $name => $default) {
                $product_fiscal[$name] = $product_settings_fiscal[$name] ?: '-';
                if ($product_fiscal[$name] === '-') $product_fiscal[$name] = $this->processor_params[$name] ?: $default;
            }

            $items[] = [
                'positionId' => $key,
                'name' => $item['product'],
                'code' => (string) $item['product_id'],
                'price' => floatval($item['price']),
                'measure' => (int) $product_fiscal['measure_fiscal'],
                'quantity' => (int) $item['amount'],
                'taxParams' => [
                    'taxType' => $product_fiscal['tax_type_fiscal']
                ],
                'paymentType' => $product_fiscal['payment_type_fiscal'],
                'paymentSubject' => (int) $product_fiscal['payment_subject_fiscal'],
                'amount' => floatval($item['subtotal'])
            ];

            $key++;
        }

        // Получаем стоимость доставки
        $shipping_cost = $this->order_info['shipping_cost'] ?? 0;
        if ($shipping_cost > 0) {
            $items[] = [
                'positionId' => $key,
                'name' => __('vtbpayments_delivery'),
                'price' => floatval($shipping_cost),
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->processor_params['tax_type_fiscal'] ?: 'none'
                ],
                'paymentType' => $this->processor_params['payment_type_delivery_fiscal'] ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => floatval($shipping_cost)
            ];
        }

        return $items;
    }


    /**
     * Получение статуса заказа из VTB API.
     *
     * @return array
     */
    private function getPaymentStatusData(): array
    {
        // Get order information from VtbApi.
        $response = $this->getVtbApi()->getOrderInfo($this->order_id_and_amount);

        return $response;
    }
}
