<?php

/**
 * Payment system VTB
 *
 * @copyright  VTB
 * @author     VTB
 *
 */

header('Content-type: text/html; charset=UTF-8');

require_once 'vendor/autoload.php';
if (!class_exists('Fivecms')) {
    require_once '../../api/Fivecms.php';
}

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;


class VtbPayCallback extends \Fivecms
{
    /**
     * @var object Current order
     */
    private $order;

    /**
     * @var mixed Selected payment method
     */
    private $paymentMethod;

    /**
     * @var array Payment system settings
     */
    private $paymentSettings; // Настройки

    /**
     * @var \Vtbpay\Classes\Common\VtbPayLogger Logging class
     */
    private $logger;


    /**
     * Constructs a new instance of VtbPayCallback.
     */
    public function __construct()
    {
        parent::__construct();

        $envPath = __DIR__ . '/config/.env';
        if (file_exists($envPath)) (new Dotenv())->load($envPath);

        $this->setVtbpayLogger();
        $orderId = $this->request->get('orderId') ?: null;

        $this->run($orderId);
    }


    /**
     * Initializes the callback.
     */
    private function run($orderId)
    {
        try {
            if (empty($orderId)) {
                throw new \Exception('Идентификатор заказа не указан.');
            }

            $this->setOrder($orderId);

            $this->setPaymentMethod();

            $this->setPaymentSettings();

            $this->checkGoods();

            $this->checkOrderPaid();

            $this->setAsPaid();

        } catch (\Exception | VtbPayException  $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Log the caught exception for debugging purposes.
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s;',
                $e->getMessage(),
                $orderId ?: ''
            ), $context);

            die($e->getMessage() . ' <a href="' . $this->config->root_url . '">Go Home</a>');
        }
    }


    /**
     * Checks the existence of an order.
     *
     * @throws \Exception Throws an exception if the order does not exist.
     */
    private function setOrder($orderId)
    {
        // Существует ли заказ
        $this->order = $this->orders->get_order((int)$orderId);

        if (empty($this->order)) {
            throw new \Exception('Оплачиваемый заказ не найден.');
        }
    }


    /**
     * Checks the validity of the payment method.
     *
     * @throws \Exception Throws an exception if the payment method is invalid.
     */
    private function setPaymentMethod()
    {
        // Наш ли способ оплаты
        $this->paymentMethod = $this->payment->get_payment_method(intval($this->order->payment_method_id));

        if (empty($this->paymentMethod)) {
            throw new \Exception('Неизвестный метод оплаты.');
        }
    }


    /**
     * Sets the payment settings.
     *
     * @throws \Exception Throws an exception if the payment settings are missing.
     */
    private function setPaymentSettings()
    {
        $this->paymentSettings = $this->payment->get_payment_settings($this->paymentMethod->id);

        if (empty($this->paymentSettings)) {
            throw new \Exception('Настройки платёжной системы не указаны.');
        }
    }


    /**
     * Create and return an instance of the VtbApi class for VtbPayments integration.
     *
     * @return VtbApi An instance of the VtbApi class.
     */
    private function getVtbApi()
    {
        $settings = $this->paymentSettings;

        return new VtbApi(
            $settings['vtbpay_client_id'],
            $settings['vtbpay_client_secret'],
            (bool) $settings['vtbpay_test_mode'],
            $settings['vtbpay_merchant_authorization'] ?? ''
        );
    }


    /**
     * Checks the availability of goods in the order.
     *
     * @throws \Exception Throws an exception if there is insufficient stock of any product.
     */
    private function checkGoods()
    {
        // Наличие товара
        $purchases = $this->orders->get_purchases([
            'order_id'=>intval($this->order->id)
        ]);

        foreach ($purchases as $purchase) {
            $variant = $this->variants->get_variant(intval($purchase->variant_id));

            if (empty($variant) || (!$variant->infinity && $variant->stock < $purchase->amount)) {
                throw new \Exception("Недостаточно товара {$purchase->product_name} {$purchase->variant_name}.");
            }
        }
    }


    /**
     * Checks if the order has been paid.
     *
     * @throws \Exception Throws an exception if the order has already been paid.
     */
    private function checkOrderPaid()
    {
        // Check if the order has already been paid
        if($this->order->paid) {
            throw new \Exception('Заказ уже оплачен.');
        }

        $response = $this->getVtbApi()->getOrderInfo($this->order->id);

        // Если заказ оплачивался, и статус заказа уже изменялся
        if (
            !isset($response['object']['status']['value']) ||
            $response['object']['status']['value'] !== 'PAID'
        ) {
            throw new \Exception('Платёж не выполнен.');
        }
    }


    /**
     * Marks the order as paid and performs necessary actions.
     */
    private function setAsPaid()
    {
        $orderId = intval($this->order->id);
        $this->orders->set_pay($orderId);
        $this->orders->close($orderId);
        $this->notify->email_order_user($orderId);
        $this->notify->email_order_admin($orderId);

        $success_url = $this->config->root_url . '/order/' . $this->order->url;
        header('location:' . $success_url, true, 302);
        exit;
    }


    /**
     * Устанавливает объект логирования для модуля.
     */
    private function setVtbpayLogger()
    {
        $sensitiveDataKeys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];
        $logger = VtbPayLogger::getInstance();

        $this->logger = $logger
                        ->setOption('additionalCommonText', 'return-' . rand(1111, 9999))
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitiveDataKeys)
                        ->setLogFilePath(
                            dirname(__DIR__, 2) . '/files/log/vtbpay-' . date('d-m-Y') . '.log'
                        )->setCustomRecording(function($message) use ($logger) {
                            if (
                                isset($this->paymentSettings['vtbpay_logging']) &&
                                $this->paymentSettings['vtbpay_logging']
                            ) $logger->writeToFile($message);
                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}

new VtbPayCallback();
