<?php

/**
 * Main handler class for Payment system VTB
 *
 * @copyright  VTB
 * @author     VTB
 *
 */

require_once 'vendor/autoload.php';

use \Vtbpay\Classes\Api\VtbApi,
    \Vtbpay\Classes\Common\VtbPayLogger,
    \Vtbpay\Classes\Exception\VtbPayException,
    \Symfony\Component\Dotenv\Dotenv;

class VtbPay extends \Fivecms
{
    const CMS_NAME = '5CMS';
    const PLUGIN_VERSION = '1.3.5';

    /**
     * @var object Current order
     */
    private $order;

    /**
     * @var mixed Selected payment method
     */
    private $paymentMethod;

    /**
     * @var array Payment system settings
     */
    private $paymentSettings; // Настройки

    /**
     * @var \Vtbpay\Classes\Common\VtbPayLogger Logging class
     */
    private $logger;


    /**
     * Конструктор для класса плагина "Платежная система ВТБ".
     */
    public function __construct()
    {
        parent::__construct();

        $envPath = __DIR__ . '/config/.env';
        if (file_exists($envPath)) (new Dotenv())->load($envPath);

        $this->setVtbpayLogger();
    }


    /**
     * Create and return an instance of the VtbApi class for VtbPay integration.
     *
     * @return VtbApi An instance of the VtbApi class.
     */
    private function getVtbApi()
    {
       $settings = $this->paymentSettings;

       return new VtbApi(
           $settings['vtbpay_client_id'],
           $settings['vtbpay_client_secret'],
           (bool) $settings['vtbpay_test_mode'],
           $settings['vtbpay_merchant_authorization'] ?? ''
       );
    }


    /**
     * Generates the checkout form for VtbPay payment.
     *
     * @param int $orderId The ID of the order for which the checkout form is generated.
     * @param string|null $buttonText The text to display on the checkout button (optional).
     * @return string The HTML representation of the checkout button with VtbPay payment link.
     * @throws \Exception Throws an exception if there are errors during the checkout process.
     */
    public function checkout_form($orderId, $buttonText = null)
    {
        if (empty($buttonText)) {
            $buttonText = 'Перейти к оплате';
        }

        try {
            $this->setOrder($orderId);

            $this->setPaymentMethod();

            $this->setPaymentSettings();

            $this->checkCurrency();

            // Check if the payment URL is present and not empty in the response.
            if (empty($payUrl = $this->getPayUrl())) {
                // Throw an exception if the payment URL is missing or empty.
                throw new \Exception('Не удалось получить URL-адрес оплаты.');
            }

            return $this->generateRedirectHtml(
                $payUrl,
                $buttonText
            );

        } catch (\Exception | VtbPayException  $e) {
            // Handle exception and log error
            $context = [
                'file_exception' => $e->getFile(),
                'line_exception' => $e->getLine(),
            ];
            if (method_exists($e, 'getContext')) $context = array_merge($e->getContext(), $context);

            // Log the caught exception for debugging purposes.
            $this->logger->error(sprintf(
                __FUNCTION__ . ' > VtbPay Exception: %s; Order id: %s;',
                $e->getMessage(),
                $orderId ?: ''
            ), $context);

            return '<span style="color:red;">' . $e->getMessage() . '</span>';
        }
    }


    /**
     * Generates HTML code for automatic redirection with a delay.
     *
     * @param string $payUrl The URL for redirection.
     * @param string $buttonText The text to display on the button.
     * @return string The generated HTML code.
     */
    private function generateRedirectHtml($payUrl, $buttonText)
    {
        return "<div>
                    <p>
                      Страница будет автоматически переадресована на платежную форму через 3 секунды.<br>
                      Нажмите на ссылку ниже для незамедлительного перехода.
                    </p>
                    <a href=\"{$payUrl}\" class=\"checkout_button\" id=\"vtbpay-button-confirm\">{$buttonText}</a>
                </div>
                <script type=\"text/javascript\">
                    const paymentButton = document.getElementById(\"vtbpay-button-confirm\");
                    if (paymentButton) {
                        setTimeout(function () {
                            paymentButton.click();
                        }, 3000); // 3000 миллисекунд = 3 секунды
                    }
                </script>";
    }


    /**
     * Checks the existence of an order.
     *
     * @param int $orderId The ID of the order to check.
     * @throws \Exception Throws an exception if the order does not exist.
     */
    private function setOrder($orderId)
    {
        // Существует ли заказ
        $this->order = $this->orders->get_order((int)$orderId);

        if (empty($this->order)) {
            throw new \Exception('Оплачиваемый заказ не найден.');
        }
    }


    /**
     * Checks the validity of the payment method.
     *
     * @throws \Exception Throws an exception if the payment method is invalid.
     */
    private function setPaymentMethod()
    {
        // Наш ли способ оплаты
        $this->paymentMethod = $this->payment->get_payment_method(intval($this->order->payment_method_id));

        if (empty($this->paymentMethod)) {
            throw new \Exception('Неизвестный метод оплаты.');
        }
    }


    /**
     * Sets the payment settings.
     *
     * @throws \Exception Throws an exception if the payment settings are missing.
     */
    private function setPaymentSettings()
    {
        $this->paymentSettings = $this->payment->get_payment_settings($this->paymentMethod->id);

        if (empty($this->paymentSettings)) {
            throw new \Exception('Настройки платёжной системы не указаны.');
        }
    }


    /**
     * Checks the currency for payment eligibility.
     *
     * @throws \Exception Throws an exception if payment is not possible in the specified currency.
     */
    private function checkCurrency()
    {
        $paymentCurrency = $this->money->get_currency(intval($this->paymentMethod->currency_id));

        if ($paymentCurrency->code !== 'RUB') {
            throw new \Exception('Оплата возможна только в рублях.');
        }
    }


    /**
     * Retrieves the payment URL for redirection.
     *
     * @return string The payment URL.
     * @throws \Exception Throws an exception if the payment URL cannot be obtained.
     */
    private function getPayUrl()
    {
        // Receive the final payment amount
        $amount = round($this->money->convert(
            $this->order->total_price,
            $this->paymentMethod->currency_id,
            false
        ), 2);

        // Return page after successful/unsuccessful payment
        $returnUrl = $this->config->root_url . '/payment/' . __CLASS__ . '/callback.php';

        $enableFiscal = (bool) ($this->paymentSettings['vtbpay_enable_fiscal'] ?? false);

        $items = [];
        if ($enableFiscal) $items = $this->getItems();

        $email = $this->getCustomerEmail();

        $response = $this->getVtbApi()->getOrderLink(
            $this->order->id,
            $email,
            time(),
            $amount,
            $returnUrl,
            false,
            $items,
            self::CMS_NAME,
            self::PLUGIN_VERSION
        );

        return $response['object']['payUrl'] ?? '';
    }


    /**
     * Получает email клиента иначе выкидывает ошибку
     *
     * @return string E-mail клиента.
     */
    private function getCustomerEmail()
    {
        $email = $this->order->email ?: $this->paymentSettings['vtbpay_email_fiscal'];
        if (!self::validateEmail($email)) {
            throw new VtbPayException('Указанный адрес электронной почты не прошёл валидацию.', [
                'email' => $email
            ]);
        }
        return $email;
    }


    /**
     * Валидация Email.
     *
     * @param string $email Email.
     *
     * @return bool
     */
    private static function validateEmail(string $email)
    {
        // Базовая проверка синтаксиса по стандарту RFC
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // Разделение email на локальную и доменную части
        list($localPart, $domainPart) = explode('@', $email);

        // Проверка длины локальной части (от 1 до 64 символов)
        $localLength = strlen($localPart);
        if ($localLength < 1 || $localLength > 64) {
            return false;
        }

        // Проверка длины доменной части (от 1 до 255 символов)
        $domainLength = strlen($domainPart);
        if ($domainLength < 1 || $domainLength > 253) {
            return false;
        }

        // Список разрешённых популярных доменных зон
        $allowedTlds = explode(',', $_ENV['EMAIL_VALIDATION_ALLOWED_TLDS'] ?? '');
        if (!empty($allowedTlds) && !empty($allowedTlds[0])) {
            // Извлечение TLD из доменной части
            $domainParts = explode('.', $domainPart);
            $tld = strtolower(end($domainParts));

            // Проверка, что TLD входит в список разрешённых
            if (!in_array($tld, $allowedTlds)) {
                return false;
            }
        }

        return true;
    }


    /**
     * Получает массив товаров заказа.
     *
     * @return array Массив товаров.
     */
    private function getItems()
    {
        $purchases = $this->orders->get_purchases([
            'order_id'=>intval($this->order->id)
        ]);

        $totalAmountProducts = 0;

        $items = [];
        foreach ($purchases as $key => $item) {
            $amount = $item->price * $item->amount;
            $totalAmountProducts += $amount;
            $items[] = [
                'positionId' => ($key + 1),
                'name' => $item->product_name,
                'code' => $item->order_id,
                'price' => floatval($item->price),
                'measure' => (int) $this->paymentSettings['vtbpay_measure_fiscal'] ?: 0,
                'quantity' => (int) $item->amount,
                'taxParams' => [
                    'taxType' => $this->paymentSettings['vtbpay_tax_type_fiscal'] ?: 'none'
                ],
                'paymentType' => $this->paymentSettings['vtbpay_payment_type_fiscal'] ?: 'full_prepayment',
                'paymentSubject' => (int) $this->paymentSettings['vtbpay_payment_subject_fiscal'] ?: 1,
                'amount' => floatval($amount)
            ];
        }

        // Стоимость доставки
        if ($this->order->delivery_price > 0) {
            $delivery = $this->delivery->get_delivery($this->order->delivery_id);

            if (!$delivery->separate_payment) $items[] = [
                'positionId' => ($key + 2),
                'name' => 'Доставка',
                'price' => floatval($delivery->price),
                'measure' => 0,
                'quantity' => 1,
                'taxParams' => [
                    'taxType' => $this->paymentSettings['vtbpay_tax_type_fiscal'] ?: 'none'
                ],
                'paymentType' => $this->paymentSettings['vtbpay_payment_type_delivery_fiscal'] ?: 'full_prepayment',
                'paymentSubject' => 4,
                'amount' => floatval($delivery->price)
            ];
        }

        // Получаем данные о скидках
        $discount = $this->order->discount ? $totalAmountProducts * ($this->order->discount / 100) : 0;
        $totalDiscount = round(
            $this->money->convert($discount +
            $this->order->bonus_discount + $this->order->coupon_discount,
            $this->paymentMethod->currency_id,
            false
        ), 2);
        if ($totalDiscount > 0) $items = $this->applyDiscountToItems($items, $totalDiscount, $totalAmountProducts);

        return $items;
    }


    /**
     * Применяет скидку к товарам заказа
     */
    private function applyDiscountToItems($items, $totalDiscount, $totalAmountProducts)
    {
        // Рассчитываем процент скидки
        $discountPercentage = $totalDiscount / $totalAmountProducts;

        foreach ($items as &$item) {
            if ($item['name'] === 'Доставка') continue;
            $itemDiscount = $item['amount'] * $discountPercentage;
            $item['amount'] = round($item['amount'] - $itemDiscount, 2);
            $item['price'] = round($item['amount'] / $item['quantity'], 2);
        }

        return $items;
    }


    /**
     * Устанавливает объект логирования для модуля.
     */
    private function setVtbpayLogger()
    {
        $sensitiveDataKeys = json_decode($_ENV['LOG_SENSITIVE_DATA_KEYS'] ?? '', true) ?: [];
        $logger = VtbPayLogger::getInstance();

        $this->logger = $logger
                        ->setOption('additionalCommonText', 'payment-' . rand(1111, 9999))
                        ->setOption('showBacktrace', true)
                        ->setOption('sensitiveDataKeys', $sensitiveDataKeys)
                        ->setLogFilePath(
                            dirname(__DIR__, 2) . '/files/log/vtbpay-' . date('d-m-Y') . '.log'
                        )->setCustomRecording(function($message) use ($logger) {
                            if (
                                isset($this->paymentSettings['vtbpay_logging']) &&
                                $this->paymentSettings['vtbpay_logging']
                            ) $logger->writeToFile($message);
                        }, VtbPayLogger::LOG_LEVEL_DEBUG);
    }
}
